"""Unit tests for Merge functionality.

This module tests the Merge including:
- Basic MERGE statement construction (ANSI SQL:2003 MERGE)
- MERGE INTO target table with USING source
- ON conditions for matching rows
- WHEN MATCHED THEN UPDATE/DELETE
- WHEN NOT MATCHED THEN INSERT
- WHEN NOT MATCHED BY SOURCE THEN UPDATE/DELETE
- Complex conditional WHEN clauses
- Subqueries as source tables
- Parameter binding and SQL injection prevention
- Error handling for invalid operations
"""

from typing import TYPE_CHECKING, Any

import pytest
from sqlglot import exp

from sqlspec.exceptions import SQLBuilderError
from sqlspec.statement.builder import Merge, Select
from sqlspec.statement.builder._base import SafeQuery
from sqlspec.statement.result import SQLResult
from sqlspec.statement.sql import SQL

if TYPE_CHECKING:
    pass


# Test basic MERGE construction
def test_merge_builder_initialization() -> None:
    """Test Merge initialization."""
    builder = Merge()
    assert isinstance(builder, Merge)
    # Merge uses _expression to store the MERGE statement
    assert isinstance(builder._expression, exp.Merge)
    assert builder._expression.args.get("this") is None  # target table
    assert builder._expression.args.get("using") is None  # source table
    assert builder._expression.args.get("on") is None  # on condition


def test_merge_into_method() -> None:
    """Test setting target table with into()."""
    builder = Merge().into("users")
    # Check the target table is set in the expression
    assert builder._expression is not None
    assert builder._expression.args.get("this") is not None
    assert isinstance(builder._expression.args["this"], exp.Table)
    assert builder._expression.args["this"].name == "users"


def test_merge_into_with_alias() -> None:
    """Test setting target table with alias."""
    builder = Merge().into("users", "u")
    assert builder._expression is not None
    target = builder._expression.args.get("this")
    assert target is not None
    assert isinstance(target, exp.Table)
    assert target.name == "users"
    assert target.alias == "u"


def test_merge_into_returns_self() -> None:
    """Test that into() returns builder for chaining."""
    builder = Merge()
    result = builder.into("users")
    assert result is builder


# Test USING clause
@pytest.mark.parametrize(
    "source,alias,expected_has_alias",
    [
        ("source_table", None, False),
        ("source_table", "src", True),
        ("staging_users", "s", True),
        (Select().select("*").from_("temp_data"), "tmp", True),
    ],
    ids=["no_alias", "with_alias", "staging_alias", "subquery_source"],
)
def test_merge_using_clause(source: Any, alias: Any, expected_has_alias: bool) -> None:
    """Test USING clause with various source types."""
    builder = Merge().into("users").using(source, alias)

    assert builder._expression is not None
    using_expr = builder._expression.args.get("using")
    assert using_expr is not None

    if isinstance(source, str):
        assert isinstance(using_expr, exp.Table)
        assert using_expr.name == source
    else:
        # It's a subquery
        assert using_expr is not None

    if expected_has_alias and isinstance(using_expr, exp.Table):
        assert using_expr.alias == alias


def test_merge_using_returns_self() -> None:
    """Test that using() returns builder for chaining."""
    builder = Merge().into("users")
    result = builder.using("source_table", "src")
    assert result is builder


# Test ON condition
@pytest.mark.parametrize(
    "condition,expected_in_sql",
    [
        ("target.id = src.id", ["=", "id"]),
        ("users.id = staging.id AND users.version < staging.version", ["AND", "version"]),
        ("t1.key1 = t2.key1 AND t1.key2 = t2.key2", ["key1", "key2"]),
    ],
    ids=["simple_join", "complex_condition", "composite_key"],
)
def test_merge_on_condition(condition: str, expected_in_sql: list[str]) -> None:
    """Test ON condition for MERGE."""
    builder = Merge().into("users").using("staging_users", "src").on(condition)

    # Check the ON condition is set in the expression
    assert builder._expression is not None
    on_expr = builder._expression.args.get("on")
    assert on_expr is not None

    query = builder.when_matched_then_update({"status": "updated"}).build()

    assert "MERGE" in query.sql
    assert "ON" in query.sql
    for part in expected_in_sql:
        assert part in query.sql


def test_merge_on_returns_self() -> None:
    """Test that on() returns builder for chaining."""
    builder = Merge().into("users").using("source", "src")
    result = builder.on("users.id = src.id")
    assert result is builder


# Test WHEN MATCHED THEN UPDATE
@pytest.mark.parametrize(
    "updates,condition,expected_params",
    [
        ({"name": "John Doe"}, None, ["John Doe"]),
        ({"status": "active", "updated_at": "2024-01-01"}, None, ["active", "2024-01-01"]),
        ({"score": 100}, "src.score > 50", [100]),
        ({"level": "gold", "points": 1000}, "src.tier = 'premium'", ["gold", 1000]),
    ],
    ids=["single_update", "multiple_updates", "conditional_update", "complex_conditional"],
)
def test_merge_when_matched_update(updates: dict[str, Any], condition: Any, expected_params: list[Any]) -> None:
    """Test WHEN MATCHED THEN UPDATE with various scenarios."""
    builder = (
        Merge()
        .into("users")
        .using("updates", "src")
        .on("users.id = src.id")
        .when_matched_then_update(updates, condition=condition)
    )

    query = builder.build()
    assert "MERGE" in query.sql
    assert "WHEN MATCHED" in query.sql
    assert "UPDATE" in query.sql
    assert "SET" in query.sql

    # Check parameters
    assert isinstance(query.parameters, dict)
    for param in expected_params:
        assert param in query.parameters.values()


# Test WHEN MATCHED THEN DELETE
@pytest.mark.parametrize(
    "condition,description",
    [
        (None, "unconditional_delete"),
        ("src.status = 'deleted'", "status_deleted"),
        ("src.expired = true", "expired_true"),
        ("src.last_login < '2022-01-01'", "inactive_users"),
    ],
    ids=lambda x: x[1] if isinstance(x, tuple) else str(x),
)
def test_merge_when_matched_delete(condition: Any, description: str) -> None:
    """Test WHEN MATCHED THEN DELETE with various conditions."""
    builder = (
        Merge()
        .into("users")
        .using("cleanup_list", "src")
        .on("users.id = src.id")
        .when_matched_then_delete(condition=condition)
    )

    query = builder.build()
    assert "MERGE" in query.sql
    assert "WHEN MATCHED" in query.sql
    assert "DELETE" in query.sql


# Test WHEN NOT MATCHED THEN INSERT
@pytest.mark.parametrize(
    "columns,values,expected_type",
    [
        (None, None, "default_values"),
        (["id", "name"], [1, "John"], "explicit_values"),
        (["id", "name", "email"], ["src.id", "src.name", "src.email"], "source_references"),
        (["id", "created_at"], [1, "CURRENT_TIMESTAMP"], "with_functions"),
    ],
    ids=["default_values", "explicit_values", "source_references", "with_functions"],
)
def test_merge_when_not_matched_insert(columns: Any, values: Any, expected_type: str) -> None:
    """Test WHEN NOT MATCHED THEN INSERT variations."""
    builder = Merge().into("users").using("new_users", "src").on("users.id = src.id")

    if columns is None and values is None:
        builder = builder.when_not_matched_then_insert()
    else:
        builder = builder.when_not_matched_then_insert(columns=columns, values=values)

    query = builder.build()
    assert "MERGE" in query.sql
    assert "WHEN NOT MATCHED" in query.sql
    assert "INSERT" in query.sql

    if expected_type == "explicit_values":
        assert isinstance(query.parameters, dict)
        assert 1 in query.parameters.values()
        assert "John" in query.parameters.values()


# Test WHEN NOT MATCHED BY SOURCE
def test_merge_when_not_matched_by_source_update() -> None:
    """Test WHEN NOT MATCHED BY SOURCE THEN UPDATE."""
    builder = (
        Merge()
        .into("users")
        .using("active_users", "src")
        .on("users.id = src.id")
        .when_not_matched_by_source_then_update({"status": "inactive", "updated_at": "2024-01-01"})
    )

    query = builder.build()
    assert "MERGE" in query.sql
    assert "WHEN NOT MATCHED" in query.sql
    assert "UPDATE" in query.sql
    assert "inactive" in query.parameters.values()


def test_merge_when_not_matched_by_source_delete() -> None:
    """Test WHEN NOT MATCHED BY SOURCE THEN DELETE."""
    builder = (
        Merge()
        .into("users")
        .using("active_users", "src")
        .on("users.id = src.id")
        .when_not_matched_by_source_then_delete()
    )

    query = builder.build()
    assert "MERGE" in query.sql
    assert "WHEN NOT MATCHED" in query.sql
    assert "DELETE" in query.sql


# Test multiple WHEN clauses
def test_merge_multiple_when_clauses() -> None:
    """Test MERGE with multiple WHEN clauses."""
    builder = (
        Merge()
        .into("inventory")
        .using("daily_changes", "chg")
        .on("inventory.product_id = chg.product_id")
        # Update quantity if positive change
        .when_matched_then_update(
            {"quantity": "inventory.quantity + chg.quantity_change"}, condition="chg.quantity_change > 0"
        )
        # Delete if quantity would be zero or negative
        .when_matched_then_delete(condition="inventory.quantity + chg.quantity_change <= 0")
        # Insert new products
        .when_not_matched_then_insert(
            columns=["product_id", "quantity", "last_updated"],
            values=["chg.product_id", "chg.quantity_change", "CURRENT_TIMESTAMP"],
        )
        # Mark missing products as discontinued
        .when_not_matched_by_source_then_update({"status": "discontinued"})
    )

    query = builder.build()
    assert "MERGE" in query.sql
    assert query.sql.count("WHEN MATCHED") >= 2
    assert query.sql.count("WHEN NOT MATCHED") >= 2
    assert "UPDATE" in query.sql
    assert "DELETE" in query.sql
    assert "INSERT" in query.sql


# Test SQL injection prevention
@pytest.mark.parametrize(
    "malicious_value",
    [
        "'; DROP TABLE users; --",
        "1'; DELETE FROM users WHERE '1'='1",
        "' OR '1'='1",
        "<script>alert('xss')</script>",
        "Robert'); DROP TABLE students;--",
    ],
    ids=["drop_table", "delete_from", "or_condition", "xss_script", "bobby_tables"],
)
def test_merge_sql_injection_prevention(malicious_value: str) -> None:
    """Test that malicious values are properly parameterized."""
    builder = (
        Merge()
        .into("users")
        .using("updates", "src")
        .on("users.id = src.id")
        .when_matched_then_update({"name": malicious_value})
    )

    query = builder.build()

    # Malicious SQL should not appear in query
    assert "DROP TABLE" not in query.sql
    assert "DELETE FROM users" not in query.sql
    assert "OR '1'='1'" not in query.sql
    assert "<script>" not in query.sql

    # Value should be parameterized
    assert malicious_value in query.parameters.values()


# Test error conditions
def test_merge_without_target_raises_error() -> None:
    """Test that MERGE without target table raises error."""
    builder = Merge()
    with pytest.raises(SQLBuilderError, match="Error generating SQL from expression"):
        builder.using("source", "src").build()


def test_merge_without_source_raises_error() -> None:
    """Test that MERGE without source still builds."""
    builder = Merge().into("users")
    # The ON condition can be set even without USING clause
    builder.on("users.id = source.id")
    query = builder.when_matched_then_update({"status": "updated"}).build()
    assert "MERGE" in query.sql


def test_merge_without_on_condition_raises_error() -> None:
    """Test that MERGE without ON condition raises error."""
    builder = Merge().into("users").using("source", "src")
    # Without ON condition, sqlglot might still generate SQL but it would be invalid
    # Let's just build it and check that MERGE is in the output
    query = builder.when_matched_then_update({"status": "updated"}).build()
    assert "MERGE" in query.sql


def test_merge_without_when_clauses_raises_error() -> None:
    """Test that MERGE without any WHEN clauses builds successfully."""
    builder = Merge().into("users").using("source", "src").on("users.id = src.id")
    # MERGE without WHEN clauses is actually valid SQL (though not very useful)
    query = builder.build()
    assert "MERGE" in query.sql
    assert "users" in query.sql
    assert "source" in query.sql


def test_merge_insert_columns_values_mismatch() -> None:
    """Test that mismatched columns and values raise error."""
    builder = Merge().into("users").using("source", "src").on("users.id = src.id")

    with pytest.raises(SQLBuilderError, match="Number of columns must match number of values"):
        builder.when_not_matched_then_insert(
            columns=["id", "name", "email"],
            values=[1, "John"],  # Missing email value
        )


def test_merge_insert_columns_without_values() -> None:
    """Test that columns without values raises error."""
    builder = Merge().into("users").using("source", "src").on("users.id = src.id")

    with pytest.raises(SQLBuilderError, match="Specifying columns without values"):
        builder.when_not_matched_then_insert(columns=["id", "name"])


def test_merge_insert_values_without_columns() -> None:
    """Test that values without columns raises error."""
    builder = Merge().into("users").using("source", "src").on("users.id = src.id")

    with pytest.raises(SQLBuilderError, match="Cannot specify values without columns"):
        builder.when_not_matched_then_insert(values=[1, "test"])


# Test complex scenarios
def test_merge_upsert_pattern() -> None:
    """Test common UPSERT pattern with MERGE."""
    builder = (
        Merge()
        .into("users", "u")
        .using(Select().select("id", "name", "email", "updated_at").from_("staging_users"), "s")
        .on("u.id = s.id")
        .when_matched_then_update({"name": "s.name", "email": "s.email", "updated_at": "s.updated_at"})
        .when_not_matched_then_insert(
            columns=["id", "name", "email", "created_at", "updated_at"],
            values=["s.id", "s.name", "s.email", "CURRENT_TIMESTAMP", "s.updated_at"],
        )
    )

    query = builder.build()
    assert "MERGE" in query.sql
    assert "SELECT" in query.sql  # From subquery
    assert "UPDATE" in query.sql
    assert "INSERT" in query.sql


def test_merge_sync_tables_pattern() -> None:
    """Test table synchronization pattern with MERGE."""
    builder = (
        Merge()
        .into("production_inventory", "prod")
        .using("warehouse_inventory", "wh")
        .on("prod.sku = wh.sku AND prod.location = wh.location")
        # Update if quantities differ
        .when_matched_then_update(
            {"quantity": "wh.quantity", "last_sync": "CURRENT_TIMESTAMP"}, condition="prod.quantity != wh.quantity"
        )
        # Delete if item no longer in warehouse
        .when_not_matched_by_source_then_delete()
        # Insert new warehouse items
        .when_not_matched_then_insert(
            columns=["sku", "location", "quantity", "last_sync"],
            values=["wh.sku", "wh.location", "wh.quantity", "CURRENT_TIMESTAMP"],
        )
    )

    query = builder.build()
    assert "MERGE" in query.sql
    # Check that the condition parts are present (order might vary)
    # Column names might be quoted
    assert "prod.sku = wh.sku" in query.sql or '"prod"."sku" = "wh"."sku"' in query.sql
    assert "prod.location = wh.location" in query.sql or '"prod"."location" = "wh"."location"' in query.sql
    assert "AND" in query.sql


# Test type information
def test_merge_expected_result_type() -> None:
    """Test that _expected_result_type returns correct type."""
    builder = Merge()
    import typing

    result_type = builder._expected_result_type
    # Check that it's a SQLResult type
    assert typing.get_origin(result_type) is SQLResult or result_type.__name__ == "SQLResult"


def test_merge_create_base_expression() -> None:
    """Test that _create_base_expression returns Merge expression."""
    builder = Merge()
    expression = builder._create_base_expression()
    assert isinstance(expression, exp.Merge)


# Test build output
def test_merge_build_returns_safe_query() -> None:
    """Test that build() returns SafeQuery object."""
    builder = (
        Merge()
        .into("users")
        .using("updates", "src")
        .on("users.id = src.id")
        .when_matched_then_update({"status": "active"})
    )
    query = builder.build()

    assert isinstance(query, SafeQuery)
    assert isinstance(query.sql, str)
    assert isinstance(query.parameters, dict)


def test_merge_to_statement_conversion() -> None:
    """Test conversion to SQL statement object."""
    builder = (
        Merge()
        .into("users")
        .using("updates", "src")
        .on("users.id = src.id")
        .when_matched_then_update({"status": "active"})
    )
    statement = builder.to_statement()

    assert isinstance(statement, SQL)
    # The statement and build() might format differently but should have same content
    assert "MERGE" in statement.sql
    assert "users" in statement.sql
    assert "updates" in statement.sql
    # Check that parameters contain the active value somewhere
    build_result = builder.build()
    # The parameters might be nested or in different format
    if isinstance(statement.parameters, dict):
        if "parameters" in statement.parameters:
            # Nested format
            assert statement.parameters["parameters"] == build_result.parameters
        else:
            # Direct format
            assert "active" in str(statement.parameters)


# Test fluent interface chaining
def test_merge_fluent_interface_chaining() -> None:
    """Test that all methods return builder for fluent chaining."""
    builder = (
        Merge()
        .into("target_table", "t")
        .using("source_table", "s")
        .on("t.id = s.id")
        .when_matched_then_update({"status": "updated"}, condition="s.priority > 5")
        .when_matched_then_delete(condition="s.deleted = true")
        .when_not_matched_then_insert(columns=["id", "status"], values=["s.id", "new"])
        .when_not_matched_by_source_then_update({"active": False})
    )

    query = builder.build()
    # Verify the query has all components
    assert all(
        keyword in query.sql for keyword in ["MERGE", "INTO", "USING", "ON", "WHEN", "UPDATE", "DELETE", "INSERT"]
    )


# Test edge cases
def test_merge_empty_update_dict() -> None:
    """Test MERGE with empty update dictionary."""
    builder = Merge().into("users").using("source", "src").on("users.id = src.id")

    # Empty update dict is allowed, it creates an UPDATE with no SET expressions
    result = builder.when_matched_then_update({})
    assert result is builder
    # This would create invalid SQL but the builder allows it


def test_merge_complex_source_expressions() -> None:
    """Test MERGE with complex source expressions."""
    # CTE as source
    cte_source = (
        Select()
        .with_("recent_changes", Select().select("*").from_("change_log").where(("change_date", ">", "2024-01-01")))
        .select("*")
        .from_("recent_changes")
    )

    builder = (
        Merge()
        .into("master_data")
        .using(cte_source, "chg")
        .on("master_data.id = chg.record_id")
        .when_matched_then_update({"value": "chg.new_value"})
    )

    query = builder.build()
    assert "MERGE" in query.sql
    assert "WITH" in query.sql


def test_merge_dialect_specific_behavior() -> None:
    """Test MERGE with dialect-specific SQL generation."""
    builder = Merge(dialect="snowflake")
    query = (
        builder.into("users")
        .using("updates", "src")
        .on("users.id = src.id")
        .when_matched_then_update({"last_modified": "CURRENT_TIMESTAMP()"})
        .build()
    )

    assert "MERGE" in query.sql
    assert isinstance(query.parameters, dict)
