from jinja2 import Template
from datetime import datetime
import pandas as pd
import json
import yaml
import hashprep

class HtmlReport:
    def generate(self, summary, full=False, output_file=None):
        template_str = """
<!DOCTYPE html>
<html>
<head>
    <title>Dataset Quality Report</title>
    <style>
        body { font-family: Arial, sans-serif; }
        table { border-collapse: collapse; width: 100%; }
        th, td { border: 1px solid #ddd; padding: 8px; }
        th { background-color: #f2f2f2; }
    </style>
</head>
<body>
    <h1>Dataset Quality Report</h1>
    <p>Generated: {{ generated }}</p>
    <p>HashPrep Version: {{ version }}</p>
    <h2>Executive Summary</h2>
    <ul>
        <li>Critical Issues: {{ critical_count }}</li>
        <li>Warnings: {{ warning_count }}</li>
        <li>Rows: {{ rows }}</li>
        <li>Columns: {{ columns }}</li>
    </ul>
    <h2>Issues Overview</h2>
    <table>
        <tr>
            <th>Category</th><th>Severity</th><th>Column</th><th>Description</th><th>Impact</th><th>Quick Fix</th>
        </tr>
        {% for issue in issues %}
        <tr>
            <td>{{ issue.category }}</td>
            <td>{{ issue.severity }}</td>
            <td>{{ issue.column }}</td>
            <td>{{ issue.description }}</td>
            <td>{{ issue.impact_score }}</td>
            <td>{{ issue.quick_fix | replace('\n', '<br>') }}</td>
        </tr>
        {% endfor %}
    </table>
    {% if full %}
    <h2>Dataset Preview</h2>
    <h3>Head</h3>
    {{ head_table | safe }}
    <h3>Tail</h3>
    {{ tail_table | safe }}
    <h3>Sample</h3>
    {{ sample_table | safe }}
    <h2>Variables</h2>
    {% for col, stats in variables.items() %}
    <h3>{{ col }}</h3>
    <pre><code>{{ yaml_dump(stats) }}</code></pre>
    {% endfor %}
    <h2>Correlations</h2>
    <h3>Numeric (Pearson)</h3>
    <pre><code>{{ json_dump(numeric_correlations.pearson) }}</code></pre>
    <h3>Categorical (Cramer's V)</h3>
    <table>
        <tr><th>Pair</th><th>Value</th></tr>
        {% for pair, val in categorical_correlations.items() %}
        <tr><td>{{ pair }}</td><td>{{ val | round(2) }}</td></tr>
        {% endfor %}
    </table>
    <h3>Mixed</h3>
    <table>
        <tr><th>Pair</th><th>F-Stat</th><th>P-Value</th></tr>
        {% for pair, stats in mixed_correlations.items() %}
        {% if 'error' not in stats %}
        <tr><td>{{ pair }}</td><td>{{ stats.f_stat | round(2) }}</td><td>{{ stats.p_value | round(4) }}</td></tr>
        {% endif %}
        {% endfor %}
    </table>
    <h2>Missing Values</h2>
    <table>
        <tr><th>Column</th><th>Count</th><th>Percentage</th></tr>
        {% for col, count in missing_values.count.items() %}
        {% if count > 0 %}
        <tr><td>{{ col }}</td><td>{{ count }}</td><td>{{ missing_values.percentage[col] }}</td></tr>
        {% endif %}
        {% endfor %}
    </table>
    <h2>Missing Patterns</h2>
    <pre><code>{{ json_dump(missing_patterns) }}</code></pre>
    {% endif %}
    <h2>Next Steps</h2>
    <ul>
        <li>Address critical issues</li>
        <li>Handle warnings</li>
        <li>Re-analyze dataset</li>
    </ul>
    <hr>
    <p>Generated by HashPrep</p>
</body>
</html>
        """
        template = Template(template_str)
        head_df = pd.DataFrame(summary['summaries']['head'])
        tail_df = pd.DataFrame(summary['summaries']['tail'])
        sample_df = pd.DataFrame(summary['summaries']['sample'])
        context = {
            'generated': datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
            'critical_count': summary['critical_count'],
            'warning_count': summary['warning_count'],
            'rows': summary['summaries']['dataset_info']['rows'],
            'columns': summary['summaries']['dataset_info']['columns'],
            'issues': summary['issues'],
            'full': full,
            'head_table': head_df.to_html(index=False),
            'tail_table': tail_df.to_html(index=False),
            'sample_table': sample_df.to_html(index=False),
            'variables': summary['summaries'].get('variables', {}),
            'numeric_correlations': summary['summaries'].get('numeric_correlations', {}),
            'categorical_correlations': summary['summaries'].get('categorical_correlations', {}),
            'mixed_correlations': summary['summaries'].get('mixed_correlations', {}),
            'missing_values': summary['summaries'].get('missing_values', {}),
            'missing_patterns': summary['summaries'].get('missing_patterns', {}),
            'yaml_dump': yaml.safe_dump,
            'json_dump': lambda x: json.dumps(x, indent=2),
            'version': hashprep.__version__
        }
        html_content = template.render(context)
        if output_file:
            with open(output_file, "w") as f:
                f.write(html_content)
        return html_content