import pandas as pd
import json
import yaml
import hashprep
from datetime import datetime

class MarkdownReport:
    def generate(self, summary, full=False, output_file=None):
        content = "# Dataset Quality Report\n\n"
        content += f"Generated: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}  \n"
        content += f"HashPrep Version: {hashprep.__version__}\n\n"
        content += "## Executive Summary\n"
        content += f"- Critical Issues: {summary['critical_count']}\n"
        content += f"- Warnings: {summary['warning_count']}\n"
        content += f"- Rows: {summary['summaries']['dataset_info']['rows']}\n"
        content += f"- Columns: {summary['summaries']['dataset_info']['columns']}\n\n"
        content += "## Issues Overview\n\n"
        content += "| Category | Severity | Column | Description | Impact | Quick Fix |\n"
        content += "|----------|----------|--------|-------------|--------|-----------|\n"
        for issue in summary["issues"]:
            quick_fix_inline = issue["quick_fix"].replace("\n", " ").replace("- ", "• ")
            content += f"| {issue['category']} | {issue['severity']} | {issue['column']} | {issue['description']} | {issue['impact_score']} | {quick_fix_inline} |\n"
        if full:
            content += "\n## Dataset Preview\n\n"
            content += "### Head\n\n" + pd.DataFrame(summary['summaries']['head']).to_markdown(index=False) + "\n\n"
            content += "### Tail\n\n" + pd.DataFrame(summary['summaries']['tail']).to_markdown(index=False) + "\n\n"
            content += "### Sample\n\n" + pd.DataFrame(summary['summaries']['sample']).to_markdown(index=False) + "\n\n"
            content += "## Variables\n\n"
            for col, stats in summary['summaries']['variables'].items():
                content += f"### {col}\n\n```yaml\n{yaml.safe_dump(stats, default_flow_style=False)}\n```\n"
            content += "## Correlations\n\n"
            content += "### Numeric (Pearson)\n\n```json\n" + json.dumps(summary['summaries'].get('numeric_correlations', {}).get('pearson', {}), indent=2) + "\n```\n"
            content += "### Categorical (Cramer's V)\n\n| Pair | Value |\n|------|-------|\n"
            for pair, val in summary['summaries'].get('categorical_correlations', {}).items():
                content += f"| {pair} | {val:.2f} |\n"
            content += "\n### Mixed\n\n| Pair | F-Stat | P-Value |\n|------|--------|---------|\n"
            for pair, stats in summary['summaries'].get('mixed_correlations', {}).items():
                if "error" not in stats:
                    content += f"| {pair} | {stats['f_stat']:.2f} | {stats['p_value']:.4f} |\n"
            content += "\n## Missing Values\n\n| Column | Count | Percentage |\n|--------|-------|------------|\n"
            for col, count in summary['summaries']['missing_values']['count'].items():
                pct = summary['summaries']['missing_values']['percentage'][col]
                if count > 0:
                    content += f"| {col} | {count} | {pct} |\n"
            content += "\n## Missing Patterns\n\n```json\n" + json.dumps(summary['summaries']['missing_patterns'], indent=2) + "\n```\n"
        content += "\n## Next Steps\n- Address critical issues\n- Handle warnings\n- Re-analyze dataset\n\n---\nGenerated by HashPrep"
        if output_file:
            with open(output_file, "w") as f:
                f.write(content)
        return content