"""Version validation checks."""

import re
from typing import Optional, Tuple
from ..models import Output, Context
from ..utils import run_git


def parse_version(version: str) -> Tuple[int, int, int]:
    """Parse semantic version string into components."""
    match = re.match(r"(\d+)\.(\d+)\.(\d+)", version)
    if not match:
        raise ValueError(f"Invalid version format: {version}")
    return int(match.group(1)), int(match.group(2)), int(match.group(3))


def check_version_format(
    ctx: Context, version: Optional[str] = None, **kwargs
) -> Output:
    """Check if version follows semantic versioning format."""
    if version is None:
        version = ctx.version

    try:
        major, minor, patch = parse_version(version)
        return Output(
            success=True,
            message=f"Version {version} is valid semver",
            data={"major": major, "minor": minor, "patch": patch},
        )
    except ValueError as e:
        return Output(
            success=False,
            message=str(e),
            details=[
                {"type": "text", "content": "Version must follow semantic versioning"},
                {"type": "text", "content": "Format: MAJOR.MINOR.PATCH (e.g., 1.2.3)"},
            ],
            next_steps=["Fix version in pyproject.toml to match X.Y.Z format"],
        )


def check_version_tagged(
    ctx: Context, version: Optional[str] = None, **kwargs
) -> Output:
    """Check if a version is tagged in git."""
    if version is None:
        version = ctx.version

    expected_tag = f"v{version}"
    result = run_git(["tag", "-l", expected_tag], cwd=ctx.root)

    if result.returncode != 0:
        return Output(
            success=False,
            message="Failed to check tags",
            details=[{"type": "text", "content": result.stderr.strip()}]
            if result.stderr
            else None,
        )

    if result.stdout.strip():
        return Output(
            success=True,
            message=f"Version {version} is tagged as {expected_tag}",
        )
    else:
        return Output(
            success=False,
            message=f"Version {version} is not tagged",
            details=[
                {"type": "text", "content": f"Expected tag: {expected_tag}"},
                {
                    "type": "text",
                    "content": "Releases should be tagged for traceability",
                },
            ],
            next_steps=[
                "Use: relkit bump <major|minor|patch> to create a tagged release",
            ],
        )


def check_version_not_released(
    ctx: Context, version: Optional[str] = None, **kwargs
) -> Output:
    """Check that a version hasn't already been released."""
    if version is None:
        version = ctx.version

    # Check if already tagged
    tag_check = check_version_tagged(ctx, version, **kwargs)
    if tag_check.success:
        return Output(
            success=False,
            message=f"Version {version} has already been released",
            details=[
                {"type": "text", "content": f"Tag v{version} already exists"},
                {"type": "text", "content": "Cannot release the same version twice"},
            ],
            next_steps=[
                "Bump to a new version: relkit bump <major|minor|patch>",
            ],
        )

    # Check if in changelog (as a released version, not unreleased)
    changelog_path = ctx.root / "CHANGELOG.md"
    if changelog_path.exists():
        content = changelog_path.read_text()
        # Look for version with date (indicates released)
        version_with_date_pattern = f"## [{version}] - "
        if version_with_date_pattern in content:
            return Output(
                success=False,
                message=f"Version {version} already in changelog",
                details=[
                    {
                        "type": "text",
                        "content": "Changelog shows this version was already released",
                    },
                    {
                        "type": "text",
                        "content": "Cannot release the same version twice",
                    },
                ],
                next_steps=[
                    "Bump to a new version: relkit bump <major|minor|patch>",
                ],
            )

    return Output(success=True, message=f"Version {version} has not been released")


def check_version_progression(
    ctx: Context,
    old_version: Optional[str] = None,
    new_version: Optional[str] = None,
    bump_type: Optional[str] = None,
    **kwargs,
) -> Output:
    """Check if version bump is logical (no skipping, correct type)."""
    if old_version is None:
        old_version = ctx.version

    if new_version is None and bump_type is None:
        return Output(
            success=False,
            message="Need either new_version or bump_type to check progression",
        )

    try:
        old_major, old_minor, old_patch = parse_version(old_version)

        if new_version:
            new_major, new_minor, new_patch = parse_version(new_version)
        else:
            # Calculate new version from bump_type
            if bump_type == "major":
                new_major, new_minor, new_patch = old_major + 1, 0, 0
            elif bump_type == "minor":
                new_major, new_minor, new_patch = old_major, old_minor + 1, 0
            elif bump_type == "patch":
                new_major, new_minor, new_patch = old_major, old_minor, old_patch + 1
            else:
                return Output(
                    success=False,
                    message=f"Invalid bump type: {bump_type}",
                    details=[
                        {"type": "text", "content": "Valid types: major, minor, patch"}
                    ],
                )
            new_version = f"{new_major}.{new_minor}.{new_patch}"

        # Check if it's actually an increase
        if (new_major, new_minor, new_patch) <= (old_major, old_minor, old_patch):
            return Output(
                success=False,
                message=f"Version {new_version} is not greater than {old_version}",
                details=[
                    {"type": "text", "content": "Versions must increase monotonically"},
                    {"type": "text", "content": "Cannot go backwards or stay the same"},
                ],
            )

        # Determine what kind of bump it is
        if new_major > old_major:
            actual_bump = "major"
            if new_minor != 0 or new_patch != 0:
                return Output(
                    success=False,
                    message="Major bump should reset minor and patch to 0",
                    details=[
                        {"type": "text", "content": f"Expected: {new_major}.0.0"},
                        {"type": "text", "content": f"Got: {new_version}"},
                    ],
                )
        elif new_minor > old_minor:
            actual_bump = "minor"
            if new_patch != 0:
                return Output(
                    success=False,
                    message="Minor bump should reset patch to 0",
                    details=[
                        {
                            "type": "text",
                            "content": f"Expected: {new_major}.{new_minor}.0",
                        },
                        {"type": "text", "content": f"Got: {new_version}"},
                    ],
                )
        elif new_patch > old_patch:
            actual_bump = "patch"
        else:
            return Output(
                success=False,
                message=f"Invalid version progression from {old_version} to {new_version}",
            )

        return Output(
            success=True,
            message=f"Valid {actual_bump} bump: {old_version} → {new_version}",
            data={
                "old_version": old_version,
                "new_version": new_version,
                "bump_type": actual_bump,
            },
        )

    except ValueError as e:
        return Output(
            success=False,
            message=str(e),
        )


def check_version_alignment(ctx: Context, **kwargs) -> Output:
    """Check if version is aligned across pyproject.toml, changelog, and git tags."""
    version = ctx.version
    issues = []

    # Check if version is in changelog
    changelog_path = ctx.root / "CHANGELOG.md"
    if changelog_path.exists():
        content = changelog_path.read_text()
        version_pattern = f"[{version}]"
        if version_pattern not in content:
            issues.append("Version not in CHANGELOG.md")
    else:
        issues.append("No CHANGELOG.md found")

    # Check if version is tagged
    tag_check = check_version_tagged(ctx, version, **kwargs)
    if not tag_check.success:
        issues.append("Version not tagged in git")

    if issues:
        return Output(
            success=False,
            message=f"Version {version} alignment issues",
            details=[{"type": "text", "content": issue} for issue in issues],
            next_steps=[
                "Use: relkit bump <major|minor|patch> for aligned release",
            ],
        )

    return Output(
        success=True,
        message=f"Version {version} is aligned across all systems",
    )
