import logging
import os

import pandas as pd

if os.getenv('BUILDING_SPHINX', 'false') == 'false':
    import readmet

try:
    from ._metadata import __version__, __title__
    from . import _corine
    from . import _dispersion
    from . import _geo
    from . import _tools
except ImportError:
    from _version import __version__, __title__
    import _corine
    import _dispersion
    import _geo
    import _tools

logger = logging.getLogger(__name__)


# -------------------------------------------------------------------------

def load_weather(working_dir: str, conf: dict = None,
                 file: str = None) -> pd.DataFrame:
    """
    Get the weather time series height `working_dir`.
    Files are evaluated in the same order as by AUSTAL:
    `zeitreihe.dmna` or `timeseries.dmna` are tried to read first,
    then the AKTERM file spezified in the config file under
    parameter 'az'

    :param working_dir: the working directory of austal(2000),
      where austal.txt resides. If path is a file, this file is read,
      ignoring the AUSTAL configuration.
    :type working_dir: str
    :param conf: (optional) AUSTAL configuration file contents as dict
    :type conf: dict

    :return: effective anemometer height
    :rtype: float

    If `conf` is provided, this configuration is evaluated,
    else the configuration file from `working_dir` is read.
    This option is indended for situation in which `conf`
    has already been read into memory for other purposes.
    """
    if file is not None:
        # filename given: determine type
        if file.endswith('.dmna'):
            ts_file = file
            az_file = None
        else:
            ts_file = None
            az_file = file
    else:
        az_file = None
        # dir name given: serach file
        if conf is None:
            austxt = _tools.find_austxt(working_dir)
            conf = _tools.get_austxt(austxt)
        working_dir_files = os.listdir(working_dir)
        for x in ['zeitreihe.dmna', 'timeseries.dmna']:
            if x in working_dir_files:
                ts_file = os.path.join(working_dir, x)
                break
        else:
            ts_file = None
    # load file: switch file type
    if ts_file is not None:
        # zeitreihe(timeseries).dmna generated by AUSTAL
        zr = readmet.dmna.DataFile(os.path.join(working_dir, ts_file)).data
        res = pd.DataFrame(index=pd.to_datetime(zr['te']))
        res['FF'] = zr['ua'].values
        res['DD'] = zr['ra'].values
        z0 = get_roughness_length(working_dir=working_dir, conf=conf)
        res['KM'] = [_dispersion.KM2021.get_index(z0, x) for x in zr['ra']]
    else:
        if az_file is None:
            # akterm file for AUSTAL
            if 'az' in conf:
                az_file = conf['az'][0]
            else:
                raise ValueError('no az defined, cannot weather')
        az = readmet.akterm.DataFile(
            file=os.path.join(working_dir, az_file))
        res = az.data[['FF', 'DD', 'KM']]
    return res


# -------------------------------------------------------------------------

def get_roughness_length(working_dir=None, conf=None):
    if working_dir is None:
        working_dir = _tools.DEFAULT_WORKING_DIR
    z0 = _tools.read_z0(working_dir, conf)
    if z0 is None:
        logger.info("no z0 defined, calculating mean z0")
        if conf is None:
            austxt = _tools.find_austxt(working_dir)
            conf = _tools.get_austxt(austxt)
        if 'xg' in conf and 'yg' in conf:
            xg = conf['xg']
            yg = conf['yg']
        elif 'xu' in conf and 'yu' in conf:
            xu = conf['xu']
            yu = conf['yu']
            xg, yg = _geo.ut2gk(xu, yu)
        else:

            raise ValueError("neither z0 nor position defined, "
                             "cannot determine z0")
        if 'hq' in conf:
            hq = conf['hq']
        else:
            logger.warning("no source height defined, assuming 10m")
            hq = 10.
        logger.debug('averaging z0 from local corine inventory')
        z0 = _corine.roughness_austal(xg, yg, hq)
        if z0 is None:
            logger.debug('averaging z0 from EEA Web API')
            z0 = _corine.roughness_web(xg, yg, hq)
        logger.info(f"z0 at position of wind measurement: {z0}")

    return z0

# -------------------------------------------------------------------------
