#!/usr/bin/env python
# -*- coding: UTF-8 -*-
#
# Copyright 2022-2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause

"""CLI application for various cryptographic operations."""

import base64
import glob
import hashlib
import logging
import os
import sys
from typing import Any, Optional, Union

import click
from click_option_group import optgroup

from spsdk.apps.utils import spsdk_logger
from spsdk.apps.utils.common_cli_options import (
    CommandsTreeGroup,
    spsdk_apps_common_options,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_output_option,
)
from spsdk.apps.utils.utils import INT, SPSDKAppError, catch_spsdk_error
from spsdk.crypto.certificate import Certificate, generate_extensions, generate_name
from spsdk.crypto.crypto_types import SPSDKEncoding
from spsdk.crypto.hash import EnumHashAlgorithm
from spsdk.crypto.keys import (
    PrivateKey,
    PrivateKeyEcc,
    PrivateKeyRsa,
    PrivateKeySM2,
    PublicKey,
    PublicKeyEcc,
    PublicKeyMLDSA,
    SPSDKKeyPassphraseMissing,
    get_ecc_curve,
    get_supported_keys_generators,
    prompt_for_passphrase,
)
from spsdk.crypto.signature_provider import get_signature_provider_from_config_str
from spsdk.crypto.utils import (
    extract_public_key,
    generate_img_csf_key,
    generate_key_pair,
    generate_srk_keys,
)
from spsdk.exceptions import SPSDKError, SPSDKIndexError, SPSDKSyntaxError
from spsdk.image.cert_block.rot import Rot
from spsdk.utils.config import Config
from spsdk.utils.database import get_common_data_file_path
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import (
    Endianness,
    get_printable_path,
    load_binary,
    load_secret,
    load_text,
    write_file,
)

logger = logging.getLogger(__name__)


@click.group(name="nxpcrypto", cls=CommandsTreeGroup)
@spsdk_apps_common_options
def main(log_level: int) -> None:
    """Collection of utilities for cryptographic operations."""
    spsdk_logger.install(level=log_level)


@main.command(name="digest", no_args_is_help=True)
@click.option(
    "-h",
    "--hash",
    "hash_name",
    required=True,
    type=click.Choice(list(hashlib.algorithms_available), case_sensitive=False),
    help="Name of a hash to use.",
)
@click.option(
    "-i",
    "--input-file",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to a file to digest.",
)
@click.option(
    "-c",
    "--compare",
    metavar="PATH | DIGEST",
    help="Reference digest to compare. It may be directly on the command line or fetched from a file.",
)
def digest(hash_name: str, input_file: str, compare: str) -> None:
    """Computes digest/hash of the given file."""
    data = load_binary(input_file)
    hasher = hashlib.new(hash_name.lower())
    hasher.update(data)
    hexdigest = hasher.hexdigest()
    click.echo(f"{hash_name.upper()}({input_file})= {hexdigest}")
    if compare:
        # assume comparing to a file
        if os.path.isfile(compare):
            with open(compare, encoding="utf-8") as f:
                compare_data = f.readline().strip()
                # assume format generated by openssl
                if "=" in compare_data:
                    ref = compare_data.split("=")[-1].strip()
                # assume hash is on the fist line
                else:
                    ref = compare_data
        else:
            ref = compare
        if ref.lower() == hexdigest.lower():
            click.echo("Digests are the same.")
        else:
            raise SPSDKAppError("Digests differ!")


@main.group(name="rot", cls=CommandsTreeGroup)
def rot_group() -> None:
    """Group of RoT commands."""


@rot_group.command(name="export", no_args_is_help=True)
@spsdk_family_option(families=Rot.get_supported_families())
@click.option(
    "-k",
    "--key",
    type=click.Path(exists=True, dir_okay=False),
    multiple=True,
    help="Path to one or multiple keys or certificates.",
)
@click.option(
    "-p",
    "--password",
    help="Password when using encrypted private keys.",
)
@spsdk_output_option(required=False)
def export(family: FamilyRevision, key: list[str], password: str, output: str) -> None:
    """Export RoT table."""
    _rot = Rot(family, keys_or_certs=key, password=password)
    rot_hash = _rot.export()
    click.echo(str(_rot))
    if output:
        write_file(rot_hash, path=output, mode="wb")
        click.echo(f"Result has been stored in: {output}")
    click.echo(f"RoT table: {rot_hash.hex()}")


@rot_group.command(name="calculate-hash", no_args_is_help=True)
@spsdk_family_option(families=Rot.get_supported_families())
@click.option(
    "-k",
    "--key",
    type=click.Path(exists=True, dir_okay=False),
    multiple=True,
    help="Path to one or multiple keys or certificates.",
)
@click.option(
    "-p",
    "--password",
    help="Password when using encrypted private keys.",
)
@click.option(
    "-b",
    "--base64",
    "_base64",
    is_flag=True,
    default=False,
    help="Apply base64 encoding on the final RoT hash.",
)
@spsdk_output_option(required=False)
def calculate_hash(
    family: FamilyRevision, key: list[str], password: str, output: str, _base64: bool
) -> None:
    """Calculate RoT hash."""
    _rot = Rot(family, keys_or_certs=key, password=password)
    rot_hash = _rot.calculate_hash()
    if _base64:
        rot_hash = base64.b64encode(rot_hash)
        click.echo(f"Base64 encoded RoT hash: '{rot_hash.decode('utf-8')}'")
    else:
        click.echo(f"RoT hash: '{rot_hash.hex()}'")
    if output:
        write_file(rot_hash, path=output, mode="wb")
        click.echo(f"Result has been stored in: {output}")


@main.group(name="cert", cls=CommandsTreeGroup)
def cert_group() -> None:
    """Group of command for working with x509 certificates."""


@cert_group.command(name="convert", no_args_is_help=True)
@click.option(
    "-e",
    "--encoding",
    type=click.Choice(["PEM", "DER"], case_sensitive=False),
    required=True,
    help="Desired output format.",
)
@click.option(
    "-i",
    "--input-file",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to certificate file to convert.",
)
@click.option(
    "-p",
    "--puk",
    is_flag=True,
    default=False,
    help="Extract public key instead of converting certificate.",
)
@spsdk_output_option()
def cert_convert(encoding: str, input_file: str, output: str, puk: bool) -> None:
    """Convert certificate format."""
    logger.info(f"Loading certificate from: {input_file}")
    cert = Certificate.load(input_file)
    encoding_type = {"PEM": SPSDKEncoding.PEM, "DER": SPSDKEncoding.DER}[encoding]
    if puk:
        cert.get_public_key().save(output, encoding_type)
        click.echo(f"The public key file has been created: {get_printable_path(output)}")
    else:
        cert.save(output, encoding_type)
        click.echo(f"The certificate file has been created: {get_printable_path(output)}")


class CertificateParametersConfig:  # pylint: disable=too-few-public-methods
    """Configuration object for creating the certificate."""

    def __init__(self, config: Config) -> None:
        """Initialize cert_config from yml config data."""
        try:
            self.issuer_private_key = config.get_input_file_name("issuer_private_key")
            self.issuer_private_key_password = config.get("issuer_private_key_password")
            self.subject_public_key = config.get_input_file_name("subject_public_key")
            self.serial_number = (
                config.get_int("serial_number") if "serial_number" in config else None
            )
            self.duration = config.get_int("duration") if "duration" in config else None
            self.issuer_name = generate_name(config.get("issuer"))
            self.subject_name = generate_name(config.get("subject"))
            self.extensions = None
            if "extensions" in config:
                self.extensions = generate_extensions(config.get_dict("extensions"))
            self.pss_padding = config.get("pss_padding")
        except KeyError as e:
            raise SPSDKError(f"Error found in configuration: {e} not found") from e


@cert_group.command(name="generate", no_args_is_help=True)
@spsdk_config_option()
@spsdk_output_option(force=True)
@click.option(
    "-e",
    "--encoding",
    required=False,
    type=click.Choice(["PEM", "DER"], case_sensitive=False),
    default="PEM",
    help="Encoding type. Default is PEM",
)
def generate(config: Config, output: str, encoding: str) -> None:
    """Generate certificate.

    The configuration template files could be generated by subcommand 'get-template'.
    """
    logger.info("Generating Certificate...")
    logger.info("Loading configuration from yml file...")
    cert_config = CertificateParametersConfig(config)
    try:
        password = (
            load_secret(cert_config.issuer_private_key_password)
            if cert_config.issuer_private_key_password
            else None
        )
        priv_key = PrivateKey.load(cert_config.issuer_private_key, password=password)
    except SPSDKKeyPassphraseMissing:
        password = prompt_for_passphrase()
        priv_key = PrivateKey.load(cert_config.issuer_private_key, password=password)
    pub_key = extract_public_key(cert_config.subject_public_key)

    certificate = Certificate.generate_certificate(
        subject=cert_config.subject_name,
        issuer=cert_config.issuer_name,
        subject_public_key=pub_key,
        issuer_private_key=priv_key,
        serial_number=cert_config.serial_number,
        duration=cert_config.duration,
        extensions=cert_config.extensions,
        pss_padding=cert_config.pss_padding,
    )
    logger.info("Saving the generated certificate to the specified path...")
    encoding_type = SPSDKEncoding.PEM if encoding.lower() == "pem" else SPSDKEncoding.DER
    certificate.save(output, encoding_type=encoding_type)
    logger.info("Certificate generated successfully...")
    click.echo(f"The certificate file has been created: {get_printable_path(output)}")


@cert_group.command(name="get-template", no_args_is_help=True)
@spsdk_output_option(force=True)
def get_template(output: str) -> None:
    """Generate the template of Certificate generation YML configuration file."""
    logger.info("Creating Certificate template...")
    write_file(load_text(get_common_data_file_path("certgen_config.yaml")), output)
    click.echo(
        f"The Certificate template has been saved into {get_printable_path(output)} YAML file"
    )


@cert_group.command(name="verify", no_args_is_help=True)
@click.option(
    "-c",
    "--certificate",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to certificate to verify",
)
@optgroup.group("Type of verification")
@optgroup.option(
    "-s",
    "--sign",
    type=click.Path(exists=True, dir_okay=False),
    help="Path to key to verify certificate signature",
)
@optgroup.option(
    "-p",
    "--puk",
    type=click.Path(exists=True, dir_okay=False),
    help="Path to key to verify public key in certificate",
)
def verify(certificate: str, sign: str, puk: str) -> None:
    """Verify signature or public key in certificate."""
    logger.info(f"Loading certificate from: {certificate}")
    cert = Certificate.load(certificate)
    if sign:
        logger.info("Performing signature verification")
        sign_algorithm = cert.signature_algorithm_oid._name
        logger.debug(f"Signature algorithm: {sign_algorithm}")
        if "ecdsa" not in sign_algorithm:
            raise SPSDKAppError(
                f"Unsupported signature algorithm: {sign_algorithm}. "
                "Only ECDSA signatures are currently supported."
            )
        verification_key = extract_public_key(sign)
        if not isinstance(verification_key, PublicKeyEcc):
            raise SPSDKError("Currently only ECC keys are supported.")
        if not cert.signature_hash_algorithm:
            raise SPSDKError("Certificate doesn't contain info about hashing alg.")

        if not verification_key.verify_signature(
            cert.signature,
            cert.tbs_certificate_bytes,
            EnumHashAlgorithm.from_label(cert.signature_hash_algorithm.name),
        ):
            raise SPSDKAppError("Invalid signature")
        click.echo("Signature is OK")

    if puk:
        logger.info("Performing public key verification")
        cert_puk = cert.get_public_key()
        other_puk = extract_public_key(puk)
        logger.debug(f"Certificate public key: {str(cert_puk)}")
        logger.debug(f"Other public key: {str(other_puk)}")

        if cert_puk == other_puk:
            click.echo("Public key in certificate matches the input")
        else:
            raise SPSDKAppError("Public key in certificate differs from the input")


@main.group(name="key", cls=CommandsTreeGroup)
def key_group() -> None:
    """Group of commands for working with asymmetric keys."""


@key_group.command(name="generate", no_args_is_help=True)
@click.option(
    "-k",
    "--key-type",
    type=click.Choice(list(get_supported_keys_generators()), case_sensitive=False),
    required=True,
    help="Type of key to generate",
)
@click.option(
    "-p",
    "--password",
    "password",
    metavar="PASSWORD",
    help="Password with which the output file will be encrypted. "
    "If not provided, the output will be unencrypted.",
)
@spsdk_output_option(force=True)
@click.option(
    "-e",
    "--encoding",
    type=click.Choice(list(SPSDKEncoding.all()), case_sensitive=False),
    default="PEM",
)
def key_generate(key_type: str, output: str, password: str, encoding: str) -> None:
    """NXP Key Generator Tool."""
    key_param = key_type.lower().strip()
    encoding_param = encoding.upper().strip()
    encoding_enum = SPSDKEncoding.all()[encoding_param]

    if output.endswith(".pub"):
        pub_key_path = output + ".pub"
    else:
        pub_key_path = os.path.splitext(output)[0] + ".pub"

    generators = get_supported_keys_generators()
    func, params = generators[key_param]

    private_key = func(**params)
    public_key = private_key.get_public_key()

    private_key.save(output, password if password else None, encoding=encoding_enum)
    public_key.save(pub_key_path, encoding=encoding_enum)

    click.echo(
        f"The key pair has been created: {get_printable_path(pub_key_path)}, {get_printable_path(output)}"
    )


@key_group.command(name="convert", no_args_is_help=True)
@click.option(
    "-e",
    "--encoding",
    type=click.Choice(["PEM", "DER", "RAW"], case_sensitive=False),
    required=True,
    help="Desired output format.",
)
@click.option(
    "-i",
    "--input-file",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to key file to convert.",
)
@spsdk_output_option()
@click.option(
    "-p",
    "--puk",
    is_flag=True,
    default=False,
    help="Extract public key instead of converting private key.",
)
def key_convert(encoding: str, input_file: str, output: str, puk: bool) -> None:
    """Convert Asymmetric key into various formats."""
    key_data = load_binary(input_file)
    key = reconstruct_key(key_data=key_data)
    if puk and isinstance(key, (PrivateKeyRsa, PrivateKeyEcc, PrivateKeySM2)):
        key = key.get_public_key()

    if encoding in ["PEM", "DER"]:
        encoding_type = {"PEM": SPSDKEncoding.PEM, "DER": SPSDKEncoding.DER}[encoding]
        out_data = key.export(encoding=encoding_type)
    elif encoding == "RAW":
        if not isinstance(key, (PrivateKeyEcc, PublicKeyEcc)):
            raise SPSDKError("Converting to RAW is supported only for ECC keys")
        key_size = key.key_size // 8
        if isinstance(key, PrivateKeyEcc):
            out_data = key.d.to_bytes(key_size, byteorder=Endianness.BIG.value)
        else:
            x = key.x.to_bytes(key_size, byteorder=Endianness.BIG.value)
            y = key.y.to_bytes(key_size, byteorder=Endianness.BIG.value)
            out_data = x + y
    else:
        raise SPSDKAppError("Desired output encoding format must be specified by -e/--encoding")

    write_file(out_data, output, mode="wb")


@key_group.command(name="verify", no_args_is_help=True)
@click.option(
    "-k1",
    "--key1",
    required=True,
    type=click.Path(exists=True, dir_okay=False),
    help="Path to key to verify.",
)
@click.option(
    "-k2",
    "--key2",
    required=True,
    type=click.Path(exists=True, dir_okay=False),
    help="Path to key for verification.",
)
def key_verify(key1: str, key2: str) -> None:
    """Check whether provided keys form a key pair or represent the same key.

    The key could be private key, public key, or certificate. All combination are allowed.
    In case of certificates, the public key within certificate is considered.
    To verify certificate signature use `nxpcrypto cert verify`.
    """
    if extract_public_key(key1) == extract_public_key(key2):
        click.echo("Keys match.")
    else:
        raise SPSDKAppError("Keys are NOT a valid pair!")


def reconstruct_key(
    key_data: bytes,
) -> Union[PrivateKey, PublicKey]:
    """Reconstruct Crypto key from PEM,DER or RAW data."""
    try:
        return PrivateKey.parse(key_data)
    except SPSDKError:
        pass
    try:
        return PublicKey.parse(key_data)
    except SPSDKError:
        pass
    # attempt to reconstruct key from raw data
    key_length = len(key_data)
    curve = get_ecc_curve(key_length)
    # everything under 49 bytes is a private key
    if key_length <= 48:
        # pylint: disable=invalid-name   # 'd' is regular name for private key number
        d = int.from_bytes(key_data, byteorder=Endianness.BIG.value)
        return PrivateKeyEcc.recreate(d=d, curve=curve)

    # public keys in binary form have exact sizes
    if key_length in [64, 96]:
        coord_length = key_length // 2
        x = int.from_bytes(key_data[:coord_length], byteorder=Endianness.BIG.value)
        y = int.from_bytes(key_data[coord_length:], byteorder=Endianness.BIG.value)
        return PublicKeyEcc.recreate(coor_x=x, coor_y=y, curve=curve)
    raise SPSDKError(f"Can't recognize key with length {key_length}")


@main.group(name="signature", cls=CommandsTreeGroup)
def signature_group() -> None:
    """Group of commands for working with signature."""


def cut_off_data_regions(data: bytes, regions: list[str]) -> bytes:
    """Get the data chunks from the input data.

    The regions are individual string written in python-like syntax. For example '[:0x10]'
    """
    if not regions:
        return data
    data_chunks = bytes()
    for region in regions:
        try:
            region = region.replace("[", "").replace("]", "")
            # if the region was defined as single index such as [0]
            if ":" not in region:
                idx = int(region, 0)
                data_chunks += data[idx].to_bytes(1, Endianness.BIG.value)
                continue
            start_s, end_s = region.split(":")
            start = int(start_s, 0) if start_s else 0
            end = int(end_s, 0) if end_s else len(data)
            data_chunks += data[start:end]
        except (SyntaxError, NameError, ValueError) as exc:
            raise SPSDKSyntaxError(f"Invalid region expression '{region}'") from exc
        except IndexError as exc:
            raise SPSDKIndexError(
                f"The region expression '{region}' is outside the data length {len(data)}"
            ) from exc
    return data_chunks


@signature_group.command(name="create", no_args_is_help=True)
@click.option(
    "-s",
    "--signer",
    type=click.STRING,
    required=True,
    help=f"""\b
        Signature provider configuration string or path to private key to be used for signing.
        Supported private keys:
        {", ".join(list(get_supported_keys_generators()))}.
        """,
)
@click.option(
    "-p",
    "--password",
    type=click.STRING,
    help="Password when using encrypted private keys.",
)
@click.option(
    "-a",
    "--algorithm",
    type=click.Choice(EnumHashAlgorithm.labels(), case_sensitive=False),
    help="Hash algorithm used when signing the message.",
)
@click.option(
    "-i",
    "--input-file",
    required=True,
    type=click.Path(exists=False, dir_okay=False),
    help="Path to file containing binary data to be signed.",
)
@click.option(
    "-e",
    "--encoding",
    type=click.Choice([SPSDKEncoding.NXP.value, SPSDKEncoding.DER.value], case_sensitive=False),
    default=SPSDKEncoding.DER.value,
    help="Encoding of output signature. This option is applicable only when signing with ECC keys.",
)
@click.option(
    "-pp",
    "--pss-padding",
    is_flag=True,
    default=False,
    help="Use PSS padding in case of RSA",
)
@click.option(
    "-r",
    "--regions",
    type=click.STRING,
    multiple=True,
    help="""\b
        Region(s) of data that will be signed. Multiple regions can be specified.

        Format of region option is similar to Python's list indices syntax:

        +--------------+--------------------------+
        | [1]          | Byte with index 1        |
        +--------------+--------------------------+
        | [:20]        | Fist 20 bytes            |
        +--------------+--------------------------+
        | [0x10:0x20]  | Between 0x10 and 0x20    |
        +--------------+--------------------------+
        | [-20:]       | Last 20 bytes            |
        +--------------+--------------------------+
        """,
)
@spsdk_output_option(force=True)
def signature_create(
    signer: str,
    password: str,
    algorithm: str,
    input_file: str,
    encoding: str,
    pss_padding: bool,
    regions: list[str],
    output: str,
) -> None:
    """Sign the data with given private key."""
    hash_alg = EnumHashAlgorithm.from_label(algorithm) if algorithm else None
    encoding_obj = SPSDKEncoding(encoding.upper()) if encoding else None
    signature_provider_obj = get_signature_provider_from_config_str(
        signer,
        password=password,
        pss_padding=pss_padding,
        hash_alg=hash_alg,
    )

    data = cut_off_data_regions(load_binary(input_file), regions)
    signature = signature_provider_obj.get_signature(data, encoding_obj)

    write_file(signature, output, mode="wb")
    click.echo(f"The data have been signed. Signature saved to: {output}")


@signature_group.command(name="verify", no_args_is_help=True)
@click.option(
    "-k",
    "--public-key",
    required=True,
    type=click.Path(exists=True, dir_okay=False),
    help=f"""\b
        Path to public key to be used for verification.

        Supported public keys:
        {", ".join(list(get_supported_keys_generators()))}.
        """,
)
@click.option(
    "-a",
    "--algorithm",
    type=click.Choice(EnumHashAlgorithm.labels(), case_sensitive=False),
    help="Hash algorithm used when signing the message. If not set, default algorithm will be used.",
)
@click.option(
    "-i",
    "--input-file",
    required=True,
    type=click.Path(exists=False, dir_okay=False),
    help="Path to file containing original binary data.",
)
@click.option(
    "-s",
    "--signature",
    required=True,
    type=click.Path(exists=True, dir_okay=False),
    help="Path to file containing data signature.",
)
@click.option(
    "-pp",
    "--pss-padding",
    is_flag=True,
    default=False,
    help="Indicate whether the signature uses PSS padding in case of RSA",
)
@click.option(
    "-r",
    "--regions",
    type=click.STRING,
    multiple=True,
    help="""\b
        Region(s) of data that will be signed. Multiple regions can be specified.

        Format of region option is similar to Python's list indices syntax:

        +--------------+--------------------------+
        | [1]          | Byte with index 1        |
        +--------------+--------------------------+
        | [:20]        | Fist 20 bytes            |
        +--------------+--------------------------+
        | [0x10:0x20]  | Between 0x10 and 0x20    |
        +--------------+--------------------------+
        | [-20:]       | Last 20 bytes            |
        +--------------+--------------------------+
        """,
)
def signature_verify(
    public_key: str,
    algorithm: Optional[str],
    input_file: str,
    signature: str,
    pss_padding: bool,
    regions: list[str],
) -> None:
    """Verify the given signature with public key."""
    result = signature_verify_command(
        public_key, algorithm, input_file, signature, pss_padding, regions
    )
    click.echo(f"Signature {'IS' if result else 'IS NOT'} matching the public key.")


def signature_verify_command(
    public_key: str,
    algorithm: Optional[str],
    input_file: str,
    signature: str,
    pss_padding: bool,
    regions: list[str],
) -> bool:
    """Verify the given signature with public key."""
    public = PublicKey.load(public_key)
    extra_params: dict[str, Any] = {"pss_padding": pss_padding}
    if algorithm:
        extra_params["algorithm"] = EnumHashAlgorithm.from_label(algorithm)
    signature_bin = load_binary(signature)
    data = load_binary(input_file)
    data = cut_off_data_regions(data, regions)
    result = public.verify_signature(signature_bin, data, **extra_params)

    # temporary fallback to MLDSA key verification if primary verification fails
    if not result:
        try:
            public = PublicKeyMLDSA.load(public_key)
            result = public.verify_signature(signature_bin, data, **extra_params)
        except SPSDKError:
            result = False

    return result


@main.group(name="pki-tree", cls=CommandsTreeGroup)
def pki_group() -> None:
    """Group of commands for generation of PKI tree."""


@pki_group.command(name="ahab", no_args_is_help=True)
@click.option(
    "-k",
    "--key-type",
    type=click.Choice(list(get_supported_keys_generators(basic=True)), case_sensitive=False),
    required=True,
    metavar="KEY-TYPE",
    help=f"""\b
        All possible options:
        {", ".join(list(get_supported_keys_generators(basic=True)))}.
        """,
)
@click.option(
    "-p",
    "--password",
    "password",
    metavar="PASSWORD",
    help="Password with which the keys will be encrypted. "
    "If not provided, the keys will be unencrypted.",
)
@spsdk_output_option(force=True, directory=True)
@click.option(
    "-e",
    "--encoding",
    type=click.Choice(list(SPSDKEncoding.cryptography_encodings()), case_sensitive=False),
    default="PEM",
)
@click.option(
    "-n",
    "--keys-number",
    type=click.IntRange(1, 4),
    default=4,
    help="Number of SRK keys and certificates that will be created (default 4)",
)
@click.option(
    "-d",
    "--duration",
    type=click.IntRange(1, 100),
    default=10,
    help="Duration of certificates validity in years (default 10)",
)
@click.option(
    "-s",
    "--serial",
    type=INT(),
    multiple=True,
    help="""Serial number of SRK certificates. Must be specified for each certificate. E.g.:
    -s 0x12345679 -s 0x1234567a for two certificates. Default is 0x1234567{9...c}.
    """,
)
@click.option(
    "-ca",
    "--srk-is-ca",
    is_flag=True,
    default=False,
    help="True if SRK is certificate authority. In this case SGK keys will be generated",
)
def ahab_tree_generate(
    key_type: str,
    output: str,
    password: str,
    encoding: str,
    keys_number: int,
    duration: int,
    serial: Optional[list[int]],
    srk_is_ca: bool,
) -> None:
    """Generates a basic AHAB PKI tree.

    If the SRKs are chosen to be CA certificates then this command will generate the
    following PKI tree:

    \b
                        CA Certificate
                           | | |
                  -------- + | +---------------
                /            |                 \\
                SRK0        SRK1       ...      SRK N
                |            |                   |
                |            |                   |
                SGK0        SGK1                SGK N

    where: N can be 1 to 4.


    If the SRKs are chosen to be non-CA certificates then this command will
    generate the following PKI tree:

    \b
                     CA Certificate
                          | | |
                 -------- + | +---------------
                /           |                 \\
            SRK0          SRK1       ...      SRK N

    """
    click.echo("Generating PKI Tree for AHAB")
    ahab_tree_generate_command(
        key_type, output, password, encoding, keys_number, duration, serial, srk_is_ca
    )


@pki_group.command(name="ahab-extend", no_args_is_help=True)
@click.option(
    "-e",
    "--encoding",
    type=click.Choice(list(SPSDKEncoding.cryptography_encodings()), case_sensitive=False),
    default="PEM",
)
@click.option(
    "-n",
    "--keys-number",
    type=INT(),
    default="1",
    help="Number of SRK keys and certificates that will be created (default 1)",
)
@click.option(
    "-d",
    "--duration",
    type=click.IntRange(1, 100),
    default=10,
    help="Duration of certificates validity in years (default 10)",
)
@click.option(
    "-s",
    "--serial",
    type=INT(),
    multiple=True,
    help="""Serial number of SRK certificates. If not specified, random number will be used.
    """,
)
@click.option(
    "-p",
    "--password",
    "password",
    metavar="PASSWORD",
    help="Password with which the keys will be encrypted. "
    "If not provided, the keys will be unencrypted.",
)
@click.option(
    "-i",
    "--input",
    "tree_path",
    type=click.Path(exists=True, file_okay=False),
    required=True,
    help="Path to the existing AHAB PKI tree.",
)
def ahab_tree_extend(
    tree_path: str,
    password: str,
    encoding: str,
    keys_number: int,
    duration: int,
    serial: Optional[list[int]],
) -> None:
    """Extend a basic AHAB PKI tree.

    This command will extend the existing AHAB PKI tree with additional SRK keys and certificates.
    If the SRKs are chosen to be CA certificates then this command will generate the
    following PKI tree:

    \b
                        CA Certificate
                           | | |
                  -------- + | +---------------
                /            |                 \\
                SRK1        SRK2       ...      SRK N
                |            |                   |
                |            |                   |
                SGK1        SGK2                SGK N


    """
    click.echo("Generating PKI Tree for AHAB")
    ahab_extend_tree_command(tree_path, password, encoding, keys_number, duration, serial)


@pki_group.command(name="hab", no_args_is_help=True)
@click.option(
    "-k",
    "--key-type",
    type=click.Choice(list(get_supported_keys_generators(basic=True)), case_sensitive=False),
    required=True,
    metavar="KEY-TYPE",
    help=f"""\b
        All possible options:
        {", ".join(list(get_supported_keys_generators(basic=True)))}.
        """,
)
@click.option(
    "-p",
    "--password",
    "password",
    metavar="PASSWORD",
    help="Password with which the keys will be encrypted. "
    "If not provided, the keys will be unencrypted.",
)
@spsdk_output_option(force=True, directory=True)
@click.option(
    "-e",
    "--encoding",
    type=click.Choice(list(SPSDKEncoding.cryptography_encodings()), case_sensitive=False),
    default="PEM",
)
@click.option(
    "-n",
    "--keys-number",
    type=click.IntRange(1, 4),
    default=4,
    help="Number of SRK keys and certificates that will be created (default 4)",
)
@click.option(
    "-d",
    "--duration",
    type=click.IntRange(1, 100),
    default=10,
    help="Duration of certificates validity in years (default 10)",
)
@click.option(
    "-s",
    "--serial",
    type=INT(),
    multiple=True,
    help="""Serial number of SRK certificates. Must be specified for each certificate. E.g.:
    -s 0x12345679 -s 0x1234567a for two certificates. Default is 0x1234567{9...c}.
    """,
)
@click.option(
    "-ca",
    "--srk-is-ca",
    is_flag=True,
    default=False,
    help="True if SRK is certificate authority. In this case IMG and CSF keys will be generated",
)
def hab_tree_generate(
    key_type: str,
    output: str,
    password: str,
    encoding: str,
    keys_number: int,
    duration: int,
    serial: Optional[list[int]],
    srk_is_ca: bool,
) -> None:
    """Generates a basic HABv4 PKI tree.

    If the SRKs are chosen to be CA certificate then this command will generate the
    following PKI tree:

    \b
                     CA Certificate
                          | | |
                 -------- + | +---------------
                /           |                 \\
             SRK0          SRK1       ...      SRK N
             / \\            / \\                / \\
            /   \\          /   \\              /   \\
        CSF0_0  IMG0_0  CSF1_0  IMG1_0 ... CSF N_0  IMG N_0


    If the SRKs are chosen to be non-CA certificate then this command will
    generate the following PKI Certificate:

    \b
                      CA Certificate
                          | | |
                 -------- + | +---------------
                /           |                 \\
            SRK0          SRK1       ...      SRK N

    """
    click.echo("Generating PKI Tree for HABv4")
    hab_tree_generate_command(
        key_type, output, password, encoding, keys_number, duration, serial, srk_is_ca
    )


@pki_group.command(name="hab-extend", no_args_is_help=True)
@click.option(
    "-e",
    "--encoding",
    type=click.Choice(list(SPSDKEncoding.cryptography_encodings()), case_sensitive=False),
    default="PEM",
)
@click.option(
    "-n",
    "--keys-number",
    type=INT(),
    default="1",
    help="Number of SRK keys and certificates that will be created (default 1)",
)
@click.option(
    "-d",
    "--duration",
    type=click.IntRange(1, 100),
    default=10,
    help="Duration of certificates validity in years (default 10)",
)
@click.option(
    "-s",
    "--serial",
    type=INT(),
    multiple=True,
    help="""Serial number of SRK certificates. If not specified, random number will be used.
    """,
)
@click.option(
    "-p",
    "--password",
    "password",
    metavar="PASSWORD",
    help="Password with which the keys will be encrypted. "
    "If not provided, the keys will be unencrypted.",
)
@click.option(
    "-i",
    "--input",
    "tree_path",
    type=click.Path(exists=True, file_okay=True),
    required=True,
    help="Path to the existing AHAB PKI tree or SRK certificate.",
)
def hab_tree_extend(
    tree_path: str,
    password: str,
    encoding: str,
    keys_number: int,
    duration: int,
    serial: Optional[list[int]],
) -> None:
    """Extends a basic HABv4 PKI tree.

    1) If the path provided is a path to root directory of tree, then this command will
    extend the existing HAB PKI tree with additional SRK keys and certificates and their
    corresponding CSF and IMG keys if SRK is CA certificate.

    2) If the path provided is a path to SRK certificate, then this command will generate
    additional CSF and IMG keys for the given SRK certificate.

    If the SRKs are chosen to be CA certificate then this command will generate the
    following PKI tree:

    \b
                     CA Certificate
                          | | |
                 -------- + | +---------------
                /           |                 \\
             SRK0          SRK1       ...      SRK N
             / \\            / \\                / \\
            /   \\          /   \\              /   \\
        CSF0_0  IMG0_0  CSF1_0  IMG1_0 ... CSF N_0  IMG N_0


    If the SRKs are chosen to be non-CA certificate then this command will
    generate the following PKI Certificate:

    \b
                      CA Certificate
                          | | |
                 -------- + | +---------------
                /           |                 \\
            SRK0          SRK1       ...      SRK N

    """
    click.echo("Extending PKI Tree for HAB")
    hab_extend_tree_command(tree_path, password, encoding, keys_number, duration, serial)


def ahab_tree_generate_command(
    key_type: str,
    output: str,
    password: str,
    encoding: str,
    keys_number: int,
    duration: int,
    serial: Optional[list[int]],
    srk_is_ca: bool,
) -> None:
    """Generate AHAB tree.

    :param key_type: key type
    :param output: output directory
    :param password: password for key protection
    :param encoding: encoding DER or PEM
    :param keys_number: number of keys to generate
    :param duration: duration of certificates in years
    :param serial: list of serial number of SRK certificates
    :param srk_is_ca: True if SRK is CA
    :raises SPSDKAppError: Serial number provided is lower than count of keys
    """
    if not serial:
        default_serials = [0x12345679, 0x1234567A, 0x1234567B, 0x1234567C]
        serial = default_serials[:keys_number]

    if len(serial) != keys_number:
        raise SPSDKAppError("Count of serial numbers does not match the count of certificates")

    encoding_param = encoding.upper().strip()
    encoding_enum = SPSDKEncoding.all()[encoding_param]

    keys_path = os.path.join(output, "keys")
    crts_path = os.path.join(output, "crts")
    os.makedirs(keys_path, exist_ok=True)
    os.makedirs(crts_path, exist_ok=True)

    # Generate key pair for CA keys
    key_prefix = "CA"
    ca_private_key, ca_public_key, ca_key_name, ca_cert_name = generate_key_pair(
        key_type=key_type,
        encoding=encoding,
        keys_path=keys_path,
        key_prefix=key_prefix,
        idx=0,
        is_ca=True,
        password=password,
    )
    ca_cert_path = os.path.join(crts_path, ca_cert_name)

    subject = ca_issuer = generate_name([{"COMMON_NAME": ca_key_name}])
    # generate CA certificate (self-signed certificate)
    ca_cert = Certificate.generate_certificate(
        subject=subject,
        issuer=ca_issuer,
        subject_public_key=ca_public_key,
        issuer_private_key=ca_private_key,
        serial_number=None,
        duration=duration * 365,
        extensions=generate_extensions(
            {"BASIC_CONSTRAINTS": {"ca": True}},
        ),
    )
    ca_cert.save(ca_cert_path, encoding_enum)
    click.echo(f"The CA0 certificate has been created: {ca_cert_path}")
    logger.info(ca_cert)

    generate_srk_keys(
        key_type=key_type,
        encoding=encoding,
        keys_path=keys_path,
        crts_path=crts_path,
        keys_number=keys_number,
        duration=duration,
        serial=serial,
        ca_private_key=ca_private_key,
        ca_issuer=ca_issuer,
        srk_is_ca=srk_is_ca,
        password=password,
        print_func=click.echo,
    )


def ahab_extend_tree_command(
    tree_path: str,
    password: str,
    encoding: str,
    keys_number: int,
    duration: int,
    serial: Optional[list[int]],
) -> None:
    """Extend existing AHAB PKI tree.

    :param tree_path: path to existing AHAB PKI tree
    :param password: password for key protection
    :param encoding: encoding DER or PEM
    :param keys_number: number of keys to generate
    :param duration: duration of certificates in years
    :param serial: list of serial number of SRK certificates
    :raises SPSDKAppError: Serial number provided is lower than count of keys
    :raises SPSDKAppError: Path to the existing AHAB PKI tree is invalid
    :raises SPSDKAppError: Path to the existing AHAB PKI tree is not a directory
    :raises SPSDKAppError: Path to the existing AHAB PKI tree does not contain keys and crts directories
    :raises SPSDKAppError: The existing AHAB PKI tree does not contain CA certificates
    """
    # Check if the tree path is valid
    if not os.path.exists(tree_path):
        raise SPSDKAppError(f"Path to the existing AHAB PKI tree is invalid: {tree_path}")

    # Check if the tree path is a directory
    if not os.path.isdir(tree_path):
        raise SPSDKAppError(f"Path to the existing AHAB PKI tree is not a directory: {tree_path}")

    # Check if the tree path contains the keys and crts directories
    keys_path = os.path.join(tree_path, "keys")
    crts_path = os.path.join(tree_path, "crts")
    if not os.path.exists(keys_path) or not os.path.exists(crts_path):
        raise SPSDKAppError(
            f"Path to the existing AHAB PKI tree does not contain keys and crts directories: {tree_path}"
        )

    # Find the CA certificate
    ca_cert_files = glob.glob(os.path.join(crts_path, "CA0*"))
    if not ca_cert_files:
        raise SPSDKAppError("The existing AHAB PKI tree does not contain CA certificates")

    if len(ca_cert_files) > 1:
        raise SPSDKAppError("The existing AHAB PKI tree contains more than one CA certificate")

    ca_cert_file = ca_cert_files[0]
    # Load the CA certificate
    ca_cert = Certificate.load(ca_cert_file)

    # Check if SRK certificates are CA certificates
    srk_is_ca = False
    sgk_files = glob.glob(os.path.join(crts_path, "SGK*"))
    if sgk_files:
        srk_is_ca = True

    # Load the keys that belong to the CA certificate, they should be in the keys directory
    ca_key_files = glob.glob(
        os.path.join(keys_path, f"{os.path.basename(ca_cert_file).replace('cert', 'key')}*")
    )

    if not ca_key_files:
        raise SPSDKAppError("The existing AHAB PKI tree does not contain CA keys")

    # Load the CA private key, try all the files that belong to the CA certificate
    ca_private_key = None
    for ca_key_file in ca_key_files:
        try:
            ca_private_key = PrivateKey.load(ca_key_file, password)
            break
        except SPSDKError:
            continue
    if not ca_private_key:
        raise SPSDKAppError("The existing AHAB PKI tree does not contain CA private key")

    # check if the CA private key matches the CA certificate
    if not ca_private_key.get_public_key() == ca_cert.get_public_key():
        raise SPSDKAppError("The CA private key does not match the CA certificate")

    # detect the key type of existing keys by parsing it from the key name
    supported_keys = list(get_supported_keys_generators(basic=True))
    key_type = None
    # Key type is part of the name of the key file
    for key in supported_keys:
        if key in os.path.basename(ca_key_files[0]):
            key_type = key
            break

    if not key_type:
        raise SPSDKAppError("The existing AHAB PKI tree does not contain supported key types")

    # count the number of existing SRK certs
    existing_srk_files = glob.glob(os.path.join(crts_path, "SRK*"))
    start_idx = len(existing_srk_files)

    if serial and len(serial) < keys_number:
        raise SPSDKAppError(
            f"Not enough serial numbers. Specified {len(serial)} numbers, but want to create {keys_number}. "
            "Each certificate requires a unique serial number."
        )

    generate_srk_keys(
        key_type=key_type,
        encoding=encoding,
        keys_path=keys_path,
        crts_path=crts_path,
        password=password,
        keys_number=keys_number,
        duration=duration,
        serial=serial,
        ca_issuer=ca_cert.issuer,
        ca_private_key=ca_private_key,
        srk_is_ca=srk_is_ca,
        start_idx=start_idx,
        print_func=click.echo,
    )


def hab_tree_generate_command(
    key_type: str,
    output: str,
    password: str,
    encoding: str,
    keys_number: int,
    duration: int,
    serial: Optional[list[int]],
    srk_is_ca: bool,
) -> None:
    """Generate HAB tree.

    :param key_type: key type
    :param output: output directory
    :param password: password for key protection
    :param encoding: encoding DER or PEM
    :param keys_number: number of keys to generate
    :param duration: duration of certificates in years
    :param serial: list of serial number of SRK certificates
    :param srk_is_ca: True if SRK is CA
    :raises SPSDKAppError: Serial number provided is lower than count of keys
    """
    if not serial:
        default_serials = [0x12345679, 0x1234567A, 0x1234567B, 0x1234567C]
        serial = default_serials[:keys_number]

    if len(serial) != keys_number:
        raise SPSDKAppError(
            f"Count of serial numbers does not match the count of certificates {len(serial)} != {keys_number}"
        )

    encoding_param = encoding.upper().strip()
    encoding_enum = SPSDKEncoding.all()[encoding_param]

    keys_path = os.path.join(output, "keys")
    crts_path = os.path.join(output, "crts")
    os.makedirs(keys_path, exist_ok=True)
    os.makedirs(crts_path, exist_ok=True)

    # Generate key pair for CA keys
    key_prefix = "CA"
    ca_private_key, ca_public_key, ca_key_name, ca_cert_name = generate_key_pair(
        key_type=key_type,
        encoding=encoding,
        keys_path=keys_path,
        key_prefix=key_prefix,
        idx=0,
        is_ca=True,
        password=password,
    )
    ca_cert_path = os.path.join(crts_path, ca_cert_name)

    subject = ca_issuer = generate_name([{"COMMON_NAME": ca_key_name}])
    # generate CA certificate (self-signed certificate)
    ca_cert = Certificate.generate_certificate(
        subject=subject,
        issuer=ca_issuer,
        subject_public_key=ca_public_key,
        issuer_private_key=ca_private_key,
        serial_number=None,
        duration=duration * 365,
        extensions=generate_extensions(
            {"BASIC_CONSTRAINTS": {"ca": True}},
        ),
    )
    ca_cert.save(ca_cert_path, encoding_enum)
    click.echo(f"The CA0 certificate has been created: {ca_cert_path}")
    logger.info(ca_cert)

    generate_srk_keys(
        key_type=key_type,
        encoding=encoding,
        keys_path=keys_path,
        crts_path=crts_path,
        password=password,
        keys_number=keys_number,
        duration=duration,
        serial=serial,
        ca_issuer=ca_cert.issuer,
        ca_private_key=ca_private_key,
        srk_is_ca=srk_is_ca,
        print_func=click.echo,
        use_img_csf=True,
    )


def hab_extend_tree_command(
    tree_path: str,
    password: str,
    encoding: str,
    keys_number: int,
    duration: int,
    serial: Optional[list[int]],
) -> None:
    """Extend existing HABv4 PKI tree.

    :param tree_path: path to existing HAB PKI tree
    :param password: password for key protection
    :param encoding: encoding DER or PEM
    :param keys_number: number of keys to generate
    :param duration: duration of certificates in years
    :param serial: list of serial number of SRK certificates
    :raises SPSDKAppError: Serial number provided is lower than count of keys
    :raises SPSDKAppError: Path to the existing HAB PKI tree is invalid
    :raises SPSDKAppError: Path to the existing HAB PKI tree is not a directory
    :raises SPSDKAppError: Path to the existing HAB PKI tree does not contain keys and crts directories
    :raises SPSDKAppError: The existing HAB PKI tree does not contain CA certificates
    """
    # Check if the tree path is valid
    if not os.path.exists(tree_path):
        raise SPSDKAppError(f"Path to the existing HAB PKI tree is invalid: {tree_path}")

    # Check if the tree path is a directory
    if not os.path.isdir(tree_path):
        hab_extend_srk(tree_path, password, encoding, keys_number, duration, serial)
    else:
        # Check if the tree path contains the keys and crts directories
        keys_path = os.path.join(tree_path, "keys")
        crts_path = os.path.join(tree_path, "crts")
        if not os.path.exists(keys_path) or not os.path.exists(crts_path):
            raise SPSDKAppError(
                f"Path to the existing HAB PKI tree does not contain keys and crts directories: {tree_path}"
            )

        # Find the CA certificate
        ca_cert_files = glob.glob(os.path.join(crts_path, "CA0*"))
        if not ca_cert_files:
            raise SPSDKAppError("The existing HAB PKI tree does not contain CA certificates")

        if len(ca_cert_files) > 1:
            raise SPSDKAppError("The existing HAB PKI tree contains more than one CA certificate")

        ca_cert_file = ca_cert_files[0]
        # Load the CA certificate
        ca_cert = Certificate.load(ca_cert_file)

        # Check if SRK certificates are CA certificates
        srk_is_ca = False
        img_files = glob.glob(os.path.join(crts_path, "IMG*"))
        csf_files = glob.glob(os.path.join(crts_path, "CSF*"))
        if img_files and csf_files:
            srk_is_ca = True

        # Load the keys that belong to the CA certificate, they should be in the keys directory
        ca_key_files = glob.glob(
            os.path.join(keys_path, f"{os.path.basename(ca_cert_file).replace('cert', 'key')}*")
        )

        if not ca_key_files:
            raise SPSDKAppError("The existing HAB PKI tree does not contain CA keys")

        # Load the CA private key, try all the files that belong to the CA certificate
        ca_private_key = None
        for ca_key_file in ca_key_files:
            try:
                ca_private_key = PrivateKey.load(ca_key_file, password)
                break
            except SPSDKError:
                continue
        if not ca_private_key:
            raise SPSDKAppError("The existing HAB PKI tree does not contain CA private key")

        # check if the CA private key matches the CA certificate
        if not ca_private_key.get_public_key() == ca_cert.get_public_key():
            raise SPSDKAppError("The CA private key does not match the CA certificate")

        # detect the key type of existing keys by parsing it from the key name
        supported_keys = list(get_supported_keys_generators(basic=True))
        key_type = None
        # Key type is part of the name of the key file
        for key in supported_keys:
            if key in os.path.basename(ca_key_files[0]):
                key_type = key
                break

        if not key_type:
            raise SPSDKAppError("The existing HAB PKI tree does not contain supported key types")

        # count the number of existing SRK certs
        existing_srk_files = glob.glob(os.path.join(crts_path, "SRK*"))
        start_idx = len(existing_srk_files)

        if serial and len(serial) < keys_number:
            raise SPSDKAppError(
                f"Not enough serial numbers. Specified {len(serial)} numbers, but want to create {keys_number}. "
                "Each certificate requires a unique serial number."
            )

        generate_srk_keys(
            key_type=key_type,
            encoding=encoding,
            keys_path=keys_path,
            crts_path=crts_path,
            password=password,
            keys_number=keys_number,
            duration=duration,
            serial=serial,
            ca_issuer=ca_cert.issuer,
            ca_private_key=ca_private_key,
            srk_is_ca=srk_is_ca,
            start_idx=start_idx,
            print_func=click.echo,
            use_img_csf=True,
        )


def hab_extend_srk(
    tree_path: str,
    password: str,
    encoding: str,
    keys_number: int,
    duration: int,
    serial: Optional[list[int]],
) -> None:
    """Extend existing HABv4 SRK PKI tree."""
    logger.info(
        f"Path to the existing HAB PKI tree is not a directory: {tree_path}, checking if the path SRK certificate"
    )

    try:
        srk_cert = Certificate.load(tree_path)
    except SPSDKError as exc:
        raise SPSDKAppError(
            f"Path to the existing HAB PKI tree is not a directory and not a valid certificate: {tree_path}"
        ) from exc

    keys_path = os.path.join(os.path.dirname(tree_path), "..", "keys")
    crts_path = os.path.join(os.path.dirname(tree_path), "..", "crts")
    if not os.path.exists(keys_path) or not os.path.exists(crts_path):
        raise SPSDKAppError(
            "Path to the existing HAB PKI tree does not contain keys and crts directories: "
            f"{keys_path}, {crts_path}"
        )

    # Load the private key file that belongs to the SRK certificate
    srk_private_key_file = os.path.join(
        keys_path,
        f"{os.path.basename(tree_path).replace('cert', 'key')}",
    )

    try:
        srk_private_key = PrivateKey.load(srk_private_key_file, password)
    except SPSDKError as exc:
        raise SPSDKAppError(
            f"Cannot load the private key file that belongs to the SRK certificate: {srk_private_key_file}"
        ) from exc

    # Check if the private key belongs to the certificate
    if not srk_private_key.get_public_key() == srk_cert.get_public_key():
        raise SPSDKAppError("The SRK private key does not match the SRK certificate")

    # detect the key type of existing key by parsing it from the key name
    supported_keys = list(get_supported_keys_generators(basic=True))
    key_type = None
    # Key type is part of the name of the key file
    for key in supported_keys:
        if key in os.path.basename(srk_private_key_file):
            key_type = key
            break

    if not key_type:
        raise SPSDKAppError("The existing HAB PKI tree does not contain supported key types")

    # Detect the index of the SRK certificate number after SRK prefix
    # for example SRK0_rsa4096_ca_cert.pem - the index is 0
    try:
        srk_idx = int(os.path.basename(tree_path).split("_")[0].replace("SRK", ""))
    except ValueError as exc:
        raise SPSDKAppError(
            f"Cannot detect the index of the SRK certificate number from the path: {tree_path}"
        ) from exc

    # Get number of IMG certs to generate that belong to the selected SRK index
    img_files = glob.glob(os.path.join(crts_path, f"IMG{srk_idx}*"))
    start_idx = len(img_files)

    if serial and len(serial) <= srk_idx:
        serial = [serial[0]] * (srk_idx + 1) if serial else None

    for key_number in range(keys_number):
        generate_img_csf_key(
            key_type=key_type,
            encoding=encoding,
            keys_path=keys_path,
            crts_path=crts_path,
            password=password,
            duration=duration,
            serial=serial,
            ca_issuer=srk_cert.issuer,
            srk_private_key=srk_private_key,
            idx=srk_idx,
            print_func=click.echo,
            key_postfix=str(start_idx + key_number),
        )


@catch_spsdk_error
def safe_main() -> None:
    """Call the main function."""
    sys.exit(main())  # pylint: disable=no-value-for-parameter


if __name__ == "__main__":
    safe_main()
