#!/usr/bin/env python
# -*- coding: UTF-8 -*-
#
# Copyright 2023-2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause
"""NXPDICE application is designed to cover DICE-related operations."""

import logging
import os
import secrets
import sys
import textwrap
from typing import Optional

import click
import colorama
import prettytable
from click_option_group import RequiredMutuallyExclusiveOptionGroup, optgroup

from spsdk.apps.utils import spsdk_logger
from spsdk.apps.utils.common_cli_options import (
    FC,
    CommandsTreeGroup,
    Config,
    port_option,
    spsdk_apps_common_options,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_mboot_interface,
    spsdk_output_option,
    timeout_option,
)
from spsdk.apps.utils.utils import SPSDKAppError, catch_spsdk_error
from spsdk.crypto.keys import EccCurve, PrivateKeyEcc
from spsdk.dice.data_container import TPDataContainer
from spsdk.dice.models import APIResponse, DICEResponse
from spsdk.dice.service_local import LocalDICEVerificationService
from spsdk.dice.service_remote import DICEVerificationService, RemoteDICEVerificationService
from spsdk.dice.target_blhost import BlhostDICETarget, DICETarget
from spsdk.dice.target_model import ModelDICETarget
from spsdk.dice.utils import HADDiff, HADDifferences, ProveGenuinity, get_supported_devices
from spsdk.mboot.interfaces.uart import MbootUARTInterface
from spsdk.mboot.protocol.base import MbootProtocolBase
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import load_binary, write_file
from spsdk.utils.registers import Register, RegsBitField

logger = logging.getLogger(__name__)


def service_options(options: FC) -> FC:
    """Verification service selection options.

    Provides: service_url: str, database: str
    """
    options = optgroup.option(
        "-db",
        "--database",
        type=click.Path(dir_okay=False),
        help="Path to local database instead of service-url.",
    )(options)
    options = optgroup.option(
        "-su",
        "--service-url",
        type=str,
        help="DICE verification service URL. Example: http://localhost:8080",
    )(options)
    # The group is created at the end, because the options are backwards
    options = optgroup(
        "Verification service selection",
        cls=RequiredMutuallyExclusiveOptionGroup,
    )(options)

    return options


def models_options(options: FC) -> FC:
    """Provides path to MCU model files: models_dir: str."""
    options = click.option(
        "-md",
        "--models-dir",
        required=False,
        type=click.Path(file_okay=False, exists=True),
        help=(
            "Path to folder with MCU model files. "
            "When using models the `--port` option is used as sub-folder name in `models-dir`"
        ),
    )(options)
    return options


def store_artifact_options(options: FC) -> FC:
    """Provides a path where to store artifacts: store-artifact: str."""
    options = click.option(
        "-s",
        "--store-artifact",
        required=False,
        type=click.Path(dir_okay=False),
        help="Path where to store artifact (data) generated by the command.",
    )(options)
    return options


def get_verification_service(
    service_url: Optional[str] = None, database: Optional[str] = None
) -> DICEVerificationService:
    """Factory method for retrieving concrete verification service instance."""
    if service_url and database:
        raise SPSDKAppError("Only one of 'service_url' and 'database must be specified!")
    if service_url:
        return RemoteDICEVerificationService(base_url=service_url)
    if database:
        return LocalDICEVerificationService(file_path=database)
    raise SPSDKAppError("One of 'service_url' and 'database must be specified!")


def get_dice_target(
    port: str,
    timeout: int = 5000,
    family: Optional[FamilyRevision] = None,
    models_dir: Optional[str] = None,
) -> DICETarget:
    """Factory method for retrieving concrete DICE target instance."""
    if port is None:
        raise SPSDKAppError("-p/--port needs to be set.")
    if models_dir:
        return ModelDICETarget(models_dir=models_dir, port=port.split(",")[0])
    assert isinstance(family, FamilyRevision)
    interfaces = MbootUARTInterface.scan(port=port, timeout=timeout)
    if len(interfaces) == 0:
        raise SPSDKAppError(f"Could not found device on '{port}'")
    assert isinstance(interfaces[0], MbootProtocolBase)
    return BlhostDICETarget(family=family, interface=interfaces[0])


@click.group(name="nxpdice", cls=CommandsTreeGroup)
@spsdk_apps_common_options
def main(log_level: int) -> int:
    """Application designed to cover DICE-related operations."""
    spsdk_logger.install(log_level or logging.WARNING)
    return 0


@main.command(name="register-ca-puk", no_args_is_help=True)
@service_options
@port_option()
@timeout_option()
@spsdk_family_option(families=get_supported_devices())
@models_options
@click.option(
    "--rkth",
    type=str,
    required=True,
    help="HEX value of RKTH",
)
@store_artifact_options
def register_ca_puk(
    service_url: str,
    database: str,
    port: str,
    timeout: int,
    family: FamilyRevision,
    store_artifact: str,
    models_dir: str,
    rkth: str,
) -> None:
    """Get NXP_CUST_DICE_CA_PUK from the device and register it in the verification service."""
    rkth_bytes = bytes.fromhex(rkth)
    if len(rkth_bytes) != 32:
        raise SPSDKAppError(f"RKTH must be 32B long. Got {len(rkth_bytes)}")
    target = get_dice_target(port=port, timeout=timeout, family=family, models_dir=models_dir)
    service = get_verification_service(service_url=service_url, database=database)

    puk_data = target.get_ca_puk(rkth=rkth_bytes)
    click.echo(f"Registering NXP_CUST_DICE_CA_PUK: {puk_data.hex()}")
    if store_artifact:
        write_file(data=puk_data, path=store_artifact, mode="wb")

    response = service.register_dice_ca_puk(key_data=puk_data)
    display_response(response=response)


@main.command(name="get-ca-puk", no_args_is_help=True)
@port_option()
@timeout_option()
@spsdk_family_option(families=get_supported_devices())
@models_options
@click.option(
    "-o",
    "--output",
    type=click.Path(dir_okay=False),
    required=True,
    help="Path where to store the NXP_CUST_DICE_CA_PUK",
)
@click.option(
    "--rkth",
    type=str,
    required=True,
    help="HEX value of RKTH",
)
@click.option(
    "--mldsa",
    is_flag=True,
    help="Get MLDSA public key instead of default ECC PUK",
)
def get_ca_puk(
    port: str,
    timeout: int,
    family: FamilyRevision,
    models_dir: str,
    output: str,
    rkth: str,
    mldsa: bool,
) -> None:
    """Get NXP_CUST_DICE_CA_PUK from the device."""
    rkth_bytes = bytes.fromhex(rkth)
    target = get_dice_target(port=port, timeout=timeout, family=family, models_dir=models_dir)

    puk_data = target.get_ca_puk(rkth=rkth_bytes, mldsa=mldsa)
    write_file(data=puk_data, path=output, mode="wb")


@main.command(name="upload-ca-puk", no_args_is_help=True)
@service_options
@click.option(
    "-c",
    "--ca-puk",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to binary file containing NXP_CUST_DICE_CA_PUK key.",
)
def upload_ca_puk(service_url: str, database: str, ca_puk: str) -> None:
    """Upload existing NXP_CUST_DICE_CA_PUK into the verification service."""
    with open(ca_puk, "rb") as f:
        puk_data = f.read()
    if len(puk_data) != 64:
        raise SPSDKAppError(f"NXP_CUST_DICE_CA_PUK should be 64B long. Got: {len(puk_data)}")

    service = get_verification_service(service_url=service_url, database=database)
    click.echo(f"Registering NXP_CUST_DICE_CA_PUK: {puk_data.hex()}")
    response = service.register_dice_ca_puk(key_data=puk_data)
    display_response(response=response)


@main.command(name="register-version", no_args_is_help=True)
@service_options
@port_option()
@timeout_option()
@spsdk_family_option(families=get_supported_devices())
@models_options
@store_artifact_options
def register_version(
    service_url: str,
    database: str,
    port: str,
    timeout: int,
    family: FamilyRevision,
    store_artifact: str,
    models_dir: str,
) -> None:
    """Register new FW version, RTF, and HAD based on DICE response."""
    target = get_dice_target(port=port, timeout=timeout, family=family, models_dir=models_dir)
    service = get_verification_service(service_url=service_url, database=database)

    response_data = target.get_dice_response(challenge=secrets.token_bytes(32))
    if store_artifact:
        write_file(data=response_data, path=store_artifact, mode="wb")

    response = service.register_version(data=response_data)
    display_response(response=response)


@main.command(name="upload-version", no_args_is_help=True)
@service_options
@click.option(
    "-r",
    "--response",
    "response_file",
    type=click.Path(exists=True, dir_okay=False),
    help="Path to DICE response binary. Info in response will be used to register new version.",
)
def upload_version(service_url: str, database: str, response_file: str) -> None:
    """Upload existing DICE response to register new FW version, RTF, and HAD."""
    with open(response_file, "rb") as f:
        response_data = f.read()
    if len(response_data) != 324:
        raise SPSDKAppError(f"DICE response should be 324B long. Got: {len(response_data)}")

    service = get_verification_service(service_url=service_url, database=database)
    response = service.register_version(data=response_data)
    display_response(response=response)


@main.command(name="verify", no_args_is_help=True)
@service_options
@port_option()
@timeout_option()
@spsdk_family_option(families=get_supported_devices())
@models_options
@store_artifact_options
def verify(
    service_url: str,
    database: str,
    port: str,
    timeout: int,
    family: FamilyRevision,
    store_artifact: str,
    models_dir: str,
) -> None:
    """Perform the DICE attestation verification."""
    target = get_dice_target(port=port, timeout=timeout, family=family, models_dir=models_dir)
    service = get_verification_service(service_url=service_url, database=database)

    challenge = service.get_challenge()
    logger.info(f"Challenge: {challenge.hex()}")
    dice_response = target.get_dice_response(challenge=challenge)
    click.echo(f"Submitting DICE Response:\n{DICEResponse.parse(data=dice_response).info()}")
    if store_artifact:
        write_file(data=dice_response, path=store_artifact, mode="wb")

    response = service.verify(data=dice_response)
    display_had_diff(response=response, family=family)
    display_response(response=response)


@main.command(name="get-response", no_args_is_help=True)
@port_option()
@timeout_option()
@spsdk_family_option(families=get_supported_devices())
@models_options
@click.option(
    "--response",
    type=click.Path(dir_okay=False),
    required=True,
    help="Path where to store the DICE response",
)
@click.option(
    "-c",
    "--challenge",
    help="Optional challenge. If not specified a random challenge will be used.",
)
def get_response(
    port: str, timeout: int, family: FamilyRevision, models_dir: str, response: str, challenge: str
) -> None:
    """Get DICE response from the device."""
    if challenge:
        challenge_data = bytes.fromhex(challenge)
    else:
        challenge_data = secrets.token_bytes(32)
    if len(challenge_data) != 32:
        raise SPSDKAppError(f"Challenge must be 32B. Got: {len(challenge_data)}")

    target = get_dice_target(port=port, timeout=timeout, family=family, models_dir=models_dir)
    response_data = target.get_dice_response(challenge=challenge_data)
    write_file(data=response_data, path=response, mode="wb")


@main.command(name="upload-response", no_args_is_help=True)
@service_options
@click.option(
    "-r",
    "--response",
    "response_file",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to binary file containing the DICE response.",
)
def upload_response(service_url: str, database: str, response_file: str) -> None:
    """Upload existing DICE response for verification."""
    with open(response_file, "rb") as f:
        dice_response = f.read()
    if len(dice_response) != 324:
        raise SPSDKAppError(f"DICE response should be 324 B long. Got: {len(dice_response)}")
    click.echo(f"Submitting DICE Response:\n{DICEResponse.parse(data=dice_response).info()}")

    service = get_verification_service(service_url=service_url, database=database)
    response = service.verify(data=dice_response, reset_challenge=True)
    display_response(response=response)


@main.command(name="create-models", no_args_is_help=True)
@click.option(
    "-md",
    "--models-dir",
    type=click.Path(file_okay=False, exists=False),
    help="Path to directory where to create models directory for debugging purposes",
)
@click.option(
    "-n",
    "--number",
    type=click.IntRange(min=1),
    required=True,
    help="Number of virtual devices to crate for debugging purposes.",
)
@click.option(
    "-p",
    "--prefix",
    type=str,
    default="com9",
    help="Prefix for device model names. Number of device will be appended to the prefix.",
)
def create_models(models_dir: str, number: int, prefix: str) -> None:
    """Create models directory for debugging purposes."""
    click.echo(f"Creating root models folder: {models_dir}")
    os.makedirs(models_dir)
    dice_ca_prk = PrivateKeyEcc.generate_key(EccCurve.SECP256R1)
    dice_ca_prk.save(file_path=os.path.join(models_dir, "dice_ca_prk.pem"))

    dice_ca_puk = dice_ca_prk.get_public_key()
    dice_ca_puk.save(file_path=os.path.join(models_dir, "dice_ca_puk.pem"))

    rtf = secrets.token_hex(32)
    # default (after-reset) HAD value
    had = "0ff0035500000022ffffffff000000000000dc000000018f000000000000000000000000000000000000000000000000"
    with open(os.path.join(models_dir, "config.yaml"), "w", encoding="utf-8") as f:
        f.writelines(
            [
                "dice_ca_prk: dice_ca_prk.pem\n",
                "dice_ca_puk: dice_ca_puk.pem\n",
                f"rtf: {rtf}\n",
                f"had: {had}\n",
                "version: 1\n",
            ]
        )
    click.echo("Root model directory created")
    for i in range(number):
        click.get_current_context().invoke(add_device, models_dir=models_dir, name=f"{prefix}{i}")


@main.command(name="add-device", no_args_is_help=True)
@models_options
@click.option("-n", "--name", type=str, required=True, help="Name for the device")
def add_device(models_dir: str, name: str) -> None:
    """Add virtual device to the models_dir."""
    click.echo(f"Creating device model: {name}")
    os.makedirs(os.path.join(models_dir, name))
    die_prk = PrivateKeyEcc.generate_key(EccCurve.SECP256R1)
    die_prk.save(file_path=os.path.join(models_dir, name, "die_prk.pem"))

    die_puk = die_prk.get_public_key()
    die_puk.save(file_path=os.path.join(models_dir, name, "die_puk.pem"))

    uuid = secrets.token_hex(16)
    with open(os.path.join(models_dir, name, "config.yaml"), "w", encoding="utf-8") as f:
        f.writelines(
            [
                f"uuid: {uuid}\n",
                "die_prk: die_prk.pem\n",
                "die_puk: die_puk.pem\n",
                "dice_ca_puk: \n",
                "dice_ca_prk: \n",
                "rtf: \n",
                "had: \n",
                "version: \n",
                "challenge: \n",
            ]
        )


def display_response(response: APIResponse) -> None:
    """Colorized display of a APIResponse."""
    string = colorama.Fore.GREEN if response.success else colorama.Fore.LIGHTRED_EX
    string += response.message + colorama.Fore.RESET
    click.echo(message=string)
    click.get_current_context().exit(0 if response.success else 1)


def display_had_diff(response: APIResponse, family: FamilyRevision) -> None:
    """Display detailed breakdown of HAD differences."""
    if not response.status == "HAD_DIFF":
        return
    if not response.expected_had or not response.actual_had:
        raise SPSDKAppError("HAD data is missing in the response")

    click.echo(colorama.Fore.YELLOW + "HAD difference detected!" + colorama.Fore.RESET)
    click.echo(f"Expected: {response.expected_had}")
    click.echo(f"Actual  : {response.actual_had}")
    had = HADDiff(family=family)
    differences = had.get_diff(response.expected_had, response.actual_had)

    if len(differences) == 0:
        click.echo(
            colorama.Fore.YELLOW
            + "Differences occurred only in RESERVED fields"
            + colorama.Fore.RESET
        )
        return

    display_had_diff_table(differences=differences)

    critical_differences = had.get_diff(
        response.expected_had, response.actual_had, critical_only=True
    )
    if len(critical_differences) == 0:
        return

    click.echo(
        colorama.Fore.LIGHTRED_EX + "Critical HAD differences detected!" + colorama.Fore.RESET
    )
    display_had_diff_table(
        differences=critical_differences, title="Critical HAD differences breakdown"
    )
    click.echo(
        colorama.Fore.LIGHTRED_EX
        + "DICE response verification failed due to critical HAD differences"
        + colorama.Fore.RESET
    )
    click.get_current_context().exit(1)


def display_had_diff_table(
    differences: HADDifferences, title: str = "HAD differences breakdown"
) -> None:
    """Display the HAD differences in a table."""
    table = prettytable.PrettyTable(
        title=title,
        field_names=["register", "bit-field", "expected", "actual", "description"],
        header_style="cap",
        hrules=True,
    )
    table.align["description"] = "l"

    def format_bitfield(b: RegsBitField) -> str:
        if b.width == 1:
            return str(b.get_value())
        form = f"#0{b.width + 2}b"
        return f"{b.get_value():{form}}"

    for d1, d2 in differences:
        if isinstance(d1, Register):
            table.add_row(
                [
                    d1.name,
                    "---",
                    d1.get_hex_value(),
                    d2.get_hex_value(),
                    "\n".join(textwrap.wrap(d1.description, 50)),
                ]
            )
        else:
            # oh dead Mypy, Pyright knows this, why don't you?
            assert isinstance(d1, RegsBitField) and isinstance(d2, RegsBitField)
            table.add_row(
                [
                    d1.parent.name,
                    d1.name,
                    format_bitfield(d1),
                    format_bitfield(d2),
                    "\n".join(textwrap.wrap(d1.description, 50)),
                ]
            )
    click.echo(table.get_formatted_string())


@main.command(name="get-pg-response", no_args_is_help=True)
@spsdk_mboot_interface()
@spsdk_family_option(families=get_supported_devices())
@spsdk_output_option()
@click.option(
    "-m",
    "--mode",
    type=click.Choice(ProveGenuinity.Mode.labels(), case_sensitive=False),
    default="ECDSA",
    show_default=True,
    help="Prove genuinity mode",
)
@click.option(
    "-ch",
    "--challenge",
    help=(
        "Challenge to be used for proving genuinity (16B = 32 chars). "
        "If not specified a random challenge will be generated"
    ),
    required=False,
)
def prove_genuinity_get_response(
    interface: MbootProtocolBase,
    family: FamilyRevision,
    output: str,
    challenge: Optional[str] = None,
    mode: str = "ECDSA",
) -> None:
    """Get Prove-Genuinity response from the target."""
    if not challenge:
        challenge_bytes = secrets.token_bytes(16)
        click.echo(f"Generated random challenge: {challenge_bytes.hex()}")
    else:
        challenge_bytes = bytes.fromhex(challenge)

    mode_enum = ProveGenuinity.Mode.from_label(mode.upper())
    pg_resp = ProveGenuinity.get_response(
        family=family,
        interface=interface,
        challenge=challenge_bytes,
        mode=mode_enum,
    )
    write_file(data=pg_resp, path=output, mode="wb")


@main.command(name="verify-pg-response", no_args_is_help=True)
@click.option(
    "-r",
    "--response",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to the Prove Genuinity response file",
)
@click.option(
    "-k",
    "--key-file",
    type=click.Path(exists=True, dir_okay=False),
    help="Path(s) to public key file(s) for verification",
    required=True,
    multiple=True,
)
@click.option(
    "-ch",
    "--challenge",
    help=(
        "Challenge used for proving genuinity (16B = 32 chars). "
        "If not specified, challenge check will be skipped."
    ),
    required=False,
)
def verify_pg_response(response: str, challenge: Optional[str], key_file: list[str]) -> None:
    """Verify the Prove Genuinity response."""
    response_data = load_binary(response)
    keys = [load_binary(key) for key in key_file]
    result = ProveGenuinity.verify_response(
        response=response_data,
        keys=keys,
        challenge=bytes.fromhex(challenge) if challenge else None,
    )
    if result:
        click.secho("Prove Genuinity response verification successful!", fg="green")
    else:
        click.secho("Prove Genuinity response verification failed!", fg="red")
        click.get_current_context().exit(1)


@main.command(name="verify-csr", no_args_is_help=True)
@click.option(
    "-c",
    "--csr-file",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to the DICE CSR file",
)
@click.option(
    "-pk",
    "--product-key-file",
    type=click.Path(exists=True, dir_okay=False),
    help="Path(s) to NXP PROD public key file(s) for verification",
    required=True,
    multiple=True,
)
@click.option(
    "-dk",
    "--dice-ca-key-file",
    type=click.Path(exists=True, dir_okay=False),
    help="Path(s) to DICE CA public key file(s) for verification",
    required=True,
    multiple=True,
)
@click.option(
    "-ch",
    "--challenge",
    help=(
        "Challenge used for proving genuinity (16B = 32 chars). "
        "If not specified, challenge check will be skipped."
    ),
    required=False,
)
@click.option(
    "-ea",
    "--export-alias-keys",
    type=click.Path(dir_okay=False),
    help=(
        "Path to export DICE Alias public keys. "
        "(entry serves as a prefix for all DICE Alias keys found in the CSR)"
    ),
)
@click.option("--strict", is_flag=True, help="Enable strict verification mode")
def verify_csr(
    csr_file: str,
    product_key_file: list[str],
    dice_ca_key_file: list[str],
    challenge: Optional[str] = None,
    export_alias_keys: Optional[str] = None,
    strict: bool = False,
) -> None:
    """Verify the DICE CSR. Optionally export the DICE Alias keys."""
    csr_data = load_binary(csr_file)
    prod_keys = [load_binary(key) for key in product_key_file]
    dice_ca_keys = [load_binary(key) for key in dice_ca_key_file]

    result, alias_keys = ProveGenuinity.verify_csr(
        csr_data=csr_data,
        prod_keys=prod_keys,
        dice_ca_keys=dice_ca_keys,
        challenge=bytes.fromhex(challenge) if challenge else None,
        extract_alias_keys=export_alias_keys is not None,
        strict=strict,
        print_fn=click.echo,
    )

    if result:
        click.secho("DICE CSR verification successful!", fg="green")
    else:
        click.secho("DICE CSR verification failed!", fg="red")
        click.get_current_context().exit(1)

    if export_alias_keys and alias_keys:
        for i, key in enumerate(alias_keys):
            export_path = f"{export_alias_keys}_{i}.pem"
            click.echo(f"Exporting DICE Alias key #{i} ({repr(key)}) to {export_path}")
            key.save(export_path)


@main.command(name="print", no_args_is_help=True)
@click.option(
    "-r",
    "--response",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to the Prove Genuinity response or CSR file",
)
def print_file(response: str) -> None:
    """Print the contents of a Prove Genuinity response or CSR file."""
    container = TPDataContainer.load(response)
    click.echo(container)


@main.command(name="get-fmc-config", no_args_is_help=True)
@spsdk_output_option()
def get_fmc_config(output: str) -> None:
    """Get config file for creating FMC certificate template."""
    import shutil
    from pathlib import Path

    from spsdk import SPSDK_DATA_FOLDER

    Path(output).parent.mkdir(exist_ok=True)
    src = Path(SPSDK_DATA_FOLDER) / "common" / "dice" / "fmc_template.yaml"

    shutil.copy(src, output)


@main.command(name="get-fmc-container", no_args_is_help=True)
@spsdk_config_option()
def get_fmc_container(config: Config) -> None:
    """Generate the FMC Alias certificate container."""
    from spsdk.dice.gen_alias import generate_fmc

    generate_fmc(config=config)


@main.command(name="make-fmc-container", no_args_is_help=True)
@spsdk_config_option()
def make_fmc_container(config: Config) -> None:
    """Create TP container from existing template and descriptor."""
    from spsdk.dice.gen_alias import make_container

    make_container(config=config)


@main.command(name="make-idevid-cert", no_args_is_help=True)
@click.option(
    "-r",
    "--response",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to the Prove Genuinity response file",
)
@click.option(
    "-sn",
    "--subject-name",
    required=True,
    help="Subject common name for the iDevID certificate",
)
@click.option(
    "-p",
    "--product-puk",
    type=click.Path(exists=True, dir_okay=False),
    required=False,
    multiple=True,
    help="Path to the product public key file(s)",
)
@click.option(
    "-cr",
    "--ca-prk-key",
    type=click.Path(exists=True, dir_okay=False),
    required=True,
    help="Path to the CA private key file for signing",
)
@click.option(
    "-cu",
    "--ca-puk-key",
    type=click.Path(exists=True, dir_okay=False),
    required=False,
    help="Path to the CA public key file for issuer setup",
)
@click.option(
    "-cn",
    "--ca-name",
    required=False,
    help="CA name for issuer setup",
)
@click.option(
    "--use-full-der-for-serial",
    is_flag=True,
    default=False,
    help="Use full DER encoding for subject serial number calculation",
)
@spsdk_output_option(help="Path to output the generated iDevID certificate")
@click.pass_context
def make_idevid_cert(
    ctx: click.Context,
    response: str,
    subject_name: str,
    ca_prk_key: str,
    ca_puk_key: str,
    ca_name: str,
    product_puk: list[str],
    output: str,
    use_full_der_for_serial: bool,
) -> None:
    """Create an iDevID certificate from a Prove Genuinity response."""
    if product_puk:
        ctx.invoke(verify_pg_response, response=response, key_file=product_puk)
    else:
        click.echo("PROD public keys were not specified, PG response verification will be skipped.")
    response_data = load_binary(response)
    ca_prk_data = load_binary(ca_prk_key)
    if ca_puk_key:
        ca_puk_data = load_binary(ca_puk_key)
    else:
        ca_puk_data = None

    if not (ca_puk_data and ca_name):
        click.echo(
            "CA public key or CA name not provided. Certificate will not contain issuer details."
        )
    cert_data = ProveGenuinity.create_cert(
        response=response_data,
        subject_common_name=subject_name,
        ca_prk=ca_prk_data,
        ca_puk=ca_puk_data,
        ca_name=ca_name,
        use_full_der_for_serial=use_full_der_for_serial,
    )
    write_file(cert_data, output, mode="wb")


@catch_spsdk_error
def safe_main() -> None:
    """Call the main function."""
    sys.exit(main())  # pylint: disable=no-value-for-parameter


if __name__ == "__main__":
    safe_main()
