#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause
"""Nxpimage AHAB group."""

import logging
import os
from typing import Optional

import click

from spsdk.apps.utils.common_cli_options import (
    CommandsTreeGroup,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_output_option,
)
from spsdk.apps.utils.utils import INT, SPSDKAppError, print_files, print_verifier_to_console
from spsdk.crypto.signature_provider import get_signature_provider_from_config_str
from spsdk.exceptions import SPSDKError
from spsdk.image.ahab.ahab_certificate import AhabCertificate, get_ahab_certificate_class
from spsdk.image.ahab.ahab_data import AhabTargetMemory, FlagsSrkSet
from spsdk.image.ahab.ahab_image import AHABImage
from spsdk.image.ahab.utils import (
    ahab_fix_signature_block_version,
    ahab_re_sign,
    ahab_sign_image,
    ahab_update_keyblob,
)
from spsdk.image.bootable_image.bimg import BootableImage
from spsdk.utils.config import Config
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import get_printable_path, load_binary, load_hex_string, write_file
from spsdk.utils.verifier import Verifier, VerifierResult

logger = logging.getLogger(__name__)


@click.group(name="ahab", cls=CommandsTreeGroup)
def ahab_group() -> None:
    """Group of sub-commands related to AHAB."""


@ahab_group.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=AHABImage)
def ahab_export_command(config: Config) -> None:
    """Generate AHAB Image from YAML/JSON configuration.

    The configuration template files could be generated by subcommand 'get-template'.
    """
    ahab_export(config)


def ahab_export(config: Config) -> None:
    """Generate AHAB Image from YAML/JSON configuration."""
    ahab = AHABImage.load_from_config(config)
    ahab.update_fields()
    ahab.verify().validate()
    ahab_data = ahab.export()

    ahab_output_file_path = config.get_output_file_name("output")
    write_file(ahab_data, ahab_output_file_path, mode="wb")

    logger.info(f"Created AHAB Image:\n{str(ahab.image_info())}")
    logger.info(f"Created AHAB Image memory map:\n{ahab.image_info().draw()}")
    click.echo(f"Success. (AHAB: {get_printable_path(ahab_output_file_path)} created.)")

    post_export_files = ahab.post_export(config.get_output_dir("output"))
    if post_export_files:
        click.echo("Exporting AHAB fuses to the output directory.")
        print_files(post_export_files)


@ahab_group.command(name="parse", no_args_is_help=True)
@spsdk_family_option(families=AHABImage.get_supported_families())
@spsdk_output_option(directory=True)
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary AHAB image to parse.",
)
@click.option(
    "-k",
    "--dek",
    type=str,
    required=False,
    help=(
        "Data encryption key, if it's specified, the parse method tries decrypt all encrypted images. "
        "It could be specified as binary/HEX text file path or directly HEX string"
    ),
)
def ahab_parse_command(family: FamilyRevision, binary: str, dek: str, output: str) -> None:
    """Parse AHAB Image into YAML configuration and binary images."""
    ahab_parse(family, binary, dek, output)


def ahab_parse(family: FamilyRevision, binary: str, dek: str, output: str) -> None:
    """Parse AHAB Image into YAML configuration and binary images."""
    data = load_binary(binary)
    parsed_folder = output
    try:
        ahab_image = ahab_parse_image(family=family, binary=data)
    except SPSDKError as exc:
        click.echo(f"Failed. (AHAB: {binary} parsing failed.: {str(exc)})")
        return
    if not os.path.exists(parsed_folder):
        os.makedirs(parsed_folder, exist_ok=True)

    logger.info(f"Identified AHAB image for {ahab_image.chip_config.target_memory.label} target")
    logger.info(f"Parsed AHAB image memory map: {ahab_image.image_info().draw()}")
    if dek:
        for container in ahab_image.ahab_containers:
            if container.flag_srk_set != FlagsSrkSet.NXP:
                if container.signature_block and container.signature_block.blob:
                    container.signature_block.blob.dek = load_hex_string(
                        dek, container.signature_block.blob._size // 8
                    )
                    container.decrypt_data()
                else:
                    logger.info("Nothing to decrypt, the container doesn't contains BLOB")
    write_file(
        ahab_image.get_config_yaml(parsed_folder), os.path.join(parsed_folder, "parsed_config.yaml")
    )

    click.echo(f"Success. (AHAB: {binary} has been parsed and stored into {parsed_folder}.)")
    print_files(ahab_image.post_export(parsed_folder))


def ahab_parse_image(family: FamilyRevision, binary: bytes) -> AHABImage:
    """Parse one AHAB Image.

    :param family: Chip family.
    :param binary: Binary to parse
    :return: AHAB image if founded
    :raise SPSDKError: In case of AHAB is not found
    """
    for target_memory in AhabTargetMemory.labels():
        try:
            ahab_image = AHABImage.parse(binary, family=family, target_memory=target_memory)
            ahab_image.verify().validate()
        except SPSDKError as exc:
            logger.debug(
                f"AHAB parse: Attempt to parse image for {target_memory} target failed: {str(exc)}"
            )
            ahab_image = None
        else:
            break
    if not ahab_image:
        raise SPSDKError("Cannot find valid AHAB image")

    return ahab_image


@ahab_group.command(name="update-keyblob", no_args_is_help=True)
@spsdk_family_option(AHABImage.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary AHAB image to update.",
)
@click.option(
    "-k",
    "--keyblob",
    type=str,
    required=True,
    help=("Path to keyblob that will be inserted into AHAB Image"),
)
@click.option(
    "-i",
    "--container-id",
    type=INT(),
    required=True,
    help="""
    ID of the container where the keyblob will be replaced.
    """,
)
@click.option(
    "-m",
    "--mem-type",
    type=click.Choice(
        [mem_type.label for mem_type in BootableImage.get_supported_memory_types()],
        case_sensitive=False,
    ),
    required=False,
    help="Select memory type. Only applicable for bootable images "
    "(image containing FCB or XMCD segments). Do not use for raw AHAB image",
)
def ahab_update_keyblob_command(
    family: FamilyRevision, binary: str, keyblob: str, container_id: int, mem_type: str
) -> None:
    """Update keyblob in AHAB image container."""
    ahab_update_keyblob(family, binary, keyblob, container_id, mem_type)
    click.echo(f"Success. (AHAB: {binary} keyblob has been updated)")


@ahab_group.command(name="get-template", no_args_is_help=True)
@spsdk_family_option(families=AHABImage.get_supported_families())
@spsdk_output_option(force=True)
@click.option(
    "-s",
    "--sign",
    is_flag=True,
    default=False,
    help="Get template just for signing (encryption). To be used with ahab sign command.",
)
def ahab_get_template_command(family: FamilyRevision, output: str, sign: bool = False) -> None:
    """Create template of configuration in YAML format.

    The template file name is specified as argument of this command.
    """
    if sign:
        ahab_get_sign_template(family, output)
    else:
        ahab_get_template(family, output)


def ahab_get_template(family: FamilyRevision, output: str) -> None:
    """Create template of configuration in YAML format."""
    click.echo(f"Creating {get_printable_path(output)} template file.")
    write_file(AHABImage.get_config_template(family), output)


@ahab_group.command(name="verify", no_args_is_help=True)
@spsdk_family_option(families=AHABImage.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary AHAB image to parse.",
)
@click.option(
    "-p",
    "--problems",
    is_flag=True,
    default=False,
    help="Show just problems in image.",
)
@click.option(
    "-k",
    "--dek",
    type=str,
    required=False,
    help=(
        "Data encryption key, if it's specified, the parse method tries decrypt all encrypted images. "
        "It could be specified as binary/HEX text file path or directly HEX string"
    ),
)
def ahab_verify_command(family: FamilyRevision, binary: str, dek: str, problems: bool) -> None:
    """Verify AHAB Image."""
    ahab_verify(family=family, binary=binary, dek=dek, problems=problems)


def ahab_verify(family: FamilyRevision, binary: str, dek: str, problems: bool) -> None:
    """Verify AHAB Image."""
    data = load_binary(binary)
    verifiers: list[Verifier] = []
    valid_image = None
    preparsed = AHABImage.pre_parse_verify(data)
    if preparsed.has_errors:
        click.echo("The image bases has error, it doesn't passed pre-parse check:")
        print_verifier_to_console(preparsed, problems)
        raise SPSDKAppError("Pre-parsed check failed")

    for target_memory in AhabTargetMemory.labels():
        ahab_image = AHABImage.parse(data, family=family, target_memory=target_memory)
        ver = ahab_image.verify()
        verifiers.append(ver)
        if not ver.has_errors:
            valid_image = ahab_image

    if not valid_image:
        click.echo(
            "The binary has errors for all memory targets! All memory targets attempts will be printed.",
            err=True,
        )
        for x, verifier in enumerate(verifiers):
            click.echo("\n" + "=" * 120)
            click.echo(
                f"The verification attempt for target memory: {AhabTargetMemory.labels()[x].upper()}".center(
                    120
                )
            )
            click.echo("=" * 120 + "\n")
            print_verifier_to_console(verifier, problems)
        raise SPSDKAppError("Verify failed")

    for cnt in valid_image.ahab_containers:
        if cnt.flag_srk_set != FlagsSrkSet.NXP and cnt.signature_block and cnt.signature_block.blob:
            cnt.signature_block.blob.dek = (
                load_hex_string(dek, cnt.signature_block.blob._size // 8) if dek else None
            )

    if not problems:
        click.echo(valid_image.image_info().draw())

    print_verifier_to_console(valid_image.verify(), problems)


def ahab_get_sign_template(family: FamilyRevision, output: str) -> None:
    """Create template of configuration in YAML format."""
    click.echo(f"Creating {get_printable_path(output)} template file.")
    write_file(AHABImage.get_signing_template(family), output)


@ahab_group.group(name="certificate", cls=CommandsTreeGroup)
def ahab_certificate_group() -> None:  # pylint: disable=unused-argument
    """Group of sub-commands related to AHAB certificate blob."""


@ahab_certificate_group.command(name="get-template", no_args_is_help=True)
@spsdk_family_option(families=AhabCertificate.get_supported_families())
@spsdk_output_option(force=True)
def ahab_cert_block_get_template_command(family: FamilyRevision, output: str) -> None:
    """Create template of configuration in YAML format."""
    ahab_cert_block_get_template(family, output)


def ahab_cert_block_get_template(family: FamilyRevision, output: str) -> None:
    """Create template of configuration in YAML format."""
    click.echo(f"Creating {get_printable_path(output)} template file.")
    write_file(get_ahab_certificate_class(family).get_config_template(family), output)


@ahab_group.command(name="re-sign", no_args_is_help=True)
@spsdk_family_option(AHABImage.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary AHAB image to update.",
)
@click.option(
    "-k",
    "--pkey",
    type=str,
    required=True,
    help="Path to private key or signature provider configuration used for signing.",
)
@click.option(
    "-k1",
    "--pkey-1",
    type=str,
    help="Path to private key or signature provider configuration used for signing second signature if it's used.",
)
@click.option(
    "-i",
    "--container-id",
    type=INT(),
    required=True,
    help="""
    ID of the container where the keyblob will be replaced.
    """,
)
@click.option(
    "-m",
    "--mem-type",
    type=click.Choice(
        [mem_type.label for mem_type in BootableImage.get_supported_memory_types()],
        case_sensitive=False,
    ),
    required=False,
    help="Select memory type. Only applicable for bootable images "
    "(image containing FCB or XMCD segments). Do not use for raw AHAB image",
)
def ahab_re_sign_command(
    family: FamilyRevision,
    binary: str,
    pkey: str,
    pkey_1: Optional[str],
    container_id: int,
    mem_type: str,
) -> None:
    """Re-sign the container in AHAB image."""
    sign_provider_0 = get_signature_provider_from_config_str(pkey, pss_padding=True)
    sign_provider_1 = None
    if pkey_1:
        sign_provider_1 = get_signature_provider_from_config_str(pkey_1, pss_padding=True)
    ahab_re_sign(
        family=family,
        binary=binary,
        container_id=container_id,
        sign_provider_0=sign_provider_0,
        sign_provider_1=sign_provider_1,
        mem_type=mem_type,
    )
    click.echo(f"Success. (AHAB: {binary} signature has been updated)")


@ahab_group.command(name="sign", no_args_is_help=True)
@spsdk_config_option()
@spsdk_output_option(force=True)
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary AHAB image to sign and optionally encrypt.",
)
@click.option(
    "-m",
    "--mem-type",
    type=click.Choice(
        [mem_type.label for mem_type in BootableImage.get_supported_memory_types()],
        case_sensitive=False,
    ),
    required=False,
    help="Select memory type.",
)
@click.option(
    "-fs",
    "--fuse-scripts",
    type=click.Path(file_okay=False, dir_okay=True, resolve_path=True),
    required=False,
    help="Directory path where fuse script files will be exported.",
)
def ahab_sign_command(
    binary: str, output: str, mem_type: str, config: Config, fuse_scripts: Optional[str]
) -> None:
    """Sign all non-NXP AHAB containers and optionally encrypt them."""
    signed_image, bimg = ahab_sign_image(
        image_path=binary,
        config=config,
        mem_type=mem_type,
    )
    write_file(signed_image, output, "wb")
    click.echo(f"Signed image saved to {output}")

    # Handle post_export actions for fuse scripts if requested
    if fuse_scripts:
        # Create the directory if it doesn't exist
        os.makedirs(fuse_scripts, exist_ok=True)

        post_export_files = []
        for segment in bimg.segments:
            if hasattr(segment, "ahab") and hasattr(segment.ahab, "post_export"):
                segment_post_export_files = segment.ahab.post_export(fuse_scripts)
                if segment_post_export_files:
                    post_export_files.extend(segment_post_export_files)

        if post_export_files:
            click.echo(f"Exporting AHAB fuses to {get_printable_path(fuse_scripts)}.")
            print_files(post_export_files)
        else:
            click.echo("No fuse scripts were generated.")


@ahab_group.command(name="fix-signature-block-version", no_args_is_help=True)
@spsdk_family_option(AHABImage.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary AHAB image to fix.",
)
@spsdk_output_option(force=True)
def ahab_fix_signature_block_command(family: FamilyRevision, binary: str, output: str) -> None:
    """Fix signature block version in AHAB v2 signature blocks.

    imx-mkimage is known to create incorrect signature block versions, so this method
    corrects the version to match the expected AHAB v2 signature block version. (1)
    """
    try:
        fixed_data = ahab_fix_signature_block_version(
            family=family,
            binary=binary,
        )
        write_file(fixed_data, output, "wb")
        click.echo(f"Fixed image saved to {output}")
    except SPSDKError as exc:
        raise SPSDKAppError(f"Failed to fix AHAB headers: {str(exc)}") from exc


@ahab_certificate_group.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=AhabCertificate)
@spsdk_output_option(required=True)
def ahab_cert_block_export_command(config: Config, output: str) -> None:
    """Generate AHAB Certificate Blob from YAML/JSON configuration.

    The configuration template files could be generated by subcommand 'get-template'.
    """
    ahab_cert_block_export(config, output)


def ahab_cert_block_export(config: Config, output: str, plugin: Optional[str] = None) -> None:
    """Generate AHAB Certificate Blob from YAML/JSON configuration."""
    family = FamilyRevision.load_from_config(config)
    cert_block = get_ahab_certificate_class(family).load_from_config(config)
    # Sign the certificate blob
    cert_block.update_fields()
    cert_data = cert_block.export()

    write_file(cert_data, output, mode="wb")

    click.echo(f"Success. (AHAB Certificate Blob: {get_printable_path(output)} created.)")


@ahab_certificate_group.command(name="parse", no_args_is_help=True)
@spsdk_family_option(families=AhabCertificate.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary AHAB Certificate blob image to parse.",
)
@click.option(
    "-s",
    "--srk_set",
    type=click.Choice(["oem", "nxp"], case_sensitive=False),
    default="oem",
    help="SRK set that has been used for certificate.",
)
@spsdk_output_option(directory=True)
def ahab_cert_block_parse_command(
    family: FamilyRevision, binary: str, srk_set: str, output: str
) -> None:
    """Parse AHAB Certificate Blob."""
    ahab_cert_block_parse(family, binary, srk_set, output)


def ahab_cert_block_parse(family: FamilyRevision, binary: str, srk_set: str, output: str) -> None:
    """Parse AHAB Certificate Blob."""
    cert_block = get_ahab_certificate_class(family).parse(load_binary(binary), family)
    logger.info(str(cert_block))
    write_file(
        cert_block.get_config_yaml(output, index=0, srk_set=FlagsSrkSet.from_attr(srk_set)),
        os.path.join(output, "certificate_config.yaml"),
    )
    click.echo(f"Success. (AHAB Certificate Blob: {binary} has been parsed into {output}.)")


@ahab_certificate_group.command(name="verify", no_args_is_help=True)
@spsdk_family_option(families=AhabCertificate.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary AHAB certificate to verify.",
)
@click.option(
    "-p",
    "--problems",
    is_flag=True,
    default=False,
    help="Show just problems in image.",
)
def ahab_cert_block_verify_command(family: FamilyRevision, binary: str, problems: bool) -> None:
    """Verify AHAB Certificate."""
    ahab_cert_block_verify(family, binary, problems)


def ahab_cert_block_verify(family: FamilyRevision, binary: str, problems: bool) -> None:
    """Verify AHAB Image."""
    data = load_binary(binary)
    # preparsed = AhabCertificate.pre_parse_verify(data)
    # if preparsed.has_errors:
    #     click.echo("The image bases has error, it doesn't passed pre-parse check:")
    #     print_verifier_to_console(preparsed)
    #     raise SPSDKAppError("Pre-parsed check failed")

    ahab_certificate = get_ahab_certificate_class(family).parse(data, family)
    ver = ahab_certificate.verify()

    results = None
    if problems:
        results = [VerifierResult.WARNING, VerifierResult.ERROR]
    click.echo(ver.draw(results))

    click.echo("Summary table of verifier results:\n" + ver.get_summary_table())
