#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause
"""Nxpimage BCA group."""

import logging

import click

from spsdk.apps.utils.common_cli_options import (
    CommandsTreeGroup,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_output_option,
)
from spsdk.image.bca.bca import BCA
from spsdk.utils.config import Config
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import get_printable_path, load_binary, write_file

logger = logging.getLogger(__name__)


@click.group(name="bca", cls=CommandsTreeGroup)
def bca_group() -> None:  # pylint: disable=unused-argument
    """Group of sub-commands related to BCA block."""


@bca_group.command(name="get-template", no_args_is_help=True)
@spsdk_family_option(families=BCA.get_supported_families())
@spsdk_output_option(force=True)
def bca_get_template_command(output: str, family: FamilyRevision) -> None:
    """Create template of configuration in YAML format."""
    bca_get_template(output, family)


def bca_get_template(output: str, family: FamilyRevision) -> None:
    """Create template of configuration in YAML format."""
    click.echo(f"Creating {get_printable_path(output)} template file.")
    write_file(BCA.get_config_template(family), output)


@bca_group.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=BCA)
@spsdk_output_option()
def bca_export_command(config: Config, output: str) -> None:
    """Generate BCA from YAML/JSON configuration.

    The configuration template files could be generated by subcommand 'get-template'.
    """
    bca_export(config, output)


def bca_export(config: Config, output: str) -> None:
    """Export BCA from YAML/JSON configuration."""
    bca_image = BCA.load_from_config(config)
    bca_data = bca_image.export()
    write_file(bca_data, output, mode="wb")

    logger.info(f"Created BCA Image:\n{str(bca_image.registers.image_info())}")
    click.echo(f"Success. (BCA: {output} created.)")


@bca_group.command(name="parse", no_args_is_help=True)
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary BCA to parse.",
)
@spsdk_family_option(families=BCA.get_supported_families())
@spsdk_output_option()
def bca_parse_command(binary: str, family: FamilyRevision, output: str) -> None:
    """Parse BCA."""
    bca_parse(binary, family, output)


# pylint: disable=unused-argument  # preparation for future updates
def bca_parse(binary: str, family: FamilyRevision, output: str) -> None:
    """Parse BCA Image into YAML configuration."""
    bca_image = BCA.parse(load_binary(binary), family=family)
    logger.info(f"Parsed BCA image memory map: {bca_image.registers.image_info().draw()}")
    config = bca_image.get_config_yaml()
    write_file(config, output)
    click.echo(f"Success. (BCA: {binary} has been parsed and stored into {output}.)")
