#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause
"""Nxpimage Bootable Image group."""

import logging
import os
from typing import Optional

import click

from spsdk.apps.utils.common_cli_options import (
    CommandsTreeGroup,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_output_option,
)
from spsdk.apps.utils.utils import SPSDKAppError, print_files, print_verifier_to_console
from spsdk.exceptions import SPSDKError
from spsdk.image.bootable_image.bimg import BootableImage
from spsdk.image.fcb.fcb import FCB
from spsdk.image.mem_type import MemoryType
from spsdk.image.wic import replace_uboot
from spsdk.image.xmcd.xmcd import XMCD
from spsdk.utils.config import Config
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import get_printable_path, load_binary, write_file
from spsdk.utils.verifier import Verifier

logger = logging.getLogger(__name__)


@click.group(name="bootable-image", cls=CommandsTreeGroup)
def bootable_image_group() -> None:
    """Group of bootable image utilities."""


@bootable_image_group.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=BootableImage)
@spsdk_output_option()
def bootable_image_export_command(config: Config, output: str) -> None:
    """Export merged boot image blocks into one bootable image.

    The configuration template files could be generated by subcommand 'get-templates'.
    """
    bootable_image_export(config, output)


def bootable_image_export(config: Config, output: str) -> None:
    """Export merged boot image blocks into one bootable image."""
    bimg_image = BootableImage.load_from_config(config)
    bimg_image_info = bimg_image.image_info()

    write_file(bimg_image_info.export(), output, mode="wb")

    logger.info(f"Created Bootable Image:\n{str(bimg_image_info)}")
    logger.info(f"Created Bootable Image memory map:\n{bimg_image_info.draw()}")
    if bimg_image.bootable_header_only:
        click.echo("It has been created just only bootable header due missing application info.")
    click.echo(f"Success. (Bootable Image: {get_printable_path(output)} created) ")

    if config.get("post_export"):
        exported_files = bimg_image.post_export(config.get("post_export"))
        if exported_files:
            click.echo("Performing post export")
            print_files(exported_files)


@bootable_image_group.command(name="parse", no_args_is_help=True)
@spsdk_family_option(families=BootableImage.get_supported_families())
@click.option(
    "-m",
    "--mem-type",
    type=click.Choice(
        [mem_type.label for mem_type in BootableImage.get_supported_memory_types()],
        case_sensitive=False,
    ),
    required=False,
    help="Select the chip used memory type.",
)
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True),
    required=True,
    help="Path to binary Bootable image to parse.",
)
@spsdk_output_option(directory=True)
def bootable_image_parse_command(
    family: FamilyRevision, mem_type: Optional[str], binary: str, output: str
) -> None:
    """Parse Bootable Image into YAML configuration and binary images."""
    memory = None
    if mem_type:
        memory = MemoryType.from_label(mem_type)
    bootable_image_parse(family, memory, binary, output)


def bootable_image_parse(
    family: FamilyRevision, mem_type: Optional[MemoryType], binary: str, output: str
) -> None:
    """Parse Bootable Image into YAML configuration and binary images."""
    data = load_binary(binary)
    if mem_type:
        preparsed = BootableImage.pre_parse_verify(data, family, mem_type)
        if preparsed.has_errors:
            click.echo("The image bases has error, it doesn't passed pre-parse check:")
            print_verifier_to_console(preparsed)
            raise SPSDKAppError("Pre-parsed check failed")

    bimg_image = BootableImage.parse(data, family=family, mem_type=mem_type)
    bimg_image_info = bimg_image.image_info()
    logger.info(f"Parsed Bootable image memory map: {bimg_image_info.draw()}")
    write_file(
        bimg_image.get_config_yaml(output),
        os.path.join(output, f"bootable_image_{family.name}_{bimg_image.mem_type.label}.yaml"),
    )
    click.echo(
        f"Success. (Bootable Image: {binary} has been parsed and stored into {get_printable_path(output)} .)"
    )


@bootable_image_group.command(name="get-templates", no_args_is_help=True)
@spsdk_family_option(families=BootableImage.get_supported_families())
@spsdk_output_option(directory=True, force=True)
def bootable_image_get_templates_command(family: FamilyRevision, output: str) -> None:
    """Create template of configurations in YAML format from all memory types.

    The template files folder name is specified as argument of this command.
    """
    bootable_image_get_templates(family, output)


def bootable_image_get_templates(family: FamilyRevision, output: str) -> None:
    """Create template of configurations in YAML format from all memory types."""
    mem_types = BootableImage.get_supported_memory_types(family)
    for mem_type in mem_types:
        output_file = os.path.join(output, f"bootimg_{family.name}_{mem_type.label}.yaml")
        click.echo(f"Creating {get_printable_path(output_file)} template file.")
        write_file(BootableImage.get_config_template(family, mem_type), output_file)


@bootable_image_group.command(name="verify", no_args_is_help=True)
@spsdk_family_option(families=BootableImage.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary bootable image to verify.",
)
@click.option(
    "-m",
    "--mem-type",
    type=click.Choice(
        [mem_type.label for mem_type in BootableImage.get_supported_memory_types()],
        case_sensitive=False,
    ),
    required=True,
    help="Select the chip used memory type.",
)
@click.option(
    "-p",
    "--problems",
    is_flag=True,
    default=False,
    help="Show just problems in image.",
)
def bootable_image_verify_command(
    family: FamilyRevision, binary: str, problems: bool, mem_type: str
) -> None:
    """Verify Bootable Image."""
    bootable_image_verify(
        family=family, binary=binary, problems=problems, mem_type=MemoryType.from_label(mem_type)
    )


def bootable_image_verify(
    family: FamilyRevision, binary: str, problems: bool, mem_type: MemoryType
) -> None:
    """Verify Bootable Image."""
    data = load_binary(binary)

    preparsed = BootableImage.pre_parse_verify(data, family, mem_type)
    if preparsed.has_errors:
        click.echo("The image bases has error, it doesn't passed pre-parse check:")
        print_verifier_to_console(preparsed)
        raise SPSDKAppError("Pre-parsed check failed")

    bimg_images = []
    bimg_images = BootableImage._parse_all(data, family=family, mem_type=mem_type, no_errors=False)
    verifiers: list[Verifier] = []
    found_good_image = False
    for img in bimg_images:
        ver = img.verify()
        verifiers.append(ver)
        if not ver.has_errors:
            found_good_image = True

    if not found_good_image and len(bimg_images):
        click.echo(
            "The binary has errors for all memory targets! All memory targets attempts will be printed.",
            err=True,
        )
    for img, ver in zip(bimg_images, verifiers):
        # Print also the images with errors detected in case that neither one is good
        print_image = not ver.has_errors or (ver.has_errors and not found_good_image)
        if print_image:
            click.echo("\n" + "=" * 120)
            click.echo(f"The result for: {img.mem_type.label}".center(120))
            click.echo("=" * 120 + "\n")
            print_verifier_to_console(ver)

    if not found_good_image:
        raise SPSDKAppError("Verify failed")


@bootable_image_group.group(name="fcb", cls=CommandsTreeGroup)
def fcb() -> None:  # pylint: disable=unused-argument
    """FCB (Flash Configuration Block) utilities."""


@fcb.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=FCB)
@spsdk_output_option()
def fcb_export_command(config: Config, output: str) -> None:
    """Export FCB Image from YAML/JSON configuration.

    The configuration template files could be generated by subcommand 'get-templates'.
    """
    fcb_export(config, output)


def fcb_export(config: Config, output: str) -> None:
    """Export FCB Image from YAML/JSON configuration."""
    fcb_image = FCB.load_from_config(config)
    fcb_data = fcb_image.export()
    write_file(fcb_data, output, mode="wb")

    logger.info(f"Created FCB Image:\n{str(fcb_image.registers.image_info())}")
    logger.info(f"Created FCB Image memory map:\n{fcb_image.registers.image_info().draw()}")
    click.echo(f"Success. (FCB: {output} created.)")


def fcb_mem_type_callback(
    ctx: click.Context, param: click.Option, value: Optional[str]
) -> MemoryType:
    """Dynamically set memory type choices based on selected family."""
    family = FamilyRevision(ctx.params["family"], ctx.params.get("revision", "latest"))
    supported_mem_types = FCB.get_supported_memory_types(family)
    # If value is None and we need to provide a default, get the first supported memory type
    if value is None:
        return supported_mem_types[0]

    supported_labels = [mem_type.label for mem_type in supported_mem_types]
    if value not in supported_labels:
        supported_str = ", ".join(supported_labels)
        raise click.BadParameter(
            f"Memory type '{value}' is not supported for family {family.name}. "
            f"Supported types: {supported_str}"
        )

    return MemoryType.from_label(value)


@fcb.command(name="parse", no_args_is_help=True)
@spsdk_family_option(families=FCB.get_supported_families())
@click.option(
    "-m",
    "--mem-type",
    type=click.Choice(
        [mem_type.label for mem_type in FCB.get_supported_memory_types()], case_sensitive=False
    ),
    required=False,  # The callback will provide default if needed
    help="Select the chip used memory type.",
    callback=fcb_mem_type_callback,
)
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary FCB image to parse.",
)
@spsdk_output_option()
def fcb_parse_command(
    family: FamilyRevision, mem_type: MemoryType, binary: str, output: str
) -> None:
    """Parse FCB Image into YAML configuration."""
    fcb_parse(family, mem_type, binary, output)


def fcb_parse(family: FamilyRevision, mem_type: MemoryType, binary: str, output: str) -> None:
    """Parse FCB Image into YAML configuration."""
    fcb_image = FCB.parse(load_binary(binary), family=family, mem_type=mem_type)
    logger.info(f"Parsed FCB image memory map: {fcb_image.registers.image_info().draw()}")
    write_file(fcb_image.get_config_yaml(), output)
    click.echo(f"Success. (FCB: {binary} has been parsed and stored into {output} .)")


@fcb.command(name="get-templates", no_args_is_help=True)
@spsdk_family_option(families=FCB.get_supported_families())
@spsdk_output_option(directory=True, force=True)
def fcb_get_templates_command(family: FamilyRevision, output: str) -> None:
    """Create template of configurations in YAML format for all memory types.

    The template files folder name is specified as argument of this command.
    """
    fcb_get_templates(family, output)


def fcb_get_templates(family: FamilyRevision, output_folder: str) -> None:
    """Create template of configurations in YAML format for all memory types."""
    mem_types = FCB.get_supported_memory_types(family)
    for mem_type in mem_types:
        output = os.path.join(output_folder, f"fcb_{family.name}_{mem_type.label}.yaml")
        click.echo(f"Creating {output} template file.")
        write_file(FCB.get_config_template(family, mem_type), output)


@bootable_image_group.group(name="xmcd", cls=CommandsTreeGroup)
def xmcd() -> None:  # pylint: disable=unused-argument
    """XMCD (External Memory Configuration Data) utilities."""


@xmcd.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=XMCD)
@spsdk_output_option()
def xmcd_export_command(config: Config, output: str) -> None:
    """Export XMCD Image from YAML/JSON configuration.

    The configuration template files could be generated by subcommand 'get-templates'.
    """
    xmcd_export(config, output)


def xmcd_export(config: Config, output: str) -> None:
    """Export XMCD Image from YAML/JSON configuration."""
    xmcd_image = XMCD.load_from_config(config)
    try:
        xmcd_image.verify().validate()
    except SPSDKError as exc:
        logger.info(f"XMCD validation failed:\n {exc}")
        raise SPSDKAppError("Loading of XMCD from configuration failed.") from exc
    xmcd_data = xmcd_image.export()
    write_file(xmcd_data, output, mode="wb")

    logger.info(f"Created XMCD :\n{str(xmcd_image.registers.image_info())}")
    logger.info(f"Created XMCD memory map:\n{xmcd_image.registers.image_info().draw()}")
    click.echo(f"Success. (XMCD: {output} created.)")


@xmcd.command(name="parse", no_args_is_help=True)
@spsdk_family_option(families=XMCD.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary XMCD image to parse.",
)
@spsdk_output_option()
def xmcd_parse_command(family: FamilyRevision, binary: str, output: str) -> None:
    """Parse XMCD Image into YAML configuration."""
    xmcd_parse(family, binary, output)


def xmcd_parse(family: FamilyRevision, binary: str, output: str) -> None:
    """Parse XMCD Image into YAML configuration."""
    xmcd_image = XMCD.parse(load_binary(binary), family=family)
    ver = xmcd_image.verify()
    if ver.has_errors:
        logger.info(f"XMCD validation failed:\n {ver.draw()}")
        raise SPSDKAppError("Parsing of XMCD binary failed.")
    logger.info(f"Parsed XMCD: {xmcd_image.registers.image_info().draw()}")
    write_file(xmcd_image.get_config_yaml(), output)
    click.echo(f"Success. (XMCD: {binary} has been parsed and stored into {output} .)")


@xmcd.command(name="get-templates", no_args_is_help=True)
@spsdk_family_option(families=XMCD.get_supported_families())
@spsdk_output_option(directory=True, force=True)
def xmcd_get_templates_command(family: FamilyRevision, output: str) -> None:
    """Create template of configurations in YAML format for all memory types.

    The template files folder name is specified as argument of this command.
    """
    xmcd_get_templates(family, output)


def xmcd_get_templates(family: FamilyRevision, output: str) -> None:
    """Create template of configurations in YAML format for all memory types."""
    mem_types = XMCD.get_supported_memory_types(family)
    for mem_type in mem_types:
        config_types = XMCD.get_supported_configuration_types(family, mem_type)
        for config_type in config_types:
            output_file = os.path.join(
                output, f"xmcd_{family.name}_{mem_type.label}_{config_type.label}.yaml"
            )
            write_file(
                XMCD.get_config_template(
                    family,
                    mem_type,
                    config_type,
                ),
                output_file,
            )
            click.echo(f"The template file has been created: {get_printable_path(output_file)}.")


@xmcd.command(name="verify", no_args_is_help=True)
@spsdk_family_option(families=XMCD.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary XMCD image to parse.",
)
@click.option(
    "-p",
    "--problems",
    is_flag=True,
    default=False,
    help="Show just problems in image.",
)
def xmcd_verify_command(family: FamilyRevision, binary: str, problems: bool) -> None:
    """Verify XMCD Image."""
    xmcd_verify(family=family, binary=binary, problems=problems)


def xmcd_verify(family: FamilyRevision, binary: str, problems: bool) -> None:
    """Verify XMCD Image."""
    preparsed = XMCD.pre_parse_verify(load_binary(binary), family)
    if preparsed.has_errors:
        click.echo("The image bases has error, it doesn't passed pre-parse check:")
        print_verifier_to_console(preparsed, problems)
        raise SPSDKAppError("Pre-parsed check failed")
    xmcd_image = XMCD.parse(load_binary(binary), family=family)
    verifier = xmcd_image.verify()
    print_verifier_to_console(verifier, problems)
    if problems:
        raise SPSDKAppError("Pre-parsed check failed")
    click.echo(xmcd_image.registers.image_info().draw())


@xmcd.command(name="crc-fuses-script", no_args_is_help=True)
@spsdk_family_option(families=XMCD.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary XMCD image.",
)
@spsdk_output_option(required=True)
def xmcd_crc_fuses_script_command(family: FamilyRevision, binary: str, output: str) -> None:
    """Generate XMCD CRC fuses script.

    Programming the CRC checksum to the fuse enables the integrity check of XMCD block.
    """
    xmcd_block = XMCD.parse(load_binary(binary), family=family)
    fuses_script = xmcd_block.create_crc_hash_fuses_script()
    write_file(fuses_script, output)
    click.echo(f"Success. (Created fuses script: {output} )")


@bootable_image_group.group(name="wic", cls=CommandsTreeGroup)
def wic() -> None:  # pylint: disable=unused-argument
    """WIC (Whole Image Creator) Yocto Linux image format."""


@wic.command(name="update-uboot", no_args_is_help=True)
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary WIC image.",
)
@click.option(
    "-u",
    "--uboot",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary U-Boot image.",
)
def replace_uboot_command(binary: str, uboot: str) -> None:
    """Replace U-Boot binary in WIC file."""
    address = replace_uboot(binary, uboot)
    click.echo(f"Replaced u-boot at address {hex(address)}")
