#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause
"""Nxpimage Certification Block group."""

import logging
import os

import click

from spsdk.apps.utils.common_cli_options import (
    CommandsTreeGroup,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_output_option,
)
from spsdk.image.cert_block.cert_blocks import CertBlock, CertBlockVx, IskCertificateLite
from spsdk.utils.config import Config
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import get_printable_path, load_binary, write_file

logger = logging.getLogger(__name__)


@click.group(name="cert-block", cls=CommandsTreeGroup)
def cert_block_group() -> None:  # pylint: disable=unused-argument
    """Group of sub-commands related to certification block."""


@cert_block_group.command(name="get-template", no_args_is_help=True)
@spsdk_family_option(families=CertBlock.get_all_supported_families())
@spsdk_output_option(force=True)
def cert_block_get_template_command(output: str, family: FamilyRevision) -> None:
    """Create template of configuration in YAML format."""
    cert_block_get_template(output, family)


def cert_block_get_template(output: str, family: FamilyRevision) -> None:
    """Create template of configuration in YAML format."""
    click.echo(f"Creating {get_printable_path(output)} template file.")
    cert_block_class = CertBlock.get_cert_block_class(family)
    write_file(cert_block_class.get_config_template(family), output)


@cert_block_group.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=CertBlock)
def cert_block_export_command(config: Config) -> None:
    """Generate Certificate Block from YAML/JSON configuration.

    The configuration template files could be generated by subcommand 'get-template'.
    """
    cert_block_export(config)


def cert_block_export(config: Config) -> None:
    """Generate Certificate Block from YAML/JSON configuration."""
    family = FamilyRevision.load_from_config(config)
    cert_block_class = CertBlock.get_cert_block_class(family)
    cert_block = cert_block_class.load_from_config(config)
    cert_data = cert_block.export()
    cert_block_output_file_path = config.get_output_file_name("containerOutputFile")

    write_file(cert_data, cert_block_output_file_path, mode="wb")

    if cert_block.rkth:
        click.echo(f"RKTH: {cert_block.rkth.hex()}")
    if hasattr(cert_block, "cert_hash"):
        assert isinstance(cert_block, CertBlockVx), "Wrong instance of cert block"
        click.echo(f"ISK Certificate hash [0:127]: {cert_block.cert_hash.hex()}\n")
        otp_script_path = os.path.join(
            os.path.dirname(cert_block_output_file_path), "otp_script.bcf"
        )
        write_file(cert_block.get_otp_script(), otp_script_path)
        click.echo(f"OTP script written to: {get_printable_path(otp_script_path)}")
    click.echo(
        f"Success. (Certificate Block: {get_printable_path(cert_block_output_file_path)} created.)"
    )


@cert_block_group.command(name="get-isk-tbs", no_args_is_help=True)
@spsdk_family_option(families=CertBlockVx.get_supported_families())
@click.option(
    "-p",
    "--public-key",
    required=True,
    type=click.Path(exists=True, dir_okay=False, resolve_path=True),
    help="Path to file with the ISK public key",
)
@spsdk_output_option()
# pylint: disable=unused-argument  # we just filter applicable chip families, the value itself is not used
def cert_block_get_isk_tbs_data(family: FamilyRevision, output: str, public_key: str) -> None:
    """Generate To-Be-Signed data for ISK Certificate created by NXP's EdgeLock2GO service."""
    puk = load_binary(path=public_key)
    isk = IskCertificateLite(pub_key=puk, constraints=0)
    tbs_data = isk.get_tbs_data()
    write_file(data=tbs_data, path=output, mode="wb")
    click.echo(f"Success. (ISK TBS data: {output} created.)")


@cert_block_group.command(name="parse", no_args_is_help=True)
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary Certificate Block image to parse.",
)
@spsdk_family_option(families=CertBlock.get_all_supported_families())
@spsdk_output_option(directory=True)
def cert_block_parse_command(binary: str, family: FamilyRevision, output: str) -> None:
    """Parse Certificate Block.

    RoTKTH is printed out in verbose mode.
    """
    cert_block_parse(binary, family, output)


# pylint: disable=unused-argument  # preparation for future updates
def cert_block_parse(binary: str, family: FamilyRevision, output: str) -> None:
    """Parse Certificate Block."""
    cert_block = CertBlock.get_cert_block_class(family).parse(load_binary(binary), family=family)
    logger.info(str(cert_block))
    write_file(cert_block.get_config_yaml(output), os.path.join(output, "cert_block_config.yaml"))
    click.echo(f"RKTH: {cert_block.rkth.hex()}")
    click.echo(f"Success. (Certificate Block: {binary} has been parsed into {output}.)")
