#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause
"""Nxpimage IEE group."""

import logging

import click

from spsdk.apps.utils.common_cli_options import (
    CommandsTreeGroup,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_output_option,
)
from spsdk.apps.utils.utils import print_files
from spsdk.image.iee.iee import Iee
from spsdk.utils.config import Config
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import write_file

logger = logging.getLogger(__name__)


@click.group(name="iee", cls=CommandsTreeGroup)
def iee_group() -> None:  # pylint: disable=unused-argument
    """Group of sub-commands related to IEE."""


@iee_group.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=Iee)
def iee_export_command(config: Config) -> None:
    """Generate IEE Images from YAML/JSON configuration.

    The configuration template files could be generated by subcommand 'get-template'.
    """
    iee_export(config)


def iee_export(config: Config) -> None:
    """Generate IEE Images from YAML/JSON configuration."""
    iee = Iee.load_from_config(config)
    click.echo("Exporting IEE files")
    print_files(iee.post_export(config.get_output_file_name("output_folder")))


@iee_group.command(name="get-template", no_args_is_help=True)
@spsdk_family_option(families=Iee.get_supported_families())
@spsdk_output_option(force=True)
def iee_get_template(family: FamilyRevision, output: str) -> None:
    """Create template of configuration in YAML format.

    The template file name is specified as argument of this command.
    """
    click.echo(f"Creating {output} template file.")
    write_file(Iee.get_config_template(family), output)
