#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause
"""Nxpimage Masterboot Image group."""

import datetime
import logging
import os

import click

from spsdk.apps.utils.common_cli_options import (
    CommandsTreeGroup,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_output_option,
)
from spsdk.image.mbi.mbi import MasterBootImage, mbi_generate_config_templates
from spsdk.utils.config import Config
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import get_printable_path, load_binary, write_file
from spsdk.utils.schema_validator import CommentedConfig

logger = logging.getLogger(__name__)


@click.group(name="mbi", cls=CommandsTreeGroup)
def mbi_group() -> None:
    """Group of sub-commands related to Master Boot Images."""


@mbi_group.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=MasterBootImage)
def mbi_export_command(config: Config) -> None:
    """Generate Master Boot Image from YAML/JSON configuration.

    The configuration template files could be generated by subcommand 'get-templates'.
    """
    mbi_export(config)


def mbi_export(config: Config) -> None:
    """Generate Master Boot Image from YAML/JSON configuration.

    :param config: Path to the YAML/JSON configuration
    """
    mbi_obj = MasterBootImage.load_from_config(config)
    mbi_data = mbi_obj.export_image()
    if mbi_obj.rkth:
        click.echo(f"RKTH: {mbi_obj.rkth.hex()}")
    mbi_output_file_path = config.get_output_file_name("masterBootOutputFile")
    logger.info(mbi_data.draw())

    write_file(mbi_data.export(), mbi_output_file_path, mode="wb")

    click.echo(f"Success. (Master Boot Image: {get_printable_path(mbi_output_file_path)} created.)")


@mbi_group.command(name="parse", no_args_is_help=True)
@spsdk_family_option(families=MasterBootImage.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary MBI image to parse.",
)
@click.option(
    "-k",
    "--dek",
    type=str,
    required=False,
    help=(
        "Data encryption key, if it's specified, the parse method tries decrypt all encrypted images. "
        "It could be specified as binary/HEX text file path or directly HEX string"
    ),
)
@spsdk_output_option(directory=True)
def mbi_parse_command(family: FamilyRevision, binary: str, dek: str, output: str) -> None:
    """Parse MBI Image into YAML configuration and binary images."""
    mbi_parse(family, binary, dek, output)


def mbi_parse(family: FamilyRevision, binary: str, dek: str, output: str) -> None:
    """Parse MBI Image into YAML configuration and binary images."""
    mbi = MasterBootImage.parse(family=family, data=load_binary(binary), dek=dek)

    if not mbi:
        click.echo(f"Failed. (MBI: {binary} parsing failed.)")
        return

    cfg = mbi.get_config(data_path=output)
    yaml_data = CommentedConfig(
        main_title=(
            f"Master Boot Image ({mbi.__class__.__name__}) recreated configuration from :"
            f"{datetime.datetime.now().strftime('%d/%m/%Y %H:%M:%S')}."
        ),
        schemas=mbi.get_validation_schemas(family),
    ).get_config(cfg)

    write_file(yaml_data, os.path.join(output, "mbi_config.yaml"))

    click.echo(f"Success. (MBI: {binary} has been parsed and stored into {output} )")


@mbi_group.command(name="get-templates", no_args_is_help=True)
@spsdk_family_option(families=MasterBootImage.get_supported_families())
@spsdk_output_option(directory=True, force=True)
def mbi_get_templates_command(family: FamilyRevision, output: str) -> None:
    """Create template of MBI configurations in YAML format."""
    mbi_get_templates(family, output)


def mbi_get_templates(family: FamilyRevision, output: str) -> None:
    """Create template of MBI configurations in YAML format."""
    templates = mbi_generate_config_templates(family)
    for file_name, template in templates.items():
        full_file_name = os.path.join(output, file_name + ".yaml")
        click.echo(f"Creating {get_printable_path(full_file_name)} template file.")
        write_file(template, full_file_name)
