#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause
"""Nxpimage OTFAD group."""

import logging
import os
from typing import Optional

import click

from spsdk.apps.utils.common_cli_options import (
    CommandsTreeGroup,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_output_option,
)
from spsdk.apps.utils.utils import INT, print_files, store_key
from spsdk.image.keystore import KeyStore
from spsdk.image.otfad.otfad import Otfad
from spsdk.utils.config import Config
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import get_printable_path, load_hex_string, write_file

logger = logging.getLogger(__name__)


@click.group(name="otfad", cls=CommandsTreeGroup)
def otfad_group() -> None:
    """Group of sub-commands related to OTFAD."""


@otfad_group.command(name="export", no_args_is_help=True)
@click.option(
    "-a",
    "--alignment",
    default="512",
    type=INT(),
    help="Alignment of key blob data blocks. Default value is 512 B if not specified.",
    required=False,
)
@click.option(
    "-i",
    "--index",
    type=INT(),
    help=(
        "OTFAD peripheral index - This is needed to generate proper "
        "indexes of fuses in optional BLHOST script. If not specified, BLHOST fuse script won't be generated"
    ),
    required=False,
)
@spsdk_config_option(klass=Otfad)
def otfad_export_command(alignment: int, config: Config, index: Optional[int] = None) -> None:
    """Generate OTFAD Images from YAML/JSON configuration.

    The configuration template files could be generated by subcommand 'get-template'.
    """
    otfad_export(alignment, config, index)


def otfad_export(alignment: int, config: Config, index: Optional[int] = None) -> None:
    """Generate OTFAD Images from YAML/JSON configuration."""
    # Add data alignment and index to the configuration
    config["data_alignment"] = alignment
    if index:
        config["index"] = index

    otfad = Otfad.load_from_config(config)
    click.echo("Exporting OTFAD files")
    print_files(otfad.post_export(config.get_output_file_name("output_folder")))


@otfad_group.command(name="get-kek", no_args_is_help=False)
@click.option(
    "-m",
    "--otp-master-key",
    type=str,
    help="OTP MASTER KEY in hexadecimal format or file name to binary or text file with key.",
)
@click.option(
    "-k",
    "--otfad-key",
    type=str,
    help="OTFAD KEY (OTFAD KEK seed) in hexadecimal format or file name to binary or text file with key.",
)
@spsdk_family_option(
    families=Otfad.get_supported_families(),
    help=(
        "Optional family, if specified, the tool generates the BLHOST scripts to load key fuses."
        " To use this feature, the '-o' options has to be also defined!"
    ),
)
@spsdk_output_option(
    required=False,
    directory=True,
    help="Optional result output folder (otfad_kek.bin/txt, optionally BLHOST scripts to load keys into Fuses)",
)
def otfad_get_kek_command(
    otp_master_key: str, otfad_key: str, family: FamilyRevision, output: str
) -> None:
    """Compute OTFAD KEK value and optionally store it into folder in various formats.

    The computed OTFAD KEK is shown in hexadecimal text, if the 'output-folder' is specified,
    it is stored in folder also in binary format.
    """
    otfad_get_kek(otp_master_key, otfad_key, family, output)


def otfad_get_kek(
    otp_master_key: str, otfad_key: str, family: FamilyRevision, output_folder: str
) -> None:
    """Compute OTFAD KEK value and optionally store it into folder in various formats."""
    omk = load_hex_string(otp_master_key, KeyStore.OTP_MASTER_KEY_SIZE)
    ok = load_hex_string(otfad_key, KeyStore.OTFAD_KEY_SIZE)  # pylint:disable=invalid-name

    otfad_kek = KeyStore.derive_otfad_kek_key(omk, ok)

    click.echo(f"OTP MASTER KEY: {omk.hex()}")
    click.echo(f"OTFAD KEY:      {ok.hex()}")
    click.echo(f"OTFAD KEK:      {otfad_kek.hex()}")

    blhost_script = None
    if family in Otfad.get_supported_families():
        blhost_script = Otfad.get_blhost_script_otp_keys(
            family, otp_master_key=omk, otfad_kek_seed=ok
        )
        if not output_folder:
            click.echo(f"OTFAD BLHOST load fuses script:\n{blhost_script}")

    if output_folder:
        store_key(os.path.join(output_folder, "otp_master_key"), omk)
        store_key(os.path.join(output_folder, "otfad_key"), ok)
        store_key(os.path.join(output_folder, "otfad_kek"), otfad_kek)
        if blhost_script:
            write_file(blhost_script, os.path.join(output_folder, "otfad_otp_fuses.bcf"))
        click.echo(f"Result files has been stored into: {output_folder}")


@otfad_group.command(name="get-template", no_args_is_help=True)
@spsdk_family_option(families=Otfad.get_supported_families())
@spsdk_output_option(force=True)
def otfad_get_template_command(family: FamilyRevision, output: str) -> None:
    """Create template of configuration in YAML format.

    The template file name is specified as argument of this command.
    """
    otfad_get_template(family, output)


def otfad_get_template(family: FamilyRevision, output: str) -> None:
    """Create template of configuration in YAML format."""
    click.echo(f"Creating {get_printable_path(output)} template file.")
    write_file(Otfad.get_config_template(family), output)
