#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause
"""Nxpimage Signed Message group."""

import logging
import os
from typing import Optional

import click

from spsdk.apps.utils.common_cli_options import (
    CommandsTreeGroup,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_output_option,
)
from spsdk.apps.utils.utils import SPSDKAppError, print_files, print_verifier_to_console
from spsdk.exceptions import SPSDKError
from spsdk.image.ahab.ahab_data import FlagsSrkSet
from spsdk.image.ahab.signed_msg import MessageCommands, SignedMessage, SignedMessageContainer
from spsdk.image.ahab.tlv import TLV, TLVTypes
from spsdk.utils.config import Config
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import get_printable_path, load_binary, load_hex_string, write_file

logger = logging.getLogger(__name__)


@click.group(name="signed-msg", cls=CommandsTreeGroup)
def signed_msg_group() -> None:  # pylint: disable=unused-argument
    """Group of sub-commands related to Signed messages."""


@signed_msg_group.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=SignedMessage)
@click.option(
    "-w",
    "--assets",
    type=click.Path(file_okay=False, dir_okay=True, resolve_path=True),
    help="Directory path to export additional assets (derived keys, certificates, etc.). "
    "If specified, the post_export method will be called to generate supplementary files.",
)
def signed_msg_export(config: Config, assets: Optional[str]) -> None:
    """Generate Signed message Image from YAML/JSON configuration.

    The configuration template files could be generated by subcommand 'get-template'.
    """
    signed_message = SignedMessage.load_from_config(config=config)
    signed_message.update_fields()
    signed_msg_data = signed_message.export()

    signed_msg_output_file_path = config.get_output_file_name("output")
    write_file(signed_msg_data, signed_msg_output_file_path, mode="wb")

    logger.info(f"Created Signed message Image:\n{str(signed_message.image_info())}")
    logger.info(f"Created Signed message Image memory map:\n{signed_message.image_info().draw()}")

    if not (
        signed_message.signed_msg_container
        and signed_message.signed_msg_container.signature_block
        and signed_message.signed_msg_container.signature_block.srk_assets
    ):
        raise SPSDKAppError("SRK assets not found in the signed message")

    logger.info(
        f"SRK hash:{signed_message.signed_msg_container.signature_block.srk_assets.compute_srk_hash().hex()}"
    )

    # Handle assets export if requested
    if assets:
        exported = signed_message.post_export(assets)
        if exported:
            click.echo("Performing post export")
            print_files(exported)

    click.echo(
        f"Success. (Signed message: {get_printable_path(signed_msg_output_file_path)} created.)"
    )


@signed_msg_group.command(name="parse", no_args_is_help=True)
@spsdk_family_option(families=SignedMessage.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary Signed message image to parse.",
)
@spsdk_output_option(directory=True)
def signed_msg_parse(family: FamilyRevision, binary: str, output: str) -> None:
    """Parse Signed message Image into YAML configuration and binary images."""
    data = load_binary(binary)
    preparsed = SignedMessage.pre_parse_verify(data)
    if preparsed.has_errors:
        click.echo("The signed message image bases has error, it doesn't passed pre-parse check:")
        click.echo(preparsed.draw())
        raise SPSDKAppError("Pre-parsed check failed")
    if not os.path.exists(output):
        os.makedirs(output, exist_ok=True)
    try:
        signed_message = SignedMessage.parse(data, family=family)
    except SPSDKError as exc:
        click.echo(f"Signed message parsing failed: {binary} ,({str(exc)})")
        return

    logger.info(f"Parsed Signed message image memory map: {signed_message.image_info().draw()}")

    write_file(signed_message.get_config_yaml(output), os.path.join(output, "parsed_config.yaml"))
    click.echo(f"Success. (Signed message: {binary} has been parsed and stored into {output}.)")
    for ix in range(signed_message.srk_count):
        srk_hash = signed_message.get_srk_hash(ix)
        assert isinstance(signed_message.signed_msg_container, SignedMessageContainer)
        file_name = os.path.join(
            output, f"{signed_message.signed_msg_container.flag_srk_set.label}_srk{ix}_hash"
        )
        write_file(srk_hash.hex().upper(), f"{file_name}.txt")
        click.echo(f"Generated SRK hash file ({file_name}.txt)")


@signed_msg_group.command(name="get-template", no_args_is_help=True)
@spsdk_family_option(families=SignedMessage.get_supported_families())
@click.option(
    "-m",
    "--message",
    required=False,
    type=click.Choice(MessageCommands.labels(), case_sensitive=False),
    help="Select only one signed message to generate specific template if needed",
)
@spsdk_output_option(force=True)
def signed_msg_get_template(family: FamilyRevision, message: Optional[str], output: str) -> None:
    """Create template of configuration in YAML format.

    The template file name is specified as argument of this command.
    """
    click.echo(f"Creating {get_printable_path(output)} template file.")
    write_file(
        SignedMessage.get_config_template(
            family, MessageCommands.from_attr(message) if message else None
        ),
        output,
    )


@signed_msg_group.command(name="verify", no_args_is_help=True)
@spsdk_family_option(families=SignedMessage.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary AHAB image to parse.",
)
@click.option(
    "-p",
    "--problems",
    is_flag=True,
    default=False,
    help="Show just problems in image.",
)
@click.option(
    "-k",
    "--dek",
    type=str,
    required=False,
    help=(
        "Data encryption key, if it's specified, the parse method tries decrypt all encrypted images. "
        "It could be specified as binary/HEX text file path or directly HEX string"
    ),
)
def signed_msg_verify_command(
    family: FamilyRevision, binary: str, dek: str, problems: bool
) -> None:
    """Verify AHAB Image."""
    signed_msg_verify(family=family, binary=binary, dek=dek, problems=problems)


def signed_msg_verify(family: FamilyRevision, binary: str, dek: str, problems: bool) -> None:
    """Verify AHAB Image."""
    data = load_binary(binary)
    preparsed = SignedMessage.pre_parse_verify(data)
    if preparsed.has_errors:
        click.echo("The signed message image bases has error, it doesn't passed pre-parse check:")
        print_verifier_to_console(preparsed)
        raise SPSDKAppError("Pre-parsed check failed")

    signed_msg_image = SignedMessage.parse(data, family=family)
    ver = signed_msg_image.verify()

    if ver.has_errors:
        click.echo("The binary has errors!", err=True)
        print_verifier_to_console(ver)
        raise SPSDKAppError("Verify failed")

    assert isinstance(signed_msg_image.signed_msg_container, SignedMessageContainer)
    if (
        signed_msg_image.signed_msg_container.flag_srk_set != FlagsSrkSet.NXP
        and signed_msg_image.signed_msg_container.signature_block
        and signed_msg_image.signed_msg_container.signature_block.blob
    ):
        signed_msg_image.signed_msg_container.signature_block.blob.dek = (
            load_hex_string(
                dek, signed_msg_image.signed_msg_container.signature_block.blob._size // 8
            )
            if dek
            else None
        )

    if not problems:
        click.echo(signed_msg_image.image_info().draw())

    print_verifier_to_console(signed_msg_image.verify())


# TLV subcommand group
@signed_msg_group.group(name="tlv", cls=CommandsTreeGroup)
def tlv_group() -> None:
    """Group of sub-commands related to TLV (Key Import) operations."""


@tlv_group.command(name="export", no_args_is_help=True)
@spsdk_config_option(klass=TLV)
def tlv_export(config: Config) -> None:
    """Generate TLV blob from configuration.

    The configuration template files could be generated by subcommand 'get-template'.
    This command creates a TLV blob for operations like Key Import.
    """
    tlv = TLV.load_from_config(config=config)
    tlv_data = tlv.export()

    tlv_output_file_path = config.get_output_file_name("output")
    write_file(tlv_data, tlv_output_file_path, mode="wb")

    logger.info(f"TLV blob size: {len(tlv_data)} bytes")
    logger.info(f"TLV details:\n{str(tlv)}")

    click.echo(f"Success. (TLV blob: {get_printable_path(tlv_output_file_path)} created.)")


@tlv_group.command(name="get-template", no_args_is_help=True)
@spsdk_family_option(families=TLV.get_supported_families())
@click.option(
    "-t",
    "--tlv-type",
    required=False,
    type=click.Choice(TLVTypes.labels(), case_sensitive=False),
    default="KEY_IMPORT",
    help="Type of TLV to parse",
)
@spsdk_output_option(force=True)
def tlv_get_template(family: FamilyRevision, tlv_type: str, output: str) -> None:
    """Create template of TLV configuration in YAML format.

    The template file name is specified as argument of this command.
    """
    tlv_class = TLV.get_tlv_class(tlv_type)
    template = tlv_class.get_config_template(family)
    write_file(template, output)
    click.echo(f"Success. (TLV template: {get_printable_path(output)} created.)")


@tlv_group.command(name="parse", no_args_is_help=True)
@spsdk_family_option(families=TLV.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary TLV blob to parse.",
)
@click.option(
    "-t",
    "--tlv-type",
    required=False,
    type=click.Choice(TLVTypes.labels(), case_sensitive=False),
    default="KEY_IMPORT",
    help="Type of TLV to parse",
)
@spsdk_output_option(directory=True)
def tlv_parse(family: FamilyRevision, binary: str, tlv_type: str, output: str) -> None:
    """Parse TLV blob into YAML configuration.

    This command parses a TLV blob and generates a configuration file.
    """
    data = load_binary(binary)

    if not os.path.exists(output):
        os.makedirs(output, exist_ok=True)

    try:
        # Get the specific TLV class based on type
        tlv_class = TLV.get_tlv_class(tlv_type)

        # Parse the TLV data using the class method
        tlv_instance = tlv_class.parse(data)
        logger.info(f"TLV blob size: {len(data)} bytes")
        logger.info(f"TLV details:\n{str(tlv_instance)}")

        write_file(tlv_instance.get_config_yaml(output), os.path.join(output, "parsed_config.yaml"))

        click.echo(f"Success. (TLV blob: {binary} has been parsed and stored into {output}.)")

    except SPSDKError as exc:
        click.echo(f"TLV blob parsing failed: {binary} ,({str(exc)})")
        raise SPSDKAppError(f"TLV parsing failed: {str(exc)}") from exc


@tlv_group.command(name="verify", no_args_is_help=True)
@spsdk_family_option(families=TLV.get_supported_families())
@click.option(
    "-b",
    "--binary",
    type=click.Path(exists=True, readable=True, resolve_path=True),
    required=True,
    help="Path to binary TLV blob to verify.",
)
@click.option(
    "-t",
    "--tlv-type",
    required=False,
    type=click.Choice(TLVTypes.labels(), case_sensitive=False),
    default="KEY_IMPORT",
    help="Type of TLV to verify",
)
@click.option(
    "-p",
    "--problems",
    is_flag=True,
    default=False,
    help="Show just problems in TLV blob.",
)
def tlv_verify(family: FamilyRevision, binary: str, tlv_type: str, problems: bool) -> None:
    """Verify TLV blob structure and content.

    This command verifies the structure and content of a TLV blob (Key Import payload).
    """
    data = load_binary(binary)

    try:
        # Get the specific TLV class based on type
        tlv_class = TLV.get_tlv_class(tlv_type)

        # Parse the TLV data using the class method
        tlv_instance = tlv_class.parse(data)

        # Verify the TLV blob
        verifier = tlv_instance.verify()

        if verifier.has_errors:
            click.echo("The TLV blob has errors!", err=True)
            print_verifier_to_console(verifier)
            raise SPSDKAppError("TLV verification failed")

        if not problems:
            click.echo(f"TLV blob size: {len(data)} bytes")
            click.echo(f"TLV blob details:\n{str(tlv_instance)}")

        print_verifier_to_console(verifier)
        click.echo("TLV blob verification passed successfully.")

    except SPSDKError as exc:
        click.echo(f"TLV blob verification failed: {binary} ,({str(exc)})")
        raise SPSDKAppError(f"TLV verification failed: {str(exc)}") from exc
