#!/usr/bin/env python
# -*- coding: UTF-8 -*-
#
# Copyright 2023-2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause

"""Console script for WPC operations."""


import logging
import sys

import click

from spsdk.apps.utils import spsdk_logger
from spsdk.apps.utils.common_cli_options import (
    CommandsTreeGroup,
    spsdk_apps_common_options,
    spsdk_config_option,
    spsdk_family_option,
    spsdk_output_option,
)
from spsdk.apps.utils.utils import catch_spsdk_error
from spsdk.utils.config import Config
from spsdk.utils.family import FamilyRevision
from spsdk.utils.misc import load_binary, write_file
from spsdk.wpc.wpc import WPC, ConfigCheckScope, WPCCertChain


@click.group(name="nxpwpc", cls=CommandsTreeGroup)
@spsdk_apps_common_options
def main(log_level: int) -> None:
    """Utility covering WPC operations."""
    spsdk_logger.install(level=log_level or logging.WARNING)


@main.command(name="insert-cert", no_args_is_help=True)
@spsdk_config_option(klass=WPC)
@click.option(
    "-s",
    "--save-debug-data",
    is_flag=True,
    default=False,
    help="Save the data being transferred (for debugging purposes).",
)
def insert_cert(config: Config, save_debug_data: bool) -> None:
    """Perform full WPC Cert chain flow.

    \b
      - Retrieve WPC ID from the target
      - Generate WPC cert chain on service
      - Insert WPC certificate into the target.

    Parameters for target and service may be passed using "-tp" or "-sp" options respectively,
    or via a config file ("-c"). The config file template can be generated using "get-template" command.
    """
    WPC.check_wpc_top_level_config(config=config, scope=ConfigCheckScope.FULL)
    wpc = WPC.load_from_config(config)
    assert wpc.target and wpc.service
    wpc_id_data = wpc.target.get_wpc_id()
    if save_debug_data:
        write_file(wpc_id_data, "x_csr.pem", mode="wb")
    wpc_cert = wpc.service.get_wpc_cert(wpc_id_data=wpc_id_data)
    if save_debug_data:
        wpc_cert.save(chain_path="x_cert_chain.bin")
    wpc.target.wpc_insert_cert(wpc_cert)
    click.echo("Inserting WPC certificate finished successfully.")


@main.command(name="get-id", no_args_is_help=True)
@spsdk_config_option(klass=WPC)
@spsdk_output_option()
def get_wpc_id(
    config: Config,
    output: str,
) -> None:
    """Get the WPC identifier for WPC service appropriate for given family."""
    WPC.check_wpc_top_level_config(config=config, scope=ConfigCheckScope.TARGET)
    wpc = WPC.load_from_config(config)
    assert wpc.target
    wpc_id_data = wpc.target.get_wpc_id()
    write_file(wpc_id_data, output, mode="wb")
    click.echo(f"Creating {output} WPC ID file.")


@main.command(name="get-cert", no_args_is_help=True)
@spsdk_config_option(klass=WPC)
@click.option(
    "-i",
    "--wpc-id",
    type=click.Path(resolve_path=True, exists=True, dir_okay=False),
    required=True,
    help="Path to WPC-ID generated by `get-id` command.",
)
@spsdk_output_option()
def get_wpc_cert(
    config: Config,
    wpc_id: str,
    output: str,
) -> None:
    """Get WPC Certificate Chain for given WPC ID appropriate for given family."""
    WPC.check_wpc_top_level_config(config=config, scope=ConfigCheckScope.SERVICE)
    wpc = WPC.load_from_config(config)
    assert wpc.service
    wpc_id_data = load_binary(wpc_id)
    wpc_cert = wpc.service.get_wpc_cert(wpc_id_data=wpc_id_data)
    wpc_cert.save(chain_path=output)
    click.echo(f"Creating {output} WPC Certificate Chain.")


@main.command(name="put-cert", no_args_is_help=True)
@spsdk_config_option(klass=WPC)
@click.option(
    "-i",
    "--wpc-cert",
    type=click.Path(resolve_path=True, exists=True, dir_okay=False),
    required=True,
    help="Path to WPC-CERT generated by `get-cert` command.",
)
def put_wpc_cert(
    config: Config,
    wpc_cert: str,
) -> None:
    """Insert the WPC Certificate Chain into the target."""
    WPC.check_wpc_top_level_config(config=config, scope=ConfigCheckScope.TARGET)
    wpc = WPC.load_from_config(config)
    assert wpc.target
    cert_chain = WPCCertChain.load(wpc_cert)
    wpc.target.wpc_insert_cert(cert_chain=cert_chain)
    click.echo("Inserting WPC certificate finished successfully.")


@main.command(name="get-template", no_args_is_help=True)
@click.option(
    "-st",
    "--service-type",
    type=click.Choice(list(WPC.SERVICES.keys()), case_sensitive=False),
    required=True,
    help="Name of the WPC service type.",
)
@click.option(
    "-tt",
    "--target-type",
    type=click.Choice(list(WPC.TARGETS.keys()), case_sensitive=False),
    required=True,
    help="Name of the MCU target type.",
)
@spsdk_family_option(families=WPC.get_supported_families())
@spsdk_output_option(force=True)
def get_template(service_type: str, target_type: str, family: FamilyRevision, output: str) -> None:
    """Generate a configuration template."""
    template = WPC.get_config_template(
        family=family,
        service=WPC.SERVICES[service_type],
        target=WPC.TARGETS[target_type],
    )
    write_file(template, output)
    click.echo(f"The WPC template for {family} has been saved into {output} YAML file")


@catch_spsdk_error
def safe_main() -> None:
    """Calls the main function."""
    sys.exit(main())  # pylint: disable=no-value-for-parameter


if __name__ == "__main__":
    safe_main()
