#!/usr/bin/env python
# -*- coding: UTF-8 -*-
#
# Copyright 2020-2025 NXP
#
# SPDX-License-Identifier: BSD-3-Clause

"""Module with Debug Authentication Challenge (DAC) Packet."""

import logging
from struct import calcsize, pack, unpack_from

from typing_extensions import Self

from spsdk.dat.protocol_version import ProtocolVersion
from spsdk.utils.database import DatabaseManager
from spsdk.utils.family import FamilyRevision, get_db

logger = logging.getLogger(__name__)


class DebugAuthenticationChallenge:
    """Base class for DebugAuthenticationChallenge."""

    def __init__(
        self,
        family: FamilyRevision,
        version: ProtocolVersion,
        socc: int,
        uuid: bytes,
        rotid_rkh_revocation: int,
        rotid_rkth_hash: bytes,
        cc_soc_pinned: int,
        cc_soc_default: int,
        cc_vu: int,
        challenge: bytes,
    ) -> None:
        """Initialize the DebugAuthenticationChallenge object.

        :param version: The string representing version: for RSA: 1.0, for ECC: 2.0, 2.1, 2.2
        :param socc: The SoC Class that this credential applies to
        :param uuid: The string representing the unique device identifier
        :param rotid_rkh_revocation: State of certificate revocation field
        :param rotid_rkth_hash: The hash of roth-meta data
        :param cc_soc_pinned: State of lock bits in the debugger configuration field
        :param cc_soc_default: State of the debugger configuration field
        :param cc_vu: The Vendor usage that the vendor has associated with this credential
        :param challenge: Randomly generated bytes from the target
        """
        self.family = family
        self.version = version
        self.socc = socc
        self.uuid = uuid
        self.rotid_rkh_revocation = rotid_rkh_revocation
        self.rotid_rkth_hash = rotid_rkth_hash
        self.cc_soc_pinned = cc_soc_pinned
        self.cc_soc_default = cc_soc_default
        self.cc_vu = cc_vu
        self.challenge = challenge

        db = get_db(self.family)
        self.swapped_version = db.get_bool(DatabaseManager.DAT, "dac_version_is_swapped", False)

    def __repr__(self) -> str:
        return f"DAC v{self.version}, SOCC: 0x{self.socc:08X}"

    def __str__(self) -> str:
        """String representation of DebugCredential."""
        msg = f"Version                : {self.version}\n"
        msg += f"SOCC                   : 0x{self.socc:08X}\n"
        msg += f"UUID                   : {self.uuid.hex().upper()}\n"
        msg += f"CC_VU                  : {self.cc_vu}\n"
        msg += f"ROTID_rkh_revocation   : {format(self.rotid_rkh_revocation, '08X')}\n"
        msg += f"ROTID_rkth_hash        : {self.rotid_rkth_hash.hex()}\n"
        msg += f"CC_soc_pinned          : {format(self.cc_soc_pinned, '08X')}\n"
        msg += f"CC_soc_default         : {format(self.cc_soc_default, '08X')}\n"
        msg += f"Challenge              : {self.challenge.hex()}\n"
        return msg

    def export(self) -> bytes:
        """Exports the DebugAuthenticationChallenge into bytes."""
        major, minor = (
            (self.version.minor, self.version.major)
            if self.swapped_version
            else (self.version.major, self.version.minor)
        )
        data = pack("<2H", *[major, minor])
        data += pack("<L", self.socc)
        data += self.uuid
        data += pack("<L", self.rotid_rkh_revocation)
        data += self.rotid_rkth_hash
        data += pack("<L", self.cc_soc_pinned)
        data += pack("<L", self.cc_soc_default)
        data += pack("<L", self.cc_vu)
        data += self.challenge
        return data

    @staticmethod
    def get_rot_hash_length(family: FamilyRevision, major_ver: int, minor_ver: int) -> int:
        """Returns length of Root Of Trust hash length.

        :param family: The chip family name
        :param major_ver: Major version
        :param minor_ver: Minor version
        :return: Length of Root Of Trust hash in bytes
        """
        if get_db(family).get_bool(DatabaseManager.DAT, "dat_is_using_sha256_always", False):
            return 32

        if major_ver == 2:
            if minor_ver == 1:
                return 48
            if minor_ver == 2:
                return 64

        return 32

    @classmethod
    def parse(cls, data: bytes, family: FamilyRevision = FamilyRevision("Unknown")) -> Self:
        """Parse the data into a DebugAuthenticationChallenge.

        :param data: Raw data as bytes
        :param family: The CPU family
        :return: DebugAuthenticationChallenge object
        """
        format_head = "<2HL16sL"
        version_major, version_minor, socc, uuid, rotid_rkh_revocation = unpack_from(
            format_head, data
        )
        db = get_db(family)

        hash_length = cls.get_rot_hash_length(family, version_major, version_minor)

        # Some soc handles version differently in DAC
        swapped_version = db.get_bool(DatabaseManager.DAT, "dac_version_is_swapped", False)
        if swapped_version:
            version_major, version_minor = version_minor, version_major

        format_tail = f"<{hash_length}s3L32s"
        (
            rotid_rkth_hash,
            cc_soc_pinned,
            cc_soc_default,
            cc_vu,
            challenge,
        ) = unpack_from(format_tail, data, calcsize(format_head))
        return cls(
            family=family,
            version=ProtocolVersion.from_version(version_major, version_minor),
            socc=socc,
            uuid=uuid,
            rotid_rkh_revocation=rotid_rkh_revocation,
            rotid_rkth_hash=rotid_rkth_hash,
            cc_soc_default=cc_soc_default,
            cc_soc_pinned=cc_soc_pinned,
            cc_vu=cc_vu,
            challenge=challenge,
        )
