"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsAuth = void 0;
const core_1 = require("@aws-cdk/core");
const k8s_manifest_1 = require("./k8s-manifest");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * (experimental) Manages mapping between IAM users and roles to Kubernetes RBAC configuration.
 *
 * @see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
 * @experimental
 */
class AwsAuth extends core_2.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.roleMappings = new Array();
        this.userMappings = new Array();
        this.accounts = new Array();
        this.stack = core_1.Stack.of(this);
        new k8s_manifest_1.KubernetesManifest(this, 'manifest', {
            cluster: props.cluster,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'ConfigMap',
                    metadata: {
                        name: 'aws-auth',
                        namespace: 'kube-system',
                    },
                    data: {
                        mapRoles: this.synthesizeMapRoles(),
                        mapUsers: this.synthesizeMapUsers(),
                        mapAccounts: this.synthesizeMapAccounts(),
                    },
                },
            ],
        });
    }
    /**
     * (experimental) Adds the specified IAM role to the `system:masters` RBAC group, which means that anyone that can assume it will be able to administer this Kubernetes system.
     *
     * @param role The IAM role to add.
     * @param username Optional user (defaults to the role ARN).
     * @experimental
     */
    addMastersRole(role, username) {
        this.addRoleMapping(role, {
            username,
            groups: ['system:masters'],
        });
    }
    /**
     * (experimental) Adds a mapping between an IAM role to a Kubernetes user and groups.
     *
     * @param role The IAM role to map.
     * @param mapping Mapping to k8s user name and groups.
     * @experimental
     */
    addRoleMapping(role, mapping) {
        this.assertSameStack(role);
        this.roleMappings.push({ role, mapping });
    }
    /**
     * (experimental) Adds a mapping between an IAM user to a Kubernetes user and groups.
     *
     * @param user The IAM user to map.
     * @param mapping Mapping to k8s user name and groups.
     * @experimental
     */
    addUserMapping(user, mapping) {
        this.assertSameStack(user);
        this.userMappings.push({ user, mapping });
    }
    /**
     * (experimental) Additional AWS account to add to the aws-auth configmap.
     *
     * @param accountId account number.
     * @experimental
     */
    addAccount(accountId) {
        this.accounts.push(accountId);
    }
    assertSameStack(construct) {
        const thisStack = core_1.Stack.of(this);
        if (core_1.Stack.of(construct) !== thisStack) {
            // aws-auth is always part of the cluster stack, and since resources commonly take
            // a dependency on the cluster, allowing those resources to be in a different stack,
            // will create a circular dependency. granted, it won't always be the case,
            // but we opted for the more causious and restrictive approach for now.
            throw new Error(`${construct.node.path} should be defined in the scope of the ${thisStack.stackName} stack to prevent circular dependencies`);
        }
    }
    synthesizeMapRoles() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.roleMappings.map(m => {
                var _a;
                return ({
                    rolearn: m.role.roleArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.role.roleArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapUsers() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.userMappings.map(m => {
                var _a;
                return ({
                    userarn: m.user.userArn,
                    username: (_a = m.mapping.username) !== null && _a !== void 0 ? _a : m.user.userArn,
                    groups: m.mapping.groups,
                });
            })),
        });
    }
    synthesizeMapAccounts() {
        return core_1.Lazy.anyValue({
            produce: () => this.stack.toJsonString(this.accounts),
        });
    }
}
exports.AwsAuth = AwsAuth;
//# sourceMappingURL=data:application/json;base64,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