"""
Basic types for path planning and navigation.
"""

from typing import Any

from ..planning.actions import ExecutionResult
from ..utils.path import Path
from ..utils.pose import Pose
from ..utils.search_graph import SearchGraph


class PathPlanner:
    """
    Generic path planner class that helps with type hinting.

    When implementing a new planner, you should subclass from this class.
    """

    plugin_name: str
    """The name of the plugin. Must be implemented by child class."""

    registered_plugins: dict[str, Any] = {}
    """List of registered path planner plugins."""

    def __init__(self) -> None:
        from ..core.robot import Robot
        from ..core.world import World

        self.robot: Robot | None = None
        self.world: World | None = None

    def __init_subclass__(cls, **kwargs: Any):
        """Registers a path planner subclass."""
        cls.registered_plugins[cls.plugin_name] = cls

    def plan(self, start: Pose, goal: Pose) -> Path:
        """
        Plans a path from start to goal.

        :param start: Start pose.
        :param goal: Goal pose.
        :return: Path from start to goal.
        """
        raise NotImplementedError("Must implement in subclass.")

    def reset(self) -> None:
        """Resets the path planner."""
        self.latest_path = Path()
        if self.robot is not None:
            self.world = self.robot.world

    def get_graphs(self) -> list[SearchGraph]:
        """
        Returns the graphs generated by the planner, if any.

        :return: List of graphs.
        """
        raise NotImplementedError("Must implement in subclass.")

    def get_latest_path(self) -> Path | None:
        """
        Returns the latest path generated by the planner, if any.

        :return: The latest path if one exists, else None.
        """
        raise NotImplementedError("Must implement in subclass.")

    def to_dict(self) -> dict[str, Any]:
        """
        Serializes the planner to a dictionary.

        :return: A dictionary containing the planner information.
        """
        raise NotImplementedError("Must implement in subclass.")


class PathExecutor:
    """
    Generic path executor class that helps with type hinting.

    When implementing a new executor, you should subclass from this class.
    """

    following_path = False
    """Flag to track path following."""

    cancel_execution = False
    """Flag to cancel from user."""

    def __init__(self) -> None:
        from ..core.robot import Robot

        self.robot: Robot | None = None
        self.cancel_all_threads = False

    def execute(
        self, path: Path, realtime_factor: float = 1.0, battery_usage: float = 0.0
    ) -> ExecutionResult:
        """
        Generates and executes a trajectory on the robot.

        :param path: Path to execute on the robot.
        :param realtime_factor: A multiplier on the execution time relative to
            real time, defaults to 1.0.
        :param battery_usage: Robot battery usage per unit distance.
        :return: An object describing the execution result.
        """
        raise NotImplementedError("Must implement in subclass.")

    def validate_sensors_for_partial_obs_hallways(self) -> None:
        """
        Validates if the lidar sensor is set up correctly with partial hallway observability.
        """
        raise NotImplementedError("Must implement in subclass.")

    def to_dict(self) -> dict[str, Any]:
        """
        Serializes the executor to a dictionary.

        :return: A dictionary containing the executor information.
        """
        raise NotImplementedError("Must implement in subclass.")
