import asyncio
from os import getenv

from base_test import BaseTest

from zabbixci import ZabbixCI
from zabbixci.cache.cleanup import Cleanup

DEV_ZABBIX_URL = getenv("ZABBIX_URL")
DEV_ZABBIX_TOKEN = getenv("ZABBIX_TOKEN")
DEV_GIT_REMOTE = getenv("REMOTE")
CACHE_PATH = getenv("CACHE_PATH")


class BaseTemplates(BaseTest):
    def setUp(self):
        self.prep()

        self.settings.ZABBIX_URL = DEV_ZABBIX_URL
        self.settings.ZABBIX_TOKEN = DEV_ZABBIX_TOKEN
        self.settings.REMOTE = DEV_GIT_REMOTE
        self.settings.SKIP_VERSION_CHECK = True
        self.settings.REGEX_MATCHING = False
        self.settings.SET_VERSION = True

        self.settings.SYNC_TEMPLATES = True
        self.settings.SYNC_ICONS = False
        self.settings.SYNC_BACKGROUNDS = False
        self.settings.SYNC_ICON_MAPS = False
        self.settings.SYNC_SCRIPTS = False

        self.settings.TEMPLATE_WHITELIST = ""
        self.settings.TEMPLATE_BLACKLIST = ""

        self.zci = ZabbixCI(self.settings)

    async def restore_state(self):
        if not self.settings.REMOTE:
            return

        self.zci._git.force_push(
            ["+refs/remotes/origin/test:refs/heads/main"],
            self.settings.REMOTE,
        )

        Cleanup.cleanup_cache(self.settings, full=True)
        self.zci.create_git()

        whitelist = self.settings.TEMPLATE_WHITELIST
        blacklist = self.settings.TEMPLATE_BLACKLIST

        # Restore the state of Zabbix
        await self.zci.pull()

        self.settings.TEMPLATE_WHITELIST = whitelist
        self.settings.TEMPLATE_BLACKLIST = blacklist

    async def asyncSetUp(self):
        self.zci.create_git()
        await self.zci.create_zabbix()
        await self.restore_state()

    async def test_push_pull_remote_defaults(self):
        # Push default Zabbix templates to remote
        await self.zci.push()

        # No changed when we have just pushed
        changed = await self.zci.pull()
        self.assertFalse(changed)

    async def test_template_change(self):
        # Rename a template
        template_id = self.zci._zabbix.get_templates_name(["Windows by Zabbix agent"])[
            0
        ]["templateid"]
        self.zci._zabbix.set_template(
            template_id, {"name": "Windows by Zabbix agent (renamed)"}
        )

        # Push changes to git
        changed = await self.zci.push()
        self.assertTrue(changed, "Template change not detected")

        # Revert changes in Zabbix
        self.zci._zabbix.set_template(template_id, {"name": "Windows by Zabbix agent"})

        # Restore to Git version
        changed = await self.zci.pull()
        self.assertTrue(changed, "Template was not restored")

        # Assert Git version is imported back into Zabbix
        matches = self.zci._zabbix.get_templates(
            [self.settings.ROOT_TEMPLATE_GROUP], ["Windows by Zabbix agent"]
        )
        self.assertEqual(len(matches), 1, "Template not found")

    async def test_template_rename(self):
        # Rename a template
        template_id = self.zci._zabbix.get_templates_name(["Linux by Zabbix agent"])[0][
            "templateid"
        ]
        self.zci._zabbix.set_template(template_id, {"host": "Linux by Zabbix 00000"})

        # Push changes to git
        changed = await self.zci.push()
        self.assertTrue(changed, "Renaming not detected")

        # Make changes in Zabbix
        self.zci._zabbix.set_template(template_id, {"host": "Linux by Zabbix agent"})

        # Restore to Git version
        changed = await self.zci.pull()
        self.assertTrue(changed, "Template was not restored")

        # Assert Git version is restored
        matches = self.zci._zabbix.get_templates(
            [self.settings.ROOT_TEMPLATE_GROUP], ["Linux by Zabbix 00000"]
        )
        self.assertEqual(len(matches), 1, "Template not found")

    async def test_template_delete(self):
        # Delete a template
        template_id = self.zci._zabbix.get_templates_name(
            ["Acronis Cyber Protect Cloud by HTTP"]
        )[0]["templateid"]
        self.zci._zabbix.delete_templates([template_id])

        # Push changes to git
        changed = await self.zci.push()
        self.assertTrue(changed, "Deletion from Zabbix not detected")

        self.settings.PULL_BRANCH = "test"

        changed = await self.zci.pull()
        self.assertTrue(changed, "Template was not restored")

        self.settings.PULL_BRANCH = "main"
        changed = await self.zci.pull()
        self.assertTrue(changed, "Template deletion from Git was not detected")

    async def test_push_to_new_branch(self):
        self.settings.PUSH_BRANCH = "new-branch"

        # Push default Zabbix templates to remote
        await self.zci.push()

        self.settings.PUSH_BRANCH = "main"

    async def asyncTearDown(self):
        await self.zci._zabbix.zapi.logout()

        # Close custom session, if it exists
        if self.zci._zabbix._client_session:
            await self.zci._zabbix._client_session.close()

        # Wait a couple of cycles for the session to close
        await asyncio.sleep(0.25)
