from contextlib import suppress
from typing import Any
from typing import TYPE_CHECKING
from typing import TypeAlias
from typing import TypeVar
from typing import Union

from pydantic import BaseModel
from pydantic.errors import PydanticUndefinedAnnotation

"""
Gathered from comments in: https://github.com/pydantic/pydantic/discussions/3089

Adds a Partial wrapper that takes any Pydantic model and makes all defined
fields optional, for usage in for examples PATCH updates.
"""

__all__ = (
    "OptionalModel",
    "Partial",
)


if TYPE_CHECKING:
    T = TypeVar("T")

    Partial: TypeAlias = Union[T]
else:

    class Partial:
        def __class_getitem__(cls, item):
            return type(f"Partial{item.__name__}", (item, OptionalModel), {})


class OptionalModel(BaseModel):
    @classmethod
    def __pydantic_init_subclass__(cls, **kwargs: Any) -> None:
        super().__pydantic_init_subclass__(**kwargs)
        required = [f for f in cls.model_fields.values() if f.default is not None]

        for field in required:
            field.default = None
            field.annotation = field.annotation | None

        with suppress(PydanticUndefinedAnnotation):
            cls.model_rebuild(force=True)
