"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdktfResolver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cdktf_1 = require("cdktf");
const exec_1 = require("./exec");
class CdktfResolver {
    constructor(props) {
        this.app = props.app;
    }
    resolve(context) {
        // detects full resources being passed as values.
        // this is legit in terraform and should resolve
        // to the resource fqn.
        // see https://github.com/hashicorp/terraform-cdk/blob/main/packages/cdktf/lib/terraform-output.ts#L79
        const addressable = this.isAddressable(context.value);
        if (!cdktf_1.Token.isUnresolved(context.value) && !addressable) {
            return;
        }
        const output = this.findOutput(context.value);
        try {
            const outputValue = this.fetchOutputValue(output);
            context.replaceValue(outputValue);
        }
        catch (err) {
            // if both cdk8s and CDKTF applications are defined within the same file,
            // a cdk8s synth is going to happen before the CDKTF deployment.
            // in this case we must swallow the error, otherwise the CDKTF deployment
            // won't be able to go through. we replace the value with something to indicate
            // that a fetching attempt was made and failed.
            context.replaceValue(`Failed fetching value for output ${output.node.path}: ${err}`);
        }
    }
    fetchOutputValue(output) {
        if (!this._outputs) {
            this._outputs = this.fetchOutputs();
        }
        const stack = cdktf_1.TerraformStack.of(output);
        return this._outputs[stack.node.id][output.friendlyUniqueId];
    }
    fetchOutputs() {
        const projectDir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdktf-project-'));
        const outputsFile = 'outputs.json';
        try {
            this.app.synth();
            const cdktfJson = {
                // `cdktf output` doesn't actually use this value,
                // so we can put whatever we want here.
                language: 'python',
                app: 'cdktf.out',
            };
            // create our own copy of the synthesized app so we can safely clean it up
            copyDirectory(this.app.outdir, path.join(projectDir, cdktfJson.app));
            // write the configuration file as it is required for any cdktf command
            fs.writeFileSync(path.join(projectDir, 'cdktf.json'), JSON.stringify(cdktfJson));
            const stacks = this.app.node.findAll().filter(c => cdktf_1.TerraformStack.isStack(c)).map(c => c.node.id);
            const command = ['cdktf output',
                '--skip-synth',
                `--output ${cdktfJson.app}`,
                `--outputs-file ${outputsFile}`,
                `${stacks.join(' ')}`];
            (0, exec_1.execSync)(command.join(' '), { cwd: projectDir });
            return JSON.parse(fs.readFileSync(path.join(projectDir, outputsFile), { encoding: 'utf-8' }));
        }
        finally {
            fs.rmSync(projectDir, { recursive: true });
        }
    }
    findOutput(value) {
        const inspectedStacks = this.app.node.findAll().filter(c => cdktf_1.TerraformStack.isStack(c));
        for (const stack of inspectedStacks) {
            const output = stack.node.findAll().filter(c => c instanceof cdktf_1.TerraformOutput && c.value === value)[0];
            // we don't really care if there are more outputs (possibly from different stacks)
            // that point to the same value. the first will suffice.
            if (output)
                return output;
        }
        // This can happen if either:
        // --------------------------
        //  1. User didn't define an output.
        //  2. User defined a complex literal output value (e.g { bucketName: bucket.bucket }).
        throw new Error(`Unable to find output defined for ${value} (Inspected stacks: ${inspectedStacks.map(s => s.node.id).join(',')})`);
    }
    isAddressable(object) {
        return object && typeof object === 'object' && !Array.isArray(object) && 'fqn' in object;
    }
}
exports.CdktfResolver = CdktfResolver;
_a = JSII_RTTI_SYMBOL_1;
CdktfResolver[_a] = { fqn: "@cdk8s/cdktf-resolver.CdktfResolver", version: "0.0.15" };
function copyDirectory(sourceDir, targetDir) {
    if (!fs.existsSync(targetDir)) {
        fs.mkdirSync(targetDir);
    }
    const files = fs.readdirSync(sourceDir);
    for (const file of files) {
        const sourceFilePath = path.join(sourceDir, file);
        const targetFilePath = path.join(targetDir, file);
        const stats = fs.statSync(sourceFilePath);
        if (stats.isDirectory()) {
            copyDirectory(sourceFilePath, targetFilePath);
        }
        else if (stats.isFile()) {
            fs.copyFileSync(sourceFilePath, targetFilePath);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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