"""Validate the handling of the vendor's v0 APIs (URLs) for Authorization.

This is used to:
  a) document the RESTful API that is provided by the vendor
  b) confirm the faked server (if any) is behaving as per a)

Testing is at HTTP request layer (e.g. GET).
Everything to/from the RESTful API is in camelCase (so those schemas are used).
"""

from __future__ import annotations

from http import HTTPMethod, HTTPStatus
from typing import TYPE_CHECKING

import pytest

import evohomeasync as evo0
from tests.const import _DBG_USE_REAL_AIOHTTP

from .common import should_fail_v0, should_work_v0, skipif_auth_failed

if TYPE_CHECKING:
    from tests.conftest import EvohomeClientv0


async def _test_usr_locations(evo: EvohomeClientv0) -> None:
    """Test /locations?userId={user_id}&allData=True"""

    usr_id: int = evo.user_account["user_id"]
    # loc_id: int = evo.location_id

    url = f"locations?userId={usr_id}&allData=True"
    _ = await should_work_v0(evo.auth, HTTPMethod.GET, url)  # FIXME: add schema

    # why isn't this one METHOD_NOT_ALLOWED?
    _ = await should_fail_v0(evo.auth, HTTPMethod.PUT, url, status=HTTPStatus.NOT_FOUND)

    url = f"locations?userId={usr_id}"
    _ = await should_work_v0(evo.auth, HTTPMethod.GET, url, schema=None)

    url = "locations?userId=123456"
    _ = await should_fail_v0(
        evo.auth, HTTPMethod.GET, url, status=HTTPStatus.UNAUTHORIZED
    )

    url = "locations?userId='123456'"
    _ = await should_fail_v0(
        evo.auth, HTTPMethod.GET, url, status=HTTPStatus.BAD_REQUEST
    )

    url = "xxxxxxx"  # NOTE: a general test, not a test specific to the 'locations' URL
    _ = await should_fail_v0(
        evo.auth,
        HTTPMethod.GET,
        url,
        status=HTTPStatus.NOT_FOUND,
        content_type="text/html",  # not the usual content-type
    )


async def _test_evo_systems(evo: EvohomeClientv0) -> None:
    """Test /evoTouchSystems?locationId={loc_id}"""

    # usr_id: int = evo.user_account["user_id"]
    loc_id = evo.locations[0].id

    #
    # TEST 0: unsupported method?
    url = f"evoTouchSystems?locationId={loc_id}"
    _ = await should_fail_v0(
        evo.auth,
        HTTPMethod.GET,
        url,
        content_type="text/html",  # not the usual content-type
        status=HTTPStatus.NOT_FOUND,
    )
    # '<!DOCTYPE html PUBLIC ... not found ...'

    #
    # TEST 0: supported method?
    json = {
        "QuickAction": "Auto",
        "QuickActionNextTime": None,
    }

    #       evoTouchSystems?locationId=$locationId
    url = f"evoTouchSystems?locationId={loc_id}"
    _ = await should_fail_v0(
        evo.auth,
        HTTPMethod.PUT,
        url,
        json=json,
        content_type="text/html",  # not the usual content-type
        status=HTTPStatus.NOT_FOUND,
    )
    # '<!DOCTYPE html PUBLIC ... not found ...'


# GET /accountInfo (is also in test_v0_urls_auth.py)
# @skipif_auth_failed
# async def test_usr_account(evohome_v0: EvohomeClientv0) -> None:
#     """Test /accountInfo"""


# GET /locations?userId={user_id}&allData=True
@skipif_auth_failed
async def test_usr_locations(evohome_v0: EvohomeClientv0) -> None:
    """Test /locations?userId={user_id}&allData=True"""

    if not _DBG_USE_REAL_AIOHTTP:
        pytest.skip("Mocked server not implemented for this test")

    try:
        await evohome_v0.update()  # get user_id and location_id

        await _test_usr_locations(evohome_v0)

    except evo0.AuthenticationFailedError:
        if not _DBG_USE_REAL_AIOHTTP:
            raise
        pytest.skip("Unable to authenticate with real server")


# PUT /evoTouchSystems?locationId={loc_id}
@skipif_auth_failed
async def test_evo_systems(evohome_v0: EvohomeClientv0) -> None:
    """Test /evoTouchSystems?locationId={loc_id}"""

    if not _DBG_USE_REAL_AIOHTTP:
        pytest.skip("Mocked server not implemented for this test")

    try:
        await evohome_v0.update()  # get user_id and location_id

        await _test_evo_systems(evohome_v0)

    except evo0.AuthenticationFailedError:
        if not _DBG_USE_REAL_AIOHTTP:
            raise
        pytest.skip("Unable to authenticate with real server")


# PUT /devices/{zone_id}/thermostat/changeableValues/heatSetpoint
# PUT /devices/{dhw_id}/thermostat/changeableValues


USER_DATA = {
    "sessionId": "BE5F40A6-1234-1234-1234-A708947D6399",
    "userInfo": {
        "userID": 1234567,
        "username": "username@email.com",
        "firstname": "David",
        "lastname": "Smith",
        "streetAddress": "1 Main Street",
        "city": "London",
        "zipcode": "NW1 1AA",
        "country": "GB",
        "telephone": "",
        "userLanguage": "en-GB",
        "isActivated": True,
        "deviceCount": 0,
        "tenantID": 5,
        "securityQuestion1": "NotUsed",
        "securityQuestion2": "NotUsed",
        "securityQuestion3": "NotUsed",
        "latestEulaAccepted": False,
    },
}

FULL_DATA = {
    "locationID": 2738909,
    "name": "My Home",
    "streetAddress": "1 Main Street",
    "city": "London",
    "state": "",
    "country": "GB",
    "zipcode": "NW1 1AA",
    "type": "Residential",
    "hasStation": True,
    "devices": [
        {
            "gatewayId": 2499896,
            "deviceID": 3933910,
            "thermostatModelType": "DOMESTIC_HOT_WATER",
            "deviceType": 128,
            "name": "",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 22.77,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "Measured",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["DHWOn", "DHWOff"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 30.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {"mode": "DHWOff", "status": "Scheduled"},
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 3933910,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 250,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 3432579,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Bathroom Dn",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 20.79,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "Measured",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 15.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 3432579,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 4,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 3449740,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Bathroom Up",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 20.26,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "Measured",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 19.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 3449740,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 7,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 3432521,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Dead Zone",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 128.0,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "NotAvailable",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 5.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 3432521,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 0,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 5333958,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Eh",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 128.0,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "NotAvailable",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 21.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 5333958,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 11,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 3432577,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Front Room",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 19.83,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "Measured",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 20.5, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 3432577,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 2,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 3449703,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Kids Room",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 19.53,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "Measured",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 16.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 3449703,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 6,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 3432578,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Kitchen",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 20.43,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "Measured",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 15.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 3432578,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 3,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 3432580,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Main Bedroom",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 20.72,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "Measured",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 16.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 3432580,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 5,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 3432576,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Main Room",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 20.14,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "Measured",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 15.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 3432576,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 1,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 3450733,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Spare Room",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 18.81,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "Measured",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 16.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 3450733,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 8,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 5333957,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "UFH",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 128.0,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "NotAvailable",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 21.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 5333957,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 10,
        },
        {
            "gatewayId": 2499896,
            "deviceID": 5333955,
            "thermostatModelType": "EMEA_ZONE",
            "deviceType": 128,
            "name": "Zv",
            "scheduleCapable": False,
            "holdUntilCapable": False,
            "thermostat": {
                "units": "Celsius",
                "indoorTemperature": 128.0,
                "outdoorTemperature": 128.0,
                "outdoorTemperatureAvailable": False,
                "outdoorHumidity": 128.0,
                "outdootHumidityAvailable": False,
                "indoorHumidity": 128.0,
                "indoorTemperatureStatus": "NotAvailable",
                "indoorHumidityStatus": "NotAvailable",
                "outdoorTemperatureStatus": "NotAvailable",
                "outdoorHumidityStatus": "NotAvailable",
                "isCommercial": False,
                "allowedModes": ["Heat", "Off"],
                "deadband": 0.0,
                "minHeatSetpoint": 5.0,
                "maxHeatSetpoint": 35.0,
                "minCoolSetpoint": 50.0,
                "maxCoolSetpoint": 90.0,
                "changeableValues": {
                    "mode": "Off",
                    "heatSetpoint": {"value": 21.0, "status": "Scheduled"},
                    "vacationHoldDays": 0,
                },
                "scheduleCapable": False,
                "vacationHoldChangeable": False,
                "vacationHoldCancelable": False,
                "scheduleHeatSp": 0.0,
                "scheduleCoolSp": 0.0,
            },
            "alertSettings": {
                "deviceID": 5333955,
                "tempHigherThanActive": True,
                "tempHigherThan": 30.0,
                "tempHigherThanMinutes": 0,
                "tempLowerThanActive": True,
                "tempLowerThan": 5.0,
                "tempLowerThanMinutes": 0,
                "faultConditionExistsActive": False,
                "faultConditionExistsHours": 0,
                "normalConditionsActive": True,
                "communicationLostActive": False,
                "communicationLostHours": 0,
                "communicationFailureActive": True,
                "communicationFailureMinutes": 15,
                "deviceLostActive": False,
                "deviceLostHours": 0,
            },
            "isUpgrading": False,
            "isAlive": True,
            "thermostatVersion": "02.00.19.33",
            "macID": "00D02DEE4E56",
            "locationID": 2738909,
            "domainID": 20054,
            "instance": 9,
        },
    ],
    "oneTouchButtons": [],
    "weather": {
        "condition": "NightClear",
        "temperature": 9.0,
        "units": "Celsius",
        "humidity": 87,
        "phrase": "Clear",
    },
    "daylightSavingTimeEnabled": True,
    "timeZone": {
        "id": "GMT Standard Time",
        "displayName": "(UTC+00:00) Dublin, Edinburgh, Lisbon, London",
        "offsetMinutes": 0,
        "currentOffsetMinutes": 0,
        "usingDaylightSavingTime": True,
    },
    "oneTouchActionsSuspended": False,
    "isLocationOwner": True,
    "locationOwnerID": 2263181,
    "locationOwnerName": "David Smith",
    "locationOwnerUserName": "null@gmail.com",
    "canSearchForContractors": True,
    "contractor": {
        "info": {"contractorID": 1839},
        "monitoring": {"levelOfAccess": "Partial", "contactPreferences": []},
    },
}
