import typing
from pathlib import Path
from typing import Annotated
from typing import Optional

import typer
from amsdal.errors import AmsdalCloudError
from amsdal.manager import AmsdalManager
from amsdal.manager import AsyncAmsdalManager
from amsdal_utils.config.manager import AmsdalConfigManager
from rich import print as rprint
from typer import Option

from amsdal_cli.commands.cloud.environments.utils import get_current_env
from amsdal_cli.commands.cloud.security.allowlist.app import allowlist_sub_app
from amsdal_cli.utils.cli_config import CliConfig
from amsdal_cli.utils.text import rich_error
from amsdal_cli.utils.text import rich_highlight
from amsdal_cli.utils.text import rich_info
from amsdal_cli.utils.text import rich_success


@allowlist_sub_app.command(name='delete, del, d')
def delete_allowlist_ip_command(
    ctx: typer.Context,
    env_name: typing.Annotated[
        typing.Optional[str],  # noqa: UP007
        Option('--env', help='Environment name. Default is the current environment from configratuion.'),
    ] = None,
    *,
    ip_address: Annotated[
        Optional[str],  # noqa: UP007
        typer.Option(
            '--ip-address',
            help=(
                'IP address, range or combination of both to delete from the allowlist. '
                'Will delete your IP if not provided.'
            ),
        ),
    ] = None,
) -> None:
    """
    Deletes your IP from the allowlist of the API.

    Args:
        ctx (typer.Context): The Typer context object.
        env_name (typing.Annotated[typing.Optional[str], Option]): The name of the environment. Defaults to the current
            environment from configuration.
        ip_address (Annotated[Optional[str], typer.Option]): The IP address, range, or combination of both to delete
            from the allowlist. Will delete your IP if not provided.

    Returns:
        None

    Examples:
        ```shell
        > amsdal cloud security allowlist delete
        > amsdal cloud security allowlist delete --ip-address 0.0.0.0
        > amsdal cloud security allowlist delete --ip-address 0.0.0.0/24
        > amsdal cloud security allowlist delete --ip-address 0.0.0.0,1.0.0.0/24
        ```
    """

    cli_config: CliConfig = ctx.meta['config']
    env_name = env_name or get_current_env(cli_config)

    if cli_config.verbose:
        rprint(rich_info(f'Deleting IP address/range from the allowlist for environment: {rich_highlight(env_name)}'))

    AmsdalConfigManager().load_config(Path('./config.yml'))

    manager: AsyncAmsdalManager | AmsdalManager
    if AmsdalConfigManager().get_config().async_mode:
        manager = AsyncAmsdalManager()
    else:
        manager = AmsdalManager()

    manager.authenticate()

    try:
        manager.cloud_actions_manager.delete_allowlist_ip(
            env_name=env_name,
            application_uuid=cli_config.application_uuid,
            application_name=cli_config.application_name,
            ip_address=ip_address,
        )
    except AmsdalCloudError as e:
        rprint(rich_error(str(e)))
        return

    if ip_address:
        msg = (
            f'IP address/range {rich_success(ip_address)} has been deleted from the allowlist. '
            'Rules should be applied in a few minutes.'
        )
    else:
        msg = 'Your IP address has been deleted from the allowlist. Rules should be applied in a few minutes.'

    rprint(msg)
