import pandas as pd
import numpy as np
import matplotlib.pyplot as plt
import matplotlib.patches as patches
from matplotlib.patches import Wedge
import seaborn as sns
from Bio.SeqUtils import nt_search
import json
import base64
from io import BytesIO
import html
import os
from pathlib import Path
from matplotlib.backends.backend_pdf import PdfPages

# 新增导入：从main.py导入版本号
try:
    from main import __version__
except ImportError:
    # 如果无法导入，使用默认版本号
    __version__ = "Unknown Version"

class MotifVisualizer:
    def __init__(self, output_dir=".", max_motifs=5): 
        self.output_dir = Path(output_dir)
        self.motif_data = []
        self.sequence_mappings = []
        self.max_motifs = max_motifs
        
    def run(self, cmd_args=None):
        """运行完整的可视化流程 - 这是pipeline.py调用的方法"""
        print("Starting motif visualization...")
        
        try:
            # 加载数据
            pattern_file = self.output_dir / 'stem_patterns_analysis.csv'
            mapping_file = self.output_dir / 'motif_mapping_detailed.csv'
            
            if not pattern_file.exists():
                print(f"Error: Pattern file not found at {pattern_file}")
                return False
                
            self.load_data(str(pattern_file), str(mapping_file))
            
            # 创建HTML报告，传递命令行参数
            output_file = self.output_dir / 'motif_visualization_report.html'
            self.create_html_report(str(output_file), max_motifs=self.max_motifs, cmd_args=cmd_args)
            
            # 导出每个motif的PDF和统计信息
            self.export_motifs_to_pdf_and_csv()
            
            print(f"Visualization completed successfully! Report saved to {output_file}")
            return True
            
        except Exception as e:
            print(f"Error during visualization: {str(e)}")
            import traceback
            traceback.print_exc()
            return False

    def export_motifs_to_pdf_and_csv(self, high_quality=True):
        """导出每个motif的PDF和统计信息到Motif文件夹"""
        print("Exporting individual motifs to PDF and CSV...")
        
        # 创建Motif文件夹
        motif_dir = self.output_dir / "Motif"
        motif_dir.mkdir(exist_ok=True)
        
        # 获取要显示的motifs（与HTML报告相同的过滤逻辑）
        sorted_motifs = sorted(self.motif_data, key=lambda x: x['adjusted_p_value'])
        positive_enriched_motifs = []
        for motif in sorted_motifs:
            enrichment = motif.get('enrichment', 1)
            if enrichment > 1 or enrichment == float('inf'):
                positive_enriched_motifs.append(motif)
        
        if len(positive_enriched_motifs) > self.max_motifs:
            display_motifs = positive_enriched_motifs[:self.max_motifs]
        else:
            display_motifs = positive_enriched_motifs
        
        # 准备统计信息数据
        stats_data = []
        
        for i, motif_info in enumerate(display_motifs):
            motif_pattern = motif_info['pattern']
            print(f"Exporting motif {i+1}/{len(display_motifs)}: {motif_pattern}")
            
            try:
                # 计算位置频率
                positional_freq = self.calculate_positional_nucleotide_frequency(motif_pattern)
                
                # 分析结构特征
                structure_info = self.analyze_motif_characteristics(motif_pattern)
                
                # 导出PDF
                self.export_motif_to_pdf(motif_info, positional_freq, structure_info, i, motif_dir)
                
                # 导出PWM和MEME格式
                self.export_motif_formats(motif_info, positional_freq, i, motif_dir)
                
                # 收集统计信息
                motif_stats = {
                    'Motif_ID': f'Motif #{i+1}',
                    'Pattern': motif_pattern,
                    'Length': structure_info['length'],
                    'Stem_Regions': structure_info['stem_regions'],
                    'Paired_Bases': structure_info['paired_bases'],
                    'Unpaired_Bases': structure_info['unpaired_bases'],
                    'Paired_Ratio': structure_info['paired_ratio'],
                    'Motif_Type': structure_info['motif_type'],
                    'Enrichment': motif_info['enrichment'],
                    'Adjusted_p_value': motif_info['adjusted_p_value'],
                    'Positive_Frequency': motif_info['positive_frequency'],
                    'Negative_Frequency': motif_info['negative_frequency'],
                    'Positive_Count': motif_info['positive_count'],
                    'Negative_Count': motif_info['negative_count']
                }
                stats_data.append(motif_stats)
                
            except Exception as e:
                print(f"Error exporting motif {motif_pattern}: {e}")
                continue
        
        # 导出统计信息到CSV
        if stats_data:
            stats_df = pd.DataFrame(stats_data)
            csv_path = motif_dir / "motif_statistics.csv"
            stats_df.to_csv(csv_path, index=False)
            print(f"Motif statistics saved to {csv_path}")
        
        print(f"All motifs exported to {motif_dir}")

    def export_motif_formats(self, motif_info, positional_freq, index, output_dir):
        """导出motif的PWM和MEME格式文件"""
        motif_pattern = motif_info['pattern']
        motif_length = len(motif_pattern)
        
        # 导出PWM格式 (.txt)
        self.export_pwm_format(motif_info, positional_freq, index, output_dir)
        
        # 导出MEME格式 (.meme)
        self.export_meme_format(motif_info, positional_freq, index, output_dir)

    def export_pwm_format(self, motif_info, positional_freq, index, output_dir):
        """导出PWM (Position-specific weight matrix) 格式"""
        motif_pattern = motif_info['pattern']
        motif_length = len(motif_pattern)
        
        pwm_path = output_dir / f"Motif_{index+1}.pwm.txt"
        
        with open(pwm_path, 'w') as f:
            # 写入头部信息
            f.write(f"# Position-Specific Weight Matrix for Motif #{index+1}\n")
            f.write(f"# Pattern: {motif_pattern}\n")
            f.write(f"# Enrichment: {motif_info['enrichment']:.2f}x\n")
            f.write(f"# Adjusted p-value: {motif_info['adjusted_p_value']:.2e}\n")
            f.write(f"# Positive count: {motif_info['positive_count']}\n")
            f.write(f"# Negative count: {motif_info['negative_count']}\n")
            f.write("# Columns: Position A C G U\n")
            f.write("#\n")
            
            # 写入PWM矩阵
            f.write("Pos\tA\tC\tG\tU\n")
            for pos in range(motif_length):
                freq_dict = positional_freq.get(pos, {'A': 0.25, 'C': 0.25, 'G': 0.25, 'U': 0.25})
                # 转换为权重（通常使用log odds ratio，这里使用频率作为权重）
                a_freq = freq_dict.get('A', 0)
                c_freq = freq_dict.get('C', 0)
                g_freq = freq_dict.get('G', 0)
                u_freq = freq_dict.get('U', 0)
                
                f.write(f"{pos+1}\t{a_freq:.6f}\t{c_freq:.6f}\t{g_freq:.6f}\t{u_freq:.6f}\n")
        
        print(f"PWM format saved: {pwm_path}")

    def export_meme_format(self, motif_info, positional_freq, index, output_dir):
        """导出MEME motif格式"""
        motif_pattern = motif_info['pattern']
        motif_length = len(motif_pattern)
        
        meme_path = output_dir / f"Motif_{index+1}.meme"
        
        with open(meme_path, 'w') as f:
            # MEME格式头部
            f.write("MEME version 5\n")
            f.write("\n")
            f.write("ALPHABET= ACGU\n")
            f.write("\n")
            f.write("strands: +\n")
            f.write("\n")
            f.write("Background letter frequencies (from uniform background):\n")
            f.write("A 0.25 C 0.25 G 0.25 U 0.25\n")
            f.write("\n")
            
            # Motif定义
            f.write(f"MOTIF Motif_{index+1} {motif_pattern}\n")
            f.write(f"letter-probability matrix: alength= 4 w= {motif_length}\n")
            
            # 写入概率矩阵
            for pos in range(motif_length):
                freq_dict = positional_freq.get(pos, {'A': 0.25, 'C': 0.25, 'G': 0.25, 'U': 0.25})
                a_prob = freq_dict.get('A', 0.25)
                c_prob = freq_dict.get('C', 0.25)
                g_prob = freq_dict.get('G', 0.25)
                u_prob = freq_dict.get('U', 0.25)
                
                # 确保概率和为1（处理浮点误差）
                total = a_prob + c_prob + g_prob + u_prob
                if total > 0:
                    a_prob /= total
                    c_prob /= total
                    g_prob /= total
                    u_prob /= total
                
                f.write(f"  {a_prob:.6f}  {c_prob:.6f}  {g_prob:.6f}  {u_prob:.6f}\n")
        
        print(f"MEME format saved: {meme_path}")

    def export_motif_to_pdf(self, motif_info, positional_freq, structure_info, index, output_dir):
        """导出单个motif的结构图和频率logo到PDF"""
        
        motif_pattern = motif_info['pattern']

        self._current_motif_index = index
            
        # 创建PDF文件
        pdf_path = output_dir / f"Motif_{index+1}.pdf"
        
        with PdfPages(pdf_path) as pdf:
            # 创建图形 - 分为两个子图，使用正常尺寸
            fig, (ax_struct, ax_logo) = plt.subplots(2, 1, figsize=(12, 10))
            
            # 1. 绘制结构图
            self._draw_structure_diagram(ax_struct, motif_pattern, positional_freq, motif_info, structure_info)
            
            # 2. 绘制频率logo - 修改：移除位置编号
            self._draw_frequency_logo_pdf(ax_logo, positional_freq, motif_pattern)
            
            # 添加总体标题
            fig.suptitle(f'Motif #{index+1}: {motif_pattern}\n'
                        f'Enrichment: {motif_info["enrichment"]:.2f}x, '
                        f'p-value: {motif_info["adjusted_p_value"]:.2e}', 
                        fontsize=14, y=0.95)
            
            plt.tight_layout(rect=[0, 0, 1, 0.96])
            pdf.savefig(fig, dpi=300, bbox_inches='tight')
            plt.close(fig)
        
        print(f"PDF saved: {pdf_path}")

    def _draw_frequency_logo_pdf(self, ax, positional_freq, motif_pattern):
        """专门用于PDF输出的频率logo绘制，移除位置编号"""
        ax.set_title('Positional Nucleotide Frequency Logo', fontsize=12)
        
        try:
            import logomaker
            
            motif_length = len(motif_pattern)
            nt_order = ['A', 'C', 'G', 'U']
            
            # 创建频率矩阵 - 使用实际频率数据
            freq_matrix = np.zeros((motif_length, 4))
            for pos in range(motif_length):
                # 使用 positional_freq 中的实际频率
                freq_dict = positional_freq.get(pos, {'A': 0, 'C': 0, 'G': 0, 'U': 0})
                total = sum(freq_dict.values())
                if total > 0:
                    # 归一化频率
                    for j, nt in enumerate(nt_order):
                        freq_matrix[pos, j] = freq_dict.get(nt, 0) / total
                else:
                    # 如果没有数据，使用均匀分布
                    for j, nt in enumerate(nt_order):
                        freq_matrix[pos, j] = 0.25
            
            # 创建DataFrame和logo
            df_freq = pd.DataFrame(freq_matrix, columns=nt_order)
            logo = logomaker.Logo(df_freq, 
                                ax=ax,
                                color_scheme={'A': '#00CC00', 'C': '#0000CC', 
                                            'G': '#FFB300', 'U': '#CC0000'},
                                vpad=0.02,
                                width=0.8)
            
            # 设置样式
            ax.set_xlim(-0.5, motif_length - 0.5)
            ax.set_ylim(0, 1)
            ax.set_ylabel('Frequency')
            ax.set_xlabel('Position')
            ax.spines['top'].set_visible(False)
            ax.spines['right'].set_visible(False)
            
            # PDF版本：使用标准的matplotlib坐标轴标签，从1开始，间隔2个碱基
            positions = list(range(1, motif_length + 1))
            # 选择要显示的标签位置：从1开始，每2个位置显示一个标签
            display_positions = [i for i in range(motif_length) if (i + 1) % 2 == 1 or i == 0 or i == motif_length - 1]
            display_labels = [str(i + 1) for i in display_positions]
            
            ax.set_xticks(display_positions)
            ax.set_xticklabels(display_labels)
            
            # 新增：在横坐标下方添加stem-loop结构标记
            self._add_structure_annotation(ax, motif_pattern, motif_length)
                
        except ImportError:
            print("Logomaker not available, using fallback visualization")
            self._draw_simple_logo(ax, positional_freq, motif_pattern,  # 传入 motif_pattern
                                {'A': '#00CC00', 'C': '#0000CC', 'G': '#FFB300', 'U': '#CC0000'})
        except Exception as e:
            print(f"Error in logomaker generation: {e}")
            self._draw_simple_logo(ax, positional_freq, motif_pattern,  # 传入 motif_pattern
                                {'A': '#00CC00', 'C': '#0000CC', 'G': '#FFB300', 'U': '#CC0000'})

    def _add_structure_annotation(self, ax, motif_pattern, motif_length):
        """在logo下方添加stem-loop结构标记"""
        # 调整坐标轴位置，为结构标记留出空间
        ax.set_position([ax.get_position().x0, 
                        ax.get_position().y0 + 0.05,  # 为结构标记留出空间
                        ax.get_position().width, 
                        ax.get_position().height - 0.05])
        
        # 在横坐标下方添加结构标记
        y_bottom = -0.25  # 在横坐标标签下方
        
        # 绘制结构标记线
        ax.axhline(y=-0.15, color='gray', linewidth=0.5, alpha=0.7)
        
        # 添加结构符号
        for pos in range(motif_length):
            char = motif_pattern[pos]
            color = 'black'
            
            # 根据结构字符设置颜色和样式
            if char == '(':
                color = 'red'
                symbol = '('  
            elif char == ')':
                color = 'red'  
                symbol = ')'  
            elif char == '.':
                color = 'blue'
                symbol = '•'   # 圆点表示未配对
            else:
                color = 'gray'
                symbol = char  # 其他字符直接显示
                
            # 在对应位置添加结构标记
            ax.text(pos, y_bottom, symbol, 
                    ha='center', va='center', 
                    fontsize=10, color=color, fontweight='bold')  # 增大字体到10
        
        # 添加结构标记说明
        ax.text(-0.5, y_bottom - 0.1, 'Structure:', 
                ha='right', va='center', fontsize=7, color='gray', style='italic')
        
        # 调整Y轴限制以包含结构标记
        current_ylim = ax.get_ylim()
        ax.set_ylim(current_ylim[0], current_ylim[1] + 0.1)  # 为结构标记留出空间

    def _draw_simple_logo_pdf(self, ax, positional_freq, motif_pattern, colors):
        """PDF版本的简单logo绘制"""
        motif_length = len(motif_pattern)
        ax.clear()
        ax.set_xlim(0, motif_length)
        ax.set_ylim(0, 2)
        ax.set_xlabel('Position')
        ax.set_ylabel('Bits')
        ax.set_title('Positional Nucleotide Frequency (Simple Logo)')
        ax.grid(True, alpha=0.3)
        
        # 计算每个位置的信息量
        for pos in range(motif_length):
            freq_dict = positional_freq.get(pos, {'A': 0.25, 'C': 0.25, 'G': 0.25, 'U': 0.25})
            
            # 计算信息量 (2 - entropy)
            import math
            entropy = 0
            for freq in freq_dict.values():
                if freq > 0:
                    entropy -= freq * math.log2(freq)
            information = 2 - entropy
            
            # 按频率排序碱基
            sorted_nts = sorted(freq_dict.items(), key=lambda x: x[1], reverse=True)
            
            # 从底部开始堆叠
            current_height = 0
            for nt, freq in sorted_nts:
                if freq > 0:
                    height = information * freq
                    # 绘制矩形
                    rect = patches.Rectangle(
                        (pos + 0.1, current_height), 0.8, height,
                        facecolor=colors[nt], edgecolor='black', linewidth=0.5, alpha=0.8
                    )
                    ax.add_patch(rect)
                    
                    # 如果高度足够，添加碱基标签
                    if height > 0.2:
                        ax.text(pos + 0.5, current_height + height/2, nt,
                            ha='center', va='center', fontweight='bold', fontsize=8)
                    
                    current_height += height
        
        # PDF版本：使用标准坐标轴标签，从1开始，间隔2个碱基
        display_positions = [i + 0.5 for i in range(motif_length) if (i + 1) % 2 == 1 or i == 0 or i == motif_length - 1]
        display_labels = [str(i + 1) for i in range(motif_length) if (i + 1) % 2 == 1 or i == 0 or i == motif_length - 1]
        
        ax.set_xticks(display_positions)
        ax.set_xticklabels(display_labels)
        
        # 新增：在横坐标下方添加stem-loop结构标记
        self._add_structure_annotation_simple(ax, motif_pattern, motif_length)

    def _add_structure_annotation_simple(self, ax, motif_pattern, motif_length):
        """简单logo版本的结构标记"""
        # 在横坐标下方添加结构标记
        y_bottom = -0.3  # 在横坐标标签下方
        
        # 绘制结构标记线
        ax.axhline(y=-0.2, color='gray', linewidth=0.5, alpha=0.7)
        
        # 添加结构符号
        for pos in range(motif_length):
            char = motif_pattern[pos]
            color = 'black'
            
            # 根据结构字符设置颜色和样式
            if char == '(':
                color = 'red'
                symbol = '('  
            elif char == ')':
                color = 'red'  
                symbol = ')' 
            elif char == '.':
                color = 'blue'
                symbol = '•'   # 圆点表示未配对
            else:
                color = 'gray'
                symbol = char  # 其他字符直接显示
                
            # 在对应位置添加结构标记
            ax.text(pos + 0.5, y_bottom, symbol, 
                    ha='center', va='center', 
                    fontsize=10, color=color, fontweight='bold')  # 增大字体到10
        
        # 添加结构标记说明
        ax.text(-0.2, y_bottom - 0.1, 'Structure:', 
                ha='right', va='center', fontsize=7, color='gray', style='italic')
        
        # 调整Y轴限制以包含结构标记
        current_ylim = ax.get_ylim()
        ax.set_ylim(current_ylim[0], current_ylim[1] + 0.15)  # 为结构标记留出空间


    def _draw_structure_diagram(self, ax, motif_pattern, positional_freq, motif_info, structure_info):
        """绘制结构图到指定的axes"""
        colors = {'A': '#00CC00', 'C': '#0000CC', 'G': '#FFB300', 'U': '#CC0000', 'N': '#CCCCCC'}
        
        ax.set_title('Stem-loop Structure Diagram', fontsize=12, pad=15)  # 增加pad值
        
        try:
            import RNA
            
            # 生成序列和结构
            sequence = ""
            base_colors = []
            dominant_bases = []
            
            for pos in range(len(motif_pattern)):
                if motif_pattern[pos] in ['(', '.', ')']:
                    freq_dict = positional_freq.get(pos, {'A': 0.25, 'C': 0.25, 'G': 0.25, 'U': 0.25})
                    max_nt = max(freq_dict.items(), key=lambda x: x[1])
                    if max_nt[1] > 0.3:  # 降低阈值，让更多碱基显示实际值
                        sequence += max_nt[0]
                        base_colors.append(colors[max_nt[0]])
                        dominant_bases.append(max_nt[0])
                    else:
                        sequence += 'N'
                        base_colors.append(colors['N'])
                        dominant_bases.append('N')
                else:
                    sequence += motif_pattern[pos]
                    base_colors.append(colors.get(motif_pattern[pos], '#CCCCCC'))
                    dominant_bases.append(motif_pattern[pos])
            
            dot_bracket = motif_pattern
            
            # 尝试使用ViennaRNA绘制
            try:
                # 创建Motif文件夹
                motif_dir = self.output_dir / "Motif"
                motif_dir.mkdir(exist_ok=True)
                
                # 直接保存到Motif文件夹，使用motif_info中的信息来命名
                motif_index = getattr(self, '_current_motif_index', 0) + 1
                svg_filename = f"Motif_{motif_index}.svg"
                svg_path = motif_dir / svg_filename
                
                RNA.svg_rna_plot(sequence, dot_bracket, str(svg_path))
                
                try:
                    import cairosvg
                    png_data = cairosvg.svg2png(url=str(svg_path))
                    img_buffer = BytesIO(png_data)
                    img = plt.imread(img_buffer)
                    ax.imshow(img)
                    
                    # 关键修改：删除位置序号标记
                    # 移除以下代码：
                    # for pos in range(motif_length):
                    #     # 在图像底部添加位置编号
                    #     x_pos = (pos + 0.5) / motif_length
                    #     ax.text(x_pos, -0.05, str(pos+1), 
                    #         transform=ax.transAxes, 
                    #         ha='center', va='top', 
                    #         fontsize=6, color='gray',
                    #         bbox=dict(boxstyle="circle,pad=0.1", fc='white', ec='gray', lw=0.5))
                    
                    ax.axis('off')
                    
                    print(f"SVG structure saved: {svg_path}")
                    
                except ImportError:
                    # 如果cairosvg不可用，显示SVG文件路径信息
                    ax.text(0.5, 0.5, f"SVG file saved at:\n{svg_path}", 
                            ha='center', va='center', transform=ax.transAxes)
                    ax.axis('off')
                
                # 注意：这里不再删除SVG文件，而是保留它
                    
            except Exception as e:
                print(f"Error using ViennaRNA SVG plot: {e}")
                # 使用备用的简单可视化
                self._draw_simple_structure(ax, motif_pattern, positional_freq, colors)
                ax.set_title(f'Stem-loop Structure (Simple): {motif_pattern}', fontsize=10)
                ax.axis('off')
                
        except ImportError:
            print("ViennaRNA not available, using simple structure visualization")
            self._draw_simple_structure(ax, motif_pattern, positional_freq, colors)
            ax.axis('off')

    def _draw_frequency_logo(self, ax, positional_freq, motif_pattern):
        """绘制频率logo到指定的axes - 使用实际频率数据"""
        ax.set_title('Positional Nucleotide Frequency Logo', fontsize=12)
        
        try:
            import logomaker
            
            motif_length = len(motif_pattern)
            nt_order = ['A', 'C', 'G', 'U']
            
            # 创建频率矩阵 - 使用实际频率数据
            freq_matrix = np.zeros((motif_length, 4))
            for pos in range(motif_length):
                # 使用 positional_freq 中的实际频率
                freq_dict = positional_freq.get(pos, {'A': 0, 'C': 0, 'G': 0, 'U': 0})
                total = sum(freq_dict.values())
                if total > 0:
                    # 归一化频率
                    for j, nt in enumerate(nt_order):
                        freq_matrix[pos, j] = freq_dict.get(nt, 0) / total
                else:
                    # 如果没有数据，使用均匀分布
                    for j, nt in enumerate(nt_order):
                        freq_matrix[pos, j] = 0.25
            
            # 创建DataFrame和logo - 减小vpad来减少纵向缝隙
            df_freq = pd.DataFrame(freq_matrix, columns=nt_order)
            logo = logomaker.Logo(df_freq, 
                                ax=ax,
                                color_scheme={'A': '#00CC00', 'C': '#0000CC', 
                                            'G': '#FFB300', 'U': '#CC0000'},
                                vpad=0.02,  # 减小纵向填充，从0.1改为0.02
                                width=0.8)
            
            # 设置样式
            ax.set_xlim(-0.5, motif_length - 0.5)
            ax.set_ylim(0, 1)
            ax.set_ylabel('Frequency')
            ax.set_xlabel('Position')
            ax.spines['top'].set_visible(False)
            ax.spines['right'].set_visible(False)
            
            # 修改：优化横坐标标签，从1开始，间隔2个碱基标记一次
            positions = list(range(1, motif_length + 1))
            # 选择要显示的标签位置：从1开始，每2个位置显示一个标签
            display_positions = [i for i in range(motif_length) if (i + 1) % 2 == 1 or i == 0 or i == motif_length - 1]
            display_labels = [str(i + 1) for i in display_positions]
            
            ax.set_xticks(display_positions)
            ax.set_xticklabels(display_labels)
            
            # 新增：在横坐标下方添加stem-loop结构标记
            self._add_structure_annotation(ax, motif_pattern, motif_length)
                
        except ImportError:
            print("Logomaker not available, using fallback visualization")
            self._draw_simple_logo(ax, positional_freq, motif_pattern,  # 传入 motif_pattern
                                {'A': '#00CC00', 'C': '#0000CC', 'G': '#FFB300', 'U': '#CC0000'})
        except Exception as e:
            print(f"Error in logomaker generation: {e}")
            self._draw_simple_logo(ax, positional_freq, motif_pattern,  # 传入 motif_pattern
                                {'A': '#00CC00', 'C': '#0000CC', 'G': '#FFB300', 'U': '#CC0000'})
                    
    def _draw_simple_structure(self, ax, motif_pattern, positional_freq, colors):
        """绘制简化的结构图作为备用"""
        ax.clear()
        
        # 简化的圆形布局
        n = len(motif_pattern)
        angles = np.linspace(0, 2 * np.pi, n, endpoint=False)
        radius = 5
        center_x, center_y = 0, 0
        
        x_coords = center_x + radius * np.cos(angles)
        y_coords = center_y + radius * np.sin(angles)
        
        # 绘制骨架
        for i in range(n):
            next_i = (i + 1) % n
            ax.plot([x_coords[i], x_coords[next_i]], [y_coords[i], y_coords[next_i]], 
                   'k-', alpha=0.5, linewidth=1)
        
        # 绘制碱基配对
        stack = []
        base_radius = 0.4
        
        for i, char in enumerate(motif_pattern):
            if char == '(':
                stack.append(i)
            elif char == ')':
                if stack:
                    j = stack.pop()
                    # 绘制配对线
                    ax.plot([x_coords[i], x_coords[j]], [y_coords[i], y_coords[j]], 
                           color='blue', linewidth=2, alpha=0.7)
        
        # 绘制碱基
        dominant_bases = []
        for pos in range(len(motif_pattern)):
            freq_dict = positional_freq.get(pos, {'A': 0.25, 'C': 0.25, 'G': 0.25, 'U': 0.25})
            max_nt = max(freq_dict.items(), key=lambda x: x[1])
            dominant_bases.append(max_nt[0] if max_nt[1] > 0.5 else 'N')
        
        for i, (x, y) in enumerate(zip(x_coords, y_coords)):
            # 绘制碱基圆圈
            circle = plt.Circle((x, y), base_radius, 
                            facecolor=colors.get(dominant_bases[i], '#CCCCCC'), 
                            alpha=0.8, 
                            edgecolor='black', 
                            linewidth=1)
            ax.add_patch(circle)
            
            # 添加碱基标签
            ax.text(x, y, dominant_bases[i], 
                   ha='center', va='center', 
                   fontsize=8, fontweight='bold')
            
            # 添加位置编号
            ax.text(x, y - base_radius - 0.2, str(i+1), 
                   ha='center', va='top', 
                   fontsize=6, color='gray')
        
        # 设置坐标范围
        padding = 1
        ax.set_xlim(min(x_coords) - base_radius - padding, max(x_coords) + base_radius + padding)
        ax.set_ylim(min(y_coords) - base_radius - padding, max(y_coords) + base_radius + padding)
        ax.set_aspect('equal')
        
    def load_data(self, pattern_file, mapping_file):
        """加载阶段3和阶段4的数据"""
        print("Loading motif data...")
        
        try:
            # 加载模式数据
            pattern_df = pd.read_csv(pattern_file)
            self.motif_data = pattern_df.to_dict('records')
            
            # 尝试加载映射数据（如果存在）
            if os.path.exists(mapping_file):
                mapping_df = pd.read_csv(mapping_file)
                self.sequence_mappings = mapping_df.to_dict('records')
                print(f"Loaded {len(self.motif_data)} motifs and {len(self.sequence_mappings)} mappings")
            else:
                print(f"Warning: Mapping file {mapping_file} not found. Using motif data only.")
                print(f"Loaded {len(self.motif_data)} motifs")
                
        except Exception as e:
            print(f"Error loading data: {str(e)}")
            raise

    def calculate_positional_nucleotide_frequency(self, motif_pattern):
        """计算每个位置的核苷酸频率 - 使用Step3的准确计数"""
        # 获取匹配该motif的所有阳性序列区域
        positive_matches = [
            m for m in self.sequence_mappings 
            if m['motif_pattern'] == motif_pattern and m['label'] == 'positive'
        ]
        
        # 如果Step4没有找到匹配，但Step3有统计，使用均匀分布
        if not positive_matches:
            motif_info = next((m for m in self.motif_data if m['pattern'] == motif_pattern), None)
            if motif_info and motif_info['positive_count'] > 0:
                motif_length = len(motif_pattern)
                return {i: {'A': 0.25, 'C': 0.25, 'G': 0.25, 'U': 0.25} for i in range(motif_length)}
            else:
                return {i: {'A': 0.25, 'C': 0.25, 'G': 0.25, 'U': 0.25} for i in range(len(motif_pattern))}
        
        # 统计每个位置的核苷酸频率
        position_freq = {}
        motif_length = len(motif_pattern)
        
        for pos in range(motif_length):
            nucleotide_counts = {'A': 0, 'C': 0, 'G': 0, 'U': 0}
            total_sequences = 0
            
            for match in positive_matches:
                if pos < len(match['sequence_region']):
                    nucleotide = match['sequence_region'][pos].upper()
                    if nucleotide in nucleotide_counts:
                        nucleotide_counts[nucleotide] += 1
                        total_sequences += 1
            
            # 计算频率
            if total_sequences > 0:
                position_freq[pos] = {
                    nt: count / total_sequences for nt, count in nucleotide_counts.items()
                }
            else:
                position_freq[pos] = {'A': 0.25, 'C': 0.25, 'G': 0.25, 'U': 0.25}
        
        return position_freq
    
    def create_stem_loop_diagram(self, motif_pattern, positional_freq, motif_info, figsize=(8, 8)):
        """使用ViennaRNA Python绑定创建stem-loop结构图"""
        import RNA
        
        # 创建图形，调整高度比例 - 修改：进一步缩小图形
        fig = plt.figure(figsize=(figsize[0], figsize[1]))
        gs = plt.GridSpec(2, 1, height_ratios=[2, 1])  # 调整比例
        ax_struct = fig.add_subplot(gs[0])
        ax_seq = fig.add_subplot(gs[1])

        # 设置颜色
        colors = {'A': '#00CC00', 'C': '#0000CC', 'G': '#FFB300', 'U': '#CC0000', 'N': '#CCCCCC'}
        
        # 1. 使用ViennaRNA绘制结构图 - 修改：缩小字体
        ax_struct.set_title(f'Stem-loop Structure: {motif_pattern}\n'
                        f'Enrichment: {motif_info["enrichment"]:.2f}x, '
                        f'p-value: {motif_info["adjusted_p_value"]:.2e}', 
                        fontsize=10, pad=20)  # 缩小字体
        
        # 根据点括号表示法生成碱基序列
        sequence = ""
        base_colors = []  # 存储每个位置的碱基颜色
        dominant_bases = []  # 存储每个位置的优势碱基
        
        for pos in range(len(motif_pattern)):
            if motif_pattern[pos] in ['(', '.', ')']:
                # 结构字符，需要生成对应的碱基序列
                freq_dict = positional_freq.get(pos, {'A': 0.25, 'C': 0.25, 'G': 0.25, 'U': 0.25})
                max_nt = max(freq_dict.items(), key=lambda x: x[1])
                if max_nt[1] > 0.3:  # 降低阈值，让更多碱基显示实际值
                    sequence += max_nt[0]  # 使用频率最高的碱基
                    base_colors.append(colors[max_nt[0]])
                    dominant_bases.append(max_nt[0])
                else:
                    sequence += 'N'  # 没有明显优势碱基
                    base_colors.append(colors['N'])
                    dominant_bases.append('N')
            else:
                sequence += motif_pattern[pos]  # 保持原始字符
                base_colors.append(colors.get(motif_pattern[pos], '#CCCCCC'))
                dominant_bases.append(motif_pattern[pos])
        
        # 使用点括号表示法作为结构
        dot_bracket = motif_pattern
        
        try:
            # 使用ViennaRNA的svg_rna_plot函数来生成结构图
            # 创建临时文件
            import tempfile
            import os
            
            with tempfile.NamedTemporaryFile(suffix='.svg', delete=False) as temp_file:
                svg_path = temp_file.name
            
            # 使用RNAplot的SVG输出
            RNA.svg_rna_plot(sequence, dot_bracket, svg_path)
            
            # 读取SVG文件并转换为matplotlib图像
            try:
                import cairosvg
                from io import BytesIO
                
                # 将SVG转换为PNG
                png_data = cairosvg.svg2png(url=svg_path)
                img_buffer = BytesIO(png_data)
                img = plt.imread(img_buffer)
                
                # 显示图像 - 修改：调整图像显示
                ax_struct.imshow(img)
                ax_struct.axis('off')
                
                print(f"SVG structure saved: {svg_path}")
                
            except ImportError:
                # 如果cairosvg不可用，显示SVG文件路径信息
                ax_struct.text(0.5, 0.5, f"SVG file generated at:\n{svg_path}", 
                            ha='center', va='center', transform=ax_struct.transAxes, fontsize=8)
                ax_struct.axis('off')
            
            # 清理临时文件
            try:
                os.unlink(svg_path)
            except:
                pass
            
        except Exception as e:
            print(f"Error using ViennaRNA SVG plot: {e}")
            try:
                # 尝试使用备用的坐标计算方法
                coords = RNA.get_xy_coordinates(dot_bracket)
                
                # 处理单个COORDINATE对象的情况
                if hasattr(coords, 'x') and hasattr(coords, 'y'):
                    # 如果是单个坐标对象，创建包含所有坐标的列表
                    x_coords = []
                    y_coords = []
                    
                    # 使用配对表来计算所有坐标
                    pair_table = RNA.ptable(dot_bracket)
                    n = len(dot_bracket)
                    
                    # 简化的圆形布局作为备用
                    angles = np.linspace(0, 2 * np.pi, n, endpoint=False)
                    radius = 4  # 缩小半径
                    center_x, center_y = 0, 0
                    
                    x_coords = center_x + radius * np.cos(angles)
                    y_coords = center_y + radius * np.sin(angles)
                    
                    # 绘制结构
                    self._draw_coordinates_structure(ax_struct, x_coords, y_coords, 
                                                dot_bracket, base_colors, dominant_bases)
                    
                else:
                    # 如果是坐标列表
                    x_coords = [coord.x for coord in coords]
                    y_coords = [coord.y for coord in coords]
                    self._draw_coordinates_structure(ax_struct, x_coords, y_coords,
                                                dot_bracket, base_colors, dominant_bases)
                    
            except Exception as e2:
                print(f"Error in coordinate-based fallback: {e2}")
                # 使用最终的简单可视化
                ax_struct.clear()
                self._draw_simple_structure(ax_struct, motif_pattern, positional_freq, colors)
                ax_struct.set_title(f'Stem-loop Structure (Fallback): {motif_pattern}\n'
                                f'Enrichment: {motif_info["enrichment"]:.2f}x', 
                                fontsize=8, pad=15)  # 缩小字体
                ax_struct.axis('off')
        
        # 2. 使用logomaker绘制碱基频率motif - 修改：缩小字体
        ax_seq.set_title('Positional Nucleotide Frequency Logo', fontsize=9)
        
        try:
            import logomaker
            
            # 准备频率矩阵 - 使用实际频率数据
            motif_length = len(motif_pattern)
            nt_order = ['A', 'C', 'G', 'U']
            
            # 创建频率矩阵 - 使用positional_freq中的实际数据
            freq_matrix = np.zeros((motif_length, 4))
            for pos in range(motif_length):
                # 使用 positional_freq 中的实际频率
                freq_dict = positional_freq.get(pos, {'A': 0, 'C': 0, 'G': 0, 'U': 0})
                total = sum(freq_dict.values())
                if total > 0:
                    # 归一化频率
                    for j, nt in enumerate(nt_order):
                        freq_matrix[pos, j] = freq_dict.get(nt, 0) / total
                else:
                    # 如果没有数据，使用均匀分布
                    for j, nt in enumerate(nt_order):
                        freq_matrix[pos, j] = 0.25
            
            # 创建DataFrame
            df_freq = pd.DataFrame(freq_matrix, columns=nt_order)
            
            # 创建logo - 减小vpad来减少纵向缝隙
            logo = logomaker.Logo(df_freq, 
                                ax=ax_seq,
                                color_scheme={'A': '#00CC00', 'C': '#0000CC', 
                                            'G': '#FFB300', 'U': '#CC0000'},
                                vpad=0.02,
                                width=0.8)
            
            # 设置样式
            ax_seq.set_xlim(-0.5, motif_length - 0.5)
            ax_seq.set_ylim(0, 1)
            ax_seq.set_ylabel('Frequency', fontsize=9)
            ax_seq.set_xlabel('Position', fontsize=9)
            ax_seq.spines['top'].set_visible(False)
            ax_seq.spines['right'].set_visible(False)
            
            # 添加位置标签 - 从1开始，间隔2个碱基
            display_positions = [i for i in range(motif_length) if (i + 1) % 2 == 1 or i == 0 or i == motif_length - 1]
            display_labels = [str(i + 1) for i in display_positions]
            
            ax_seq.set_xticks(display_positions)
            ax_seq.set_xticklabels(display_labels)
            ax_seq.tick_params(axis='both', which='major', labelsize=8)
            
            # 新增：在横坐标下方添加stem-loop结构标记
            self._add_structure_annotation(ax_seq, motif_pattern, motif_length)
                
        except ImportError:
            print("Logomaker not available, using fallback visualization")
            # 如果logomaker不可用，使用备用的简单可视化
            self._draw_simple_logo(ax_seq, positional_freq, motif_pattern, colors)
        except Exception as e:
            print(f"Error in logomaker generation: {e}")
            # 出错时使用备用可视化
            self._draw_simple_logo(ax_seq, positional_freq, motif_pattern, colors)
        
        plt.tight_layout()
        return fig
    
    def _draw_simple_logo(self, ax, positional_freq, motif_pattern, colors):
        """手动绘制简化的序列logo作为备用"""
        motif_length = len(motif_pattern)
        ax.clear()
        ax.set_xlim(0, motif_length)
        ax.set_ylim(0, 2)  # 最大信息量为2 bits
        ax.set_xlabel('Position')
        ax.set_ylabel('Bits')
        ax.set_title('Positional Nucleotide Frequency (Simple Logo)')
        ax.grid(True, alpha=0.3)
        
        # 计算每个位置的信息量
        for pos in range(motif_length):
            freq_dict = positional_freq.get(pos, {'A': 0.25, 'C': 0.25, 'G': 0.25, 'U': 0.25})
            
            # 计算信息量 (2 - entropy)
            import math
            entropy = 0
            for freq in freq_dict.values():
                if freq > 0:
                    entropy -= freq * math.log2(freq)
            information = 2 - entropy  # 最大为2 bits
            
            # 按频率排序碱基
            sorted_nts = sorted(freq_dict.items(), key=lambda x: x[1], reverse=True)
            
            # 从底部开始堆叠
            current_height = 0
            for nt, freq in sorted_nts:
                if freq > 0:
                    height = information * freq
                    # 绘制矩形
                    rect = patches.Rectangle(
                        (pos + 0.1, current_height), 0.8, height,
                        facecolor=colors[nt], edgecolor='black', linewidth=0.5, alpha=0.8
                    )
                    ax.add_patch(rect)
                    
                    # 如果高度足够，添加碱基标签
                    if height > 0.2:
                        ax.text(pos + 0.5, current_height + height/2, nt,
                            ha='center', va='center', fontweight='bold', fontsize=8)
                    
                    current_height += height
        
        # 设置横坐标标签
        display_positions = [i + 0.5 for i in range(motif_length) if (i + 1) % 2 == 1 or i == 0 or i == motif_length - 1]
        display_labels = [str(i + 1) for i in range(motif_length) if (i + 1) % 2 == 1 or i == 0 or i == motif_length - 1]
        
        ax.set_xticks(display_positions)
        ax.set_xticklabels(display_labels)
        
        # 新增：在横坐标下方添加stem-loop结构标记
        self._add_structure_annotation_simple(ax, motif_pattern, motif_length)
            
    def _draw_coordinates_structure(self, ax, x_coords, y_coords, dot_bracket, base_colors, dominant_bases):
        """根据坐标绘制RNA结构"""
        import RNA
        
        # 设置坐标范围
        padding = 2
        x_min, x_max = min(x_coords) - padding, max(x_coords) + padding
        y_min, y_max = min(y_coords) - padding, max(y_coords) + padding
        
        ax.set_xlim(x_min, x_max)
        ax.set_ylim(y_min, y_max)
        ax.set_aspect('equal')
        ax.axis('off')
        
        # 绘制碱基配对线
        pair_table = RNA.ptable(dot_bracket)
        for i in range(1, len(pair_table)):
            j = pair_table[i]
            if j > i:  # 有效的碱基配对
                x1, y1 = x_coords[i-1], y_coords[i-1]
                x2, y2 = x_coords[j-1], y_coords[j-1]
                
                # 绘制配对线
                ax.plot([x1, x2], [y1, y2], 
                    color='#2E86AB', linewidth=2, alpha=0.7, zorder=1)
        
        # 绘制碱基圆圈和标签
        base_radius = 0.3
        for i, (x, y) in enumerate(zip(x_coords, y_coords)):
            # 绘制碱基圆圈
            circle = plt.Circle((x, y), base_radius, 
                            facecolor=base_colors[i], 
                            alpha=0.8, 
                            edgecolor='black', 
                            linewidth=1,
                            zorder=2)
            ax.add_patch(circle)
            
            # 添加碱基标签
            ax.text(x, y, dominant_bases[i], 
                ha='center', va='center', 
                fontsize=8, fontweight='bold',
                zorder=3)
            
            # 添加位置编号
            ax.text(x, y - base_radius - 0.1, str(i+1), 
                ha='center', va='top', 
                fontsize=6, color='gray')
        
        # 绘制骨架连线
        for i in range(len(x_coords) - 1):
            x1, y1 = x_coords[i], y_coords[i]
            x2, y2 = x_coords[i+1], y_coords[i+1]
            
            # 计算方向向量
            dx, dy = x2 - x1, y2 - y1
            length = np.sqrt(dx**2 + dy**2)
            
            if length > 0:
                # 归一化
                dx, dy = dx/length, dy/length
                
                # 调整起点和终点以避免与碱基圆圈重叠
                start_x = x1 + dx * base_radius
                start_y = y1 + dy * base_radius
                end_x = x2 - dx * base_radius
                end_y = y2 - dy * base_radius
                
                ax.plot([start_x, end_x], [start_y, end_y], 
                    color='black', linewidth=1, alpha=0.5, zorder=1)


    def analyze_motif_characteristics(self, motif_pattern):
        """分析motif的结构特征"""
        length = len(motif_pattern)
        stem_regions = motif_pattern.count('(')  # 假设对称
        paired_bases = motif_pattern.count('(') + motif_pattern.count(')')
        unpaired_bases = motif_pattern.count('.')
        paired_ratio = paired_bases / length if length > 0 else 0
        
        # 判断motif类型
        if stem_regions >= 6:  
            motif_type = "Long Stem"
        elif stem_regions >= 4:  
            motif_type = "Medium Stem"
        elif stem_regions >= 2:  
            motif_type = "Short Stem"
        elif stem_regions == 1:
            motif_type = "Single BP"
        elif unpaired_bases >= length * 0.7:  
            motif_type = "Loop-rich"
        else:
            motif_type = "Complex"
        
        return {
            'length': length,
            'stem_regions': stem_regions,
            'paired_bases': paired_bases,
            'unpaired_bases': unpaired_bases,
            'paired_ratio': paired_ratio,
            'motif_type': motif_type
        }
    
    def create_motif_card(self, motif_info, index):
        """创建单个motif的信息卡片 - 使用Step4的映射计数"""
        motif_pattern = motif_info['pattern']
        
        # 计算位置频率
        positional_freq = self.calculate_positional_nucleotide_frequency(motif_pattern)
        
        # 分析结构特征
        structure_info = self.analyze_motif_characteristics(motif_pattern)
        
        # 创建图形 - 修改：缩小图形尺寸
        fig = self.create_stem_loop_diagram(motif_pattern, positional_freq, motif_info, figsize=(8, 8))  # 缩小尺寸
        
        # 转换图形为HTML
        img_buffer = BytesIO()
        fig.savefig(img_buffer, format='png', dpi=120, bbox_inches='tight')  # 降低DPI
        img_buffer.seek(0)
        img_data = base64.b64encode(img_buffer.getvalue()).decode()
        plt.close(fig)
        
        # 获取Step4的映射计数
        mapped_positive_count = self.get_mapped_positive_count(motif_pattern)
        mapped_negative_count = self.get_mapped_negative_count(motif_pattern)
        
        # 计算映射百分比（相对于输入序列总数）
        total_positive_seqs = self.get_total_positive_sequences()
        total_negative_seqs = self.get_total_negative_sequences()
        
        mapped_positive_percentage = (mapped_positive_count / total_positive_seqs * 100) if total_positive_seqs > 0 else 0
        mapped_negative_percentage = (mapped_negative_count / total_negative_seqs * 100) if total_negative_seqs > 0 else 0
        
        # 使用Step3中的准确计数作为参考
        step3_positive_count = motif_info.get('positive_count', 0)
        step3_negative_count = motif_info.get('negative_count', 0)
        

        # 创建HTML卡片 - 修改：完全删除结构标注部分
        card_html = f"""
        <div class="motif-card">
            <div class="motif-header">
                <h3>Motif #{index + 1}: {html.escape(motif_pattern)}</h3>
                <div class="motif-stats">
                    <span class="stat-badge enrichment">Enrichment: {motif_info['enrichment']:.2f}x</span>
                    <span class="stat-badge p-value">p-value: {motif_info['adjusted_p_value']:.2e}</span>
                    <span class="stat-badge length">Length: {structure_info['length']}</span>
                    <span class="stat-badge type">{structure_info['motif_type']}</span>
                </div>
            </div>
            
            <div class="motif-content">
                <div class="motif-image">
                    <img src="data:image/png;base64,{img_data}" alt="Motif structure" style="max-width: 80%; margin: 10px 0;">
                    <!-- 完全删除结构标注部分 -->
                </div>
                
                <div class="motif-details">
                    <table class="motif-table">
                        <tr><th>Characteristic</th><th>Value</th></tr>
                        <tr><td>Length</td><td>{structure_info['length']}</td></tr>
                        <tr><td>Stem Regions</td><td>{structure_info['stem_regions']}</td></tr>
                        <tr><td>Paired Bases</td><td>{structure_info['paired_bases']}</td></tr>
                        <tr><td>Unpaired Bases</td><td>{structure_info['unpaired_bases']}</td></tr>
                        <tr><td>Paired Ratio</td><td>{structure_info['paired_ratio']:.3f}</td></tr>
                        <tr><td>Motif Type</td><td>{structure_info['motif_type']}</td></tr>
                        <tr><td>Enrichment</td><td>{motif_info['enrichment']:.2f}x</td></tr>
                        <tr><td>Adjusted p-value</td><td>{motif_info['adjusted_p_value']:.2e}</td></tr>
                        <tr><td>Step3 Positive Count</td><td>{step3_positive_count}</td></tr>
                        <tr><td>Step3 Negative Count</td><td>{step3_negative_count}</td></tr>
                        <tr><td>Mapped Motif Count</td><td>{mapped_positive_count}</td></tr>
                        <tr><td>Mapped Motif %</td><td>{mapped_positive_percentage:.1f}%</td></tr>
                    </table>
                </div>
            </div>
        </div>
        """
        
        return card_html
    


    def get_mapped_positive_count(self, motif_pattern):
        """获取Step4中映射到的阳性序列计数"""
        if not hasattr(self, '_mapped_counts_initialized'):
            self._initialize_mapped_counts()
        
        return self.mapped_positive_counts.get(motif_pattern, 0)

    def get_mapped_negative_count(self, motif_pattern):
        """获取Step4中映射到的阴性序列计数"""
        if not hasattr(self, '_mapped_counts_initialized'):
            self._initialize_mapped_counts()
        
        return self.mapped_negative_counts.get(motif_pattern, 0)

    def _initialize_mapped_counts(self):
        """初始化映射计数缓存"""
        self.mapped_positive_counts = {}
        self.mapped_negative_counts = {}
        
        # 从Step4的映射文件中读取计数
        mapping_file = self.output_dir / 'motif_mapping_detailed.csv'
        if mapping_file.exists():
            try:
                mapping_df = pd.read_csv(mapping_file)
                
                # 按motif_pattern和label分组计数
                for motif_pattern in mapping_df['motif_pattern'].unique():
                    motif_data = mapping_df[mapping_df['motif_pattern'] == motif_pattern]
                    
                    # 计算唯一的序列计数（而不是stem实例计数）
                    positive_seqs = motif_data[motif_data['label'] == 'positive']['sequence_name'].nunique()
                    negative_seqs = motif_data[motif_data['label'] == 'negative']['sequence_name'].nunique()
                    
                    self.mapped_positive_counts[motif_pattern] = positive_seqs
                    self.mapped_negative_counts[motif_pattern] = negative_seqs
                    
            except Exception as e:
                print(f"Warning: Could not load mapping counts: {e}")
                # 如果无法读取映射文件，使用Step3的计数作为备选
                for motif in self.motif_data:
                    pattern = motif['pattern']
                    self.mapped_positive_counts[pattern] = motif.get('positive_count', 0)
                    self.mapped_negative_counts[pattern] = motif.get('negative_count', 0)
        else:
            # 如果没有映射文件，使用Step3的计数
            for motif in self.motif_data:
                pattern = motif['pattern']
                self.mapped_positive_counts[pattern] = motif.get('positive_count', 0)
                self.mapped_negative_counts[pattern] = motif.get('negative_count', 0)
        
        self._mapped_counts_initialized = True

    def get_total_positive_sequences(self):
        """获取输入阳性序列总数"""
        if not hasattr(self, '_total_sequences_initialized'):
            self._initialize_total_sequences()
        
        return self.total_positive_seqs

    def get_total_negative_sequences(self):
        """获取输入阴性序列总数"""
        if not hasattr(self, '_total_sequences_initialized'):
            self._initialize_total_sequences()
        
        return self.total_negative_seqs

    def _initialize_total_sequences(self):
        """初始化总序列数缓存"""
        # 尝试从Step4的统计文件中读取
        stats_file = self.output_dir / 'motif_mapping_statistics.csv'
        if stats_file.exists():
            try:
                stats_df = pd.read_csv(stats_file)
                if 'total_positive_sequences' in stats_df.columns:
                    self.total_positive_seqs = stats_df['total_positive_sequences'].iloc[0]
                if 'total_negative_sequences' in stats_df.columns:
                    self.total_negative_seqs = stats_df['total_negative_sequences'].iloc[0]
            except:
                pass
        
        # 如果无法从统计文件读取，尝试从原始序列文件计算
        if not hasattr(self, 'total_positive_seqs'):
            positive_file = self.output_dir / 'positive_structures.txt'
            if positive_file.exists():
                with open(positive_file, 'r') as f:
                    self.total_positive_seqs = len([line for line in f if line.startswith('>')])
            else:
                self.total_positive_seqs = 0
        
        if not hasattr(self, 'total_negative_seqs'):
            negative_file = self.output_dir / 'negative_structures.txt'
            if negative_file.exists():
                with open(negative_file, 'r') as f:
                    self.total_negative_seqs = len([line for line in f if line.startswith('>')])
            else:
                self.total_negative_seqs = 0
        
        self._total_sequences_initialized = True

    def create_html_report(self, output_file='motif_visualization_report.html', max_motifs=5, cmd_args=None):
        """创建完整的HTML报告，包含所有图表和表格"""
        # 使用传入的 max_motifs 或实例变量
        if max_motifs is None:
            max_motifs = self.max_motifs
        
        print("Creating comprehensive HTML report...")
        print(f"Maximum motifs to display: {max_motifs}")
        
        # 生成命令行信息部分
        cmd_section = self.create_command_line_section(cmd_args)
        
        # 排序motif（按调整后p值）
        sorted_motifs = sorted(self.motif_data, key=lambda x: x['adjusted_p_value'])
        
        # 过滤正富集的motif
        positive_enriched_motifs = []
        for motif in sorted_motifs:
            enrichment = motif.get('enrichment', 1)
            # 检查enrichment是否为正富集（>1或inf）
            if enrichment > 1 or enrichment == float('inf'):
                positive_enriched_motifs.append(motif)
        
        # 计算所有分析motif中的正富集数量（用于显示）
        total_positive_enriched_count = len(positive_enriched_motifs)
        
        print(f"Original motifs: {len(sorted_motifs)}")
        print(f"Positively enriched motifs: {total_positive_enriched_count}")
        
        # 限制显示数量以避免性能问题
        if len(positive_enriched_motifs) > max_motifs:
            print(f"Displaying top {max_motifs} positively enriched motifs")
            display_motifs = positive_enriched_motifs[:max_motifs]
        else:
            print(f"Displaying all {len(positive_enriched_motifs)} positively enriched motifs")
            display_motifs = positive_enriched_motifs
        
        # 生成motif卡片 - 关键修改：使用 display_motifs 而不是 positive_enriched_motifs
        motif_cards = []
        for i, motif_info in enumerate(display_motifs):  # 这里改为 display_motifs
            print(f"Processing motif {i+1}/{len(display_motifs)}: {motif_info['pattern']}")
            try:
                card_html = self.create_motif_card(motif_info, i)
                motif_cards.append(card_html)
            except Exception as e:
                print(f"Error processing motif {motif_info['pattern']}: {e}")
                continue

        # 加载和嵌入所有图表
        additional_sections = self.create_additional_sections()
        
        # 获取输入序列总数
        total_positive_seqs = self.get_total_positive_sequences()
        total_negative_seqs = self.get_total_negative_sequences()
        
        # 创建完整的HTML
        html_content = f"""
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>RNA Stem-loop Motif Analysis Report</title>
            <style>
                body {{
                    font-family: 'Helvetica Neue', Arial, sans-serif;
                    line-height: 1.6;
                    margin: 0;
                    padding: 20px;
                    background-color: #f8f9fa;
                    color: #333;
                }}
                .container {{
                    max-width: 1400px;
                    margin: 0 auto;
                    background: white;
                    padding: 30px;
                    border-radius: 5px;
                    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
                }}
                .header {{
                    text-align: center;
                    margin-bottom: 40px;
                    border-bottom: 2px solid #2c3e50;
                    padding-bottom: 20px;
                }}
                .header h1 {{
                    color: #2c3e50;
                    margin-bottom: 10px;
                    font-weight: 300;
                    font-size: 28px;
                }}
                .section {{
                    margin-bottom: 40px;
                    padding: 20px;
                    background: #fff;
                    border-radius: 5px;
                    border-left: 4px solid #3498db;
                }}
                .section h2 {{
                    color: #2c3e50;
                    margin-top: 0;
                    border-bottom: 1px solid #ecf0f1;
                    padding-bottom: 10px;
                    font-weight: 400;
                    font-size: 20px;
                }}
                .summary-stats {{
                    display: grid;
                    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
                    gap: 10px;
                    margin-top: 15px;
                }}
                .stat-item {{
                    background: white;
                    padding: 12px;
                    border-radius: 3px;
                    text-align: center;
                    border: 1px solid #ecf0f1;
                }}
                .stat-value {{
                    font-size: 20px;
                    font-weight: 600;
                    color: #2c3e50;
                }}
                .stat-label {{
                    font-size: 12px;
                    color: #7f8c8d;
                }}
                .input-sequences {{
                    background: #e8f4fd;
                    border-left: 4px solid #3498db;
                }}
                .motif-analysis {{
                    background: #e8f5e8;
                    border-left: 4px solid #27ae60;
                }}
                .motif-card {{
                    border: 1px solid #ddd;
                    border-radius: 5px;
                    margin-bottom: 25px;
                    padding: 20px;
                    background: #fff;
                }}
                .motif-header {{
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    margin-bottom: 15px;
                    flex-wrap: wrap;
                }}
                .motif-header h3 {{
                    margin: 0;
                    color: #2c3e50;
                    font-family: 'Courier New', monospace;
                    background: #f8f9fa;
                    padding: 6px 10px;
                    border-radius: 3px;
                    border: 1px solid #dee2e6;
                    font-size: 14px;
                }}
                .motif-stats {{
                    display: flex;
                    gap: 8px;
                    flex-wrap: wrap;
                }}
                .position-line {{
                    display: flex;
                    justify-content: space-between;
                    margin-bottom: 5px;
                }}
                .position-number {{
                    font-size: 10px;
                    color: #6c757d;
                    font-weight: bold;
                    min-width: 20px;
                    text-align: center;
                }}
                .position-space {{
                    min-width: 20px;
                }}
                .structure-line {{
                    display: flex;
                    justify-content: space-between;
                }}
                .structure-symbol {{
                    font-size: 14px;
                    font-weight: bold;
                    min-width: 20px;
                    text-align: center;
                }}
                .structure-paired {{
                    color: #dc3545; /* 红色 */
                }}
                .structure-unpaired {{
                    color: #007bff; /* 蓝色 */
                }}
                .structure-other {{
                    color: #6c757d; /* 灰色 */
                }}
                .stat-badge {{
                    padding: 4px 8px;
                    border-radius: 3px;
                    font-size: 11px;
                    font-weight: 500;
                }}
                .enrichment {{ background: #e8f5e8; color: #27ae60; border: 1px solid #27ae60; }}
                .p-value {{ background: #ffeaea; color: #e74c3c; border: 1px solid #e74c3c; }}
                .length {{ background: #e3f2fd; color: #2980b9; border: 1px solid #2980b9; }}
                .type {{ background: #fff3e0; color: #f39c12; border: 1px solid #f39c12; }}
                .motif-content {{
                    display: grid;
                    grid-template-columns: 1fr 1fr;
                    gap: 20px;
                    align-items: start;
                }}
                .motif-image img {{
                    max-width: 100%;
                    height: auto;
                    border: 1px solid #ddd;
                    border-radius: 3px;
                }}
                .motif-table {{
                    width: 100%;
                    border-collapse: collapse;
                    font-size: 12px;
                }}
                .motif-table th, .motif-table td {{
                    padding: 6px 8px;
                    text-align: left;
                    border-bottom: 1px solid #ecf0f1;
                }}
                .motif-table th {{
                    background-color: #f8f9fa;
                    font-weight: 600;
                    color: #2c3e50;
                }}
                .chart-container {{
                    text-align: center;
                    margin: 15px 0;
                }}
                .chart-container img {{
                    max-width: 80%;  /* 缩小图表尺寸 */
                    height: auto;
                    border: 1px solid #ddd;
                    border-radius: 3px;
                }}
                @media (max-width: 768px) {{
                    .motif-content {{
                        grid-template-columns: 1fr;
                    }}
                }}
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1>StemSage de novo RNA Stem-loop Motif Analysis Report</h1>
                    <p>Analysis of enriched stem-loop motifs in RBP-binding regions</p>
                </div>
                
                <div class="toc">
                    <h2>📑 Table of Contents</h2>
                    <ul>
                        <li><a href="#command">Command Line</a></li>
                        <li><a href="#summary">Analysis Summary</a></li>
                        <li><a href="#filtering">Filtering Criteria</a></li>
                        <li><a href="#motifs">Enriched Motifs Visualization</a></li>
                        <li><a href="#stage1">Stage 1: Feature Analysis</a></li>
                        <li><a href="#stage2">Stage 2: Model Performance</a></li>
                        <li><a href="#stage3">Stage 3: Pattern Mining</a></li>
                        <li><a href="#stage4">Stage 4: Motif Mapping</a></li>
                        <li><a href="#tables">Key Data Tables</a></li>
                    </ul>
                </div>
                
                <div id="command" class="section">
                    <h2>🖥️ Command Line Execution</h2>
                    {cmd_section}
                </div>
                
                <div id="summary" class="section">
                    <h2>📊 Analysis Summary</h2>
                    
                    <div class="section input-sequences">
                        <h3>📥 Input Sequences</h3>
                        <div class="summary-stats">
                            <div class="stat-item">
                                <div class="stat-value">{total_positive_seqs}</div>
                                <div class="stat-label">Positive Sequences</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-value">{total_negative_seqs}</div>
                                <div class="stat-label">Negative Sequences</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-value">{total_positive_seqs + total_negative_seqs}</div>
                                <div class="stat-label">Total Sequences</div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="section motif-analysis">
                        <h3>🔍 Motif Analysis Results</h3>
                        <div class="summary-stats">
                            <div class="stat-item">
                                <div class="stat-value">{len(self.motif_data)}</div>
                                <div class="stat-label">Total Motifs Analyzed</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-value">{total_positive_enriched_count}</div>
                                <div class="stat-label">Positively Enriched</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-value">{sum(1 for m in positive_enriched_motifs if m['adjusted_p_value'] < 0.05)}</div>
                                <div class="stat-label">Significant (p < 0.05)</div>
                            </div>
                        </div>
                    </div>
                </div>
                                                
                <div id="motifs" class="section">
                    <h2>🔬 Enriched Stem-loop Motifs Visualization</h2>
                    <p>This section displays the most statistically significant stem-loop motifs found in RBP-binding sequences. Each motif is visualized with its secondary structure and positional nucleotide frequencies.</p>
                    {"".join(motif_cards) if motif_cards else "<p>No positively enriched motifs found matching the criteria.</p>"}
                </div>
                
                {additional_sections}
                
                <div class="footer">
                    <p><em>Report generated on {pd.Timestamp.now().strftime('%Y-%m-%d %H:%M:%S')}</em></p>
                    <p><em>StemSage Version: {__version__}</em></p>
                </div>
            </div>
        </body>
        </html>
        """
        
        # 保存HTML文件
        with open(output_file, 'w', encoding='utf-8') as f:
            f.write(html_content)
        
        print(f"Comprehensive HTML report saved to {output_file}")

    def create_command_line_section(self, cmd_args):
        """创建命令行参数显示部分"""
        if not cmd_args:
            return """
            <div class="command-line">
                <h3>Command Line Information</h3>
                <p><em>Command line arguments not available for this run.</em></p>
            </div>
            """
        
        # 构建命令行显示
        cmd_parts = ["StemSage"]
        
        # 定义参数显示顺序和默认值
        param_order = [
            ('positive_bed', '--positive_bed'),
            ('negative_bed', '--negative_bed'), 
            ('positive_fasta', '--positive_fasta'),
            ('negative_fasta', '--negative_fasta'),
            ('genome_fa', '--genome_fa'),
            ('out', '--out'),
            ('extend', '--extend', 50),
            ('model', '--model', 'xgboost'),
            ('test_size', '--test_size', 0.2),
            ('random_state', '--random_state', 42),
            ('cv_folds', '--cv_folds', 5),
            ('threads', '--threads', 1),
            ('max_motifs', '--max_motifs', 5),
            ('max_stem_length', '--max_stem_length', 5),
            ('min_stem_length', '--min_stem_length', 1),
            ('similar_matching', '--similar_matching', True),
            ('similarity_threshold', '--similarity_threshold', 0.9),
            ('max_length_diff', '--max_length_diff', 1),
        ]
        
        for param_info in param_order:
            if len(param_info) == 2:
                param_name, flag = param_info
                default_value = None
            else:
                param_name, flag, default_value = param_info
            
            # 使用 getattr 安全获取属性值
            value = getattr(cmd_args, param_name, None)
            
            # 检查值是否有效（不是None且不是空字符串）
            if value is not None and value != "":
                if param_name in ['positive_bed', 'negative_bed', 'positive_fasta', 'negative_fasta', 'genome_fa', 'out']:
                    # 文件路径参数 - 显示实际路径
                    cmd_parts.append(f'<span class="cmd-arg">{flag}</span> <span class="cmd-value">"{value}"</span>')
                else:
                    # 其他参数
                    cmd_parts.append(f'<span class="cmd-arg">{flag}</span> <span class="cmd-value">{value}</span>')
            elif default_value is not None:
                # 使用默认值的参数
                cmd_parts.append(f'<span class="cmd-arg">{flag}</span> <span class="cmd-default">{default_value} (default)</span>')
        
        cmd_display = " \\\n    ".join(cmd_parts)
        
        return f"""
        <div class="command-line">
            <h3>Command Line Execution</h3>
            <pre>{cmd_display}</pre>
            <p><small>Note: Parameters with <span class="cmd-default">(default)</span> were not explicitly specified and use default values.</small></p>
        </div>
        """
        
    def create_additional_sections(self):
        """创建包含所有阶段图表和表格的额外部分"""
        sections_html = ""
        
        # 生成实际图表并嵌入
        try:
            # Stage 1: Feature Analysis
            sections_html += self.create_stage1_section()
            
            # Stage 2: Model Performance  
            sections_html += self.create_stage2_section()
            
            # Stage 3: Pattern Mining
            sections_html += self.create_stage3_section()
            
            # Stage 4: Motif Mapping
            sections_html += self.create_stage4_section()
            
            # Key Data Tables
            sections_html += self.create_data_tables_section()
            
        except Exception as e:
            print(f"Warning: Could not generate additional sections: {e}")
            sections_html = """
            <div class="section">
                <h2>📊 Additional Analysis Sections</h2>
                <p>Additional visualization sections require the corresponding analysis files to be available.</p>
            </div>
            """
        
        return sections_html

    def create_stage1_section(self):
        """创建阶段1的特征分析部分 - 调整图表大小"""
        feature_file = self.output_dir / 'rna_features_dataset.csv'
        if not feature_file.exists():
            return """
            <div class="section">
                <h2>Stage 1: Feature Analysis</h2>
                <p>Feature analysis data not available.</p>
            </div>
            """
        
        charts_html = ""
        chart_files = {
            'feature_differences.png': 'Top Feature Differences',
            'feature_correlation.png': 'Feature Correlation Matrix', 
            'feature_distributions.png': 'Feature Distributions'
        }
        
        for chart_file, description in chart_files.items():
            chart_path = self.output_dir / chart_file
            if chart_path.exists():
                try:
                    import base64
                    with open(chart_path, 'rb') as f:
                        img_data = base64.b64encode(f.read()).decode()
                    
                    # 调整图表显示尺寸为60%
                    charts_html += f"""
                    <div class="chart-container">
                        <img src="data:image/png;base64,{img_data}" alt="{description}" style="max-width: 60%;">
                        <div class="chart-description">{description}</div>
                    </div>
                    """
                except Exception as e:
                    print(f"Warning: Could not embed {chart_file}: {e}")
        
        return f"""
        <div class="section">
            <h2>Stage 1: Feature Analysis</h2>
            <p>Analysis of sequence features in RBP-binding sequences.</p>
            {charts_html}
        </div>
        """

    def create_stage2_section(self):
        """创建阶段2的模型性能部分"""
        # 检查模型分析图表文件是否存在
        model_chart_file = self.output_dir / 'model_analysis_xgboost.png'
        if not model_chart_file.exists():
            return """
            <div id="stage2" class="section">
                <h2>📈 Stage 2: Model Performance</h2>
                <p>Model performance data not available. Please ensure 'model_analysis_xgboost.png' exists.</p>
            </div>
            """
        
        # 嵌入模型分析图表
        try:
            import base64
            with open(model_chart_file, 'rb') as f:
                img_data = base64.b64encode(f.read()).decode()
            
            charts_html = f"""
            <div class="chart-container">
                <img src="data:image/png;base64,{img_data}" alt="XGBoost Model Analysis">
                <div class="chart-description">XGBoost Model Performance Analysis</div>
            </div>
            """
            
            # 检查是否有其他模型相关文件
            additional_info = ""
            model_files = list(self.output_dir.glob('*_stem_classifier.pkl'))
            if model_files:
                additional_info = f"<p><strong>Trained Models:</strong> {len(model_files)} model(s) saved</p>"
            
            return f"""
            <div id="stage2" class="section">
                <h2>📈 Stage 2: Model Performance</h2>
                <p>Evaluation of XGBoost machine learning model performance in predicting RBP-binding sequences.</p>
                {additional_info}
                {charts_html}
            </div>
            """
            
        except Exception as e:
            print(f"Warning: Could not embed model analysis chart: {e}")
            return f"""
            <div id="stage2" class="section">
                <h2>📈 Stage 2: Model Performance</h2>
                <p>Model performance chart exists but could not be embedded: {e}</p>
            </div>
            """
    

    
    def create_stage3_section(self):
        """创建阶段3的模式挖掘部分 - 删除统计表格"""
        pattern_file = self.output_dir / 'stem_patterns_analysis.csv'
        if not pattern_file.exists():
            return """
            <div class="section">
                <h2>Stage 3: Pattern Mining</h2>
                <p>Pattern mining data not available.</p>
            </div>
            """
        
        # 只保留图表，删除统计表格
        chart_variants = [
            'pattern_analysis.png',
            'stem_pattern_analysis.png', 
            'pattern_mining_results.png'
        ]
        
        charts_html = ""
        for chart_file in chart_variants:
            chart_path = self.output_dir / chart_file
            if chart_path.exists():
                try:
                    import base64
                    with open(chart_path, 'rb') as f:
                        img_data = base64.b64encode(f.read()).decode()
                    
                    charts_html += f"""
                    <div class="chart-container">
                        <img src="data:image/png;base64,{img_data}" alt="Pattern Analysis">
                        <div class="chart-description">{chart_file}</div>
                    </div>
                    """
                    break
                except Exception as e:
                    print(f"Warning: Could not embed {chart_file}: {e}")
        
        return f"""
        <div class="section">
            <h2>Stage 3: Pattern Mining</h2>
            <p>Discovery of frequent stem-loop patterns in RBP-binding sequences.</p>
            {charts_html}
        </div>
        """

    def create_stage4_section(self):
        """创建阶段4的motif映射部分"""
        mapping_file = self.output_dir / 'motif_mapping_detailed.csv'
        if not mapping_file.exists():
            return """
            <div id="stage4" class="section">
                <h2>🗺️ Stage 4: Motif Mapping</h2>
                <p>Motif mapping data not available.</p>
            </div>
            """
        
        # 如果有motif映射可视化图片，可以嵌入
        viz_file = self.output_dir / 'motif_mapping_visualization.png'
        if viz_file.exists():
            try:
                import base64
                with open(viz_file, 'rb') as f:
                    img_data = base64.b64encode(f.read()).decode()
                
                return f"""
                <div id="stage4" class="section">
                    <h2>🗺️ Stage 4: Motif Mapping</h2>
                    <p>Mapping of discovered motifs to sequence regions and enrichment analysis.</p>
                    
                    <div class="chart-container">
                        <img src="data:image/png;base64,{img_data}" alt="Motif Mapping Visualization">
                        <div class="chart-description">Motif occurrence and co-occurrence patterns</div>
                    </div>
                </div>
                """
            except Exception as e:
                print(f"Warning: Could not embed motif mapping visualization: {e}")
        
        return """
        <div id="stage4" class="section">
            <h2>🗺️ Stage 4: Motif Mapping</h2>
            <p>Mapping of discovered motifs to sequence regions and enrichment analysis.</p>
            <p><em>Motif mapping completed successfully. See detailed mapping files for results.</em></p>
        </div>
        """

    def create_data_tables_section(self):
        """创建关键数据表格部分"""
        # 创建实际的motif数据表格
        table_rows = ""
        if self.motif_data:
            top_motifs = sorted(self.motif_data, key=lambda x: x['adjusted_p_value'])[:10]
            for motif in top_motifs:
                table_rows += f"""
                <tr>
                    <td><code>{html.escape(motif['pattern'][:30])}{'...' if len(motif['pattern']) > 30 else ''}</code></td>
                    <td>{motif.get('enrichment', 0):.2f}</td>
                    <td>{motif.get('adjusted_p_value', 1):.2e}</td>
                    <td>{motif.get('positive_frequency', 0):.4f}</td>
                    <td>{motif.get('negative_frequency', 0):.4f}</td>
                </tr>
                """
        
        return f"""
        <div id="tables" class="section">
            <h2>📋 Key Data Tables</h2>
            <p>Summary tables of important results and statistics.</p>
            
            <div class="table-container">
                <h3>Top Enriched Motifs</h3>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Motif Pattern</th>
                            <th>Enrichment</th>
                            <th>Adjusted p-value</th>
                            <th>Positive Frequency</th>
                            <th>Negative Frequency</th>
                        </tr>
                    </thead>
                    <tbody>
                        {table_rows if table_rows else '<tr><td colspan="5">No motif data available</td></tr>'}
                    </tbody>
                </table>
            </div>
            
            <div class="table-container">
                <h3>Available Output Files</h3>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>File</th>
                            <th>Description</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr><td>stem_patterns_analysis.csv</td><td>Detailed pattern analysis results</td></tr>
                        <tr><td>stem_patterns_motifs.csv</td><td>Significant motif patterns</td></tr>
                        <tr><td>motif_mapping_detailed.csv</td><td>Detailed motif-sequence mappings</td></tr>
                        <tr><td>motif_mapping_matrix.csv</td><td>Sequence-motif occurrence matrix</td></tr>
                        <tr><td>rna_features_dataset.csv</td><td>Feature dataset for machine learning</td></tr>
                    </tbody>
                </table>
            </div>
        </div>
        """

def main():
    # 创建可视化器实例
    visualizer = MotifVisualizer()
    
    # 使用 run 方法执行完整流程
    success = visualizer.run()
    
    if success:
        print("Visualization completed successfully!")
    else:
        print("Visualization failed!")

if __name__ == "__main__":
    main()