#!/bin/bash

input_bed="__INPUT_BED__"
output_fasta="__OUTPUT_FASTA__"
reference_fa="__REFERENCE_FA__"
extend_bp=__EXTEND_BP__

# 确保输出目录存在
output_dir=$(dirname "$output_fasta")
mkdir -p "$output_dir"

# 临时文件（放在输出目录中）
temp_bed1="$output_dir/temp_processed.bed"
temp_bed_sorted="$output_dir/temp_sorted.bed"
temp_bed_extended="$output_dir/temp_extended.bed"

echo "开始处理BED文件: $input_bed"
echo "输出目录: $output_dir"

# 步骤0: 延伸BED区域
echo "步骤0: 延伸BED区域 ±$extend_bp bp..."
awk -v extend=$extend_bp 'BEGIN{OFS="\t"}
NF >= 3 && $0 !~ /^#/ {
    chr = $1;
    start = ($2 - extend > 0) ? $2 - extend : 0;
    end = $3 + extend;
    strand = (NF >= 6) ? $6 : ".";
    name = (NF >= 4) ? $4 : "region_" NR;
    score = (NF >= 5) ? $5 : ".";
    print chr, start, end, name, score, strand;
}' "$input_bed" > "$temp_bed_extended"

# 步骤1: 检查并添加chr前缀，过滤染色体，构建自定义名称
echo "步骤1: 检查染色体前缀并过滤非经典染色体..."
awk '
BEGIN {
    split("1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22 23 X Y", valid);
    for(i in valid) valid_chr[valid[i]] = 1;
    for(i in valid) valid_chr["chr" valid[i]] = 1;
    OFS = "\t"
}
NF >= 3 && $0 !~ /^#/ {
    chr = $1;
    start = $2;
    end = $3;
    strand = (NF >= 6) ? $6 : ".";
    
    raw_chr = chr;
    gsub(/^chr/, "", raw_chr);
    
    if (valid_chr[chr] || valid_chr[raw_chr]) {
        if (chr !~ /^chr/) chr = "chr" chr;
        # 创建自定义名称
        name = "crosslink_region_" NR "_" chr "_" start "_" end;
        print chr, start, end, name, ".", strand;
    }
}' "$temp_bed_extended" > "$temp_bed1"

# 步骤2: 统计每条染色体上的序列数目
echo "步骤2: 统计各染色体序列数目..."
echo "染色体统计结果:"
awk '{print $1}' "$temp_bed1" | sort | uniq -c | sort -k1,1nr

# 步骤3: 对BED文件进行排序
echo "步骤3: 对BED文件进行排序..."
sort -k1,1 -k2,2n "$temp_bed1" > "$temp_bed_sorted"

# 步骤4: 使用bedtools提取FASTA序列
echo "步骤4: 使用bedtools提取FASTA序列..."
bedtools getfasta -fi "$reference_fa" -bed "$temp_bed_sorted" -fo "$output_fasta" -s -name

if [ $? -eq 0 ] && [ -f "$output_fasta" ]; then
    echo "bedtools succeed."
else
    echo "bedtools failed."
    exit 1
fi

echo "Output file: $output_fasta"

# 显示结果统计
echo "The number of sequence:"
grep -c "^>" "$output_fasta"

echo "Head of the FASTA file:"
head -n 10 "$output_fasta"

# 清理临时文件
rm -f "$temp_bed1" "$temp_bed_sorted" "$temp_bed_extended"

echo "Finish transforming bed to FASTA."