"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdktfResolver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const os = require("os");
const path = require("path");
const cdktf_1 = require("cdktf");
const exec_1 = require("./exec");
class CdktfResolver {
    constructor(props) {
        this.app = props.app;
    }
    resolve(context) {
        // detects full resources being passed as values.
        // this is legit in terraform and should resolve
        // to the resource fqn.
        // see https://github.com/hashicorp/terraform-cdk/blob/main/packages/cdktf/lib/terraform-output.ts#L79
        const addressable = this.isAddressable(context.value);
        if (!cdktf_1.Token.isUnresolved(context.value) && !addressable) {
            return;
        }
        const output = this.findOutput(context.value);
        try {
            const outputValue = this.fetchOutputValue(output);
            context.replaceValue(outputValue);
        }
        catch (err) {
            // if both cdk8s and CDKTF applications are defined within the same file,
            // a cdk8s synth is going to happen before the CDKTF deployment.
            // in this case we must swallow the error, otherwise the CDKTF deployment
            // won't be able to go through. we replace the value with something to indicate
            // that a fetching attempt was made and failed.
            context.replaceValue(`Failed fetching value for output ${output.node.path}: ${err}`);
        }
    }
    fetchOutputValue(output) {
        if (!this._outputs) {
            this._outputs = this.fetchOutputs();
        }
        const stack = cdktf_1.TerraformStack.of(output);
        return this._outputs[stack.node.id][output.friendlyUniqueId];
    }
    fetchOutputs() {
        const projectDir = fs.mkdtempSync(path.join(os.tmpdir(), 'cdktf-project-'));
        const outputsFile = 'outputs.json';
        try {
            this.app.synth();
            const cdktfJson = {
                // `cdktf output` doesn't actually use this value,
                // so we can put whatever we want here.
                language: 'python',
                app: 'cdktf.out',
            };
            // create our own copy of the synthesized app so we can safely clean it up
            copyDirectory(this.app.outdir, path.join(projectDir, cdktfJson.app));
            // write the configuration file as it is required for any cdktf command
            fs.writeFileSync(path.join(projectDir, 'cdktf.json'), JSON.stringify(cdktfJson));
            const stacks = this.app.node.findAll().filter(c => cdktf_1.TerraformStack.isStack(c)).map(c => c.node.id);
            const command = ['cdktf output',
                '--skip-synth',
                `--output ${cdktfJson.app}`,
                `--outputs-file ${outputsFile}`,
                `${stacks.join(' ')}`];
            (0, exec_1.execSync)(command.join(' '), { cwd: projectDir });
            return JSON.parse(fs.readFileSync(path.join(projectDir, outputsFile), { encoding: 'utf-8' }));
        }
        finally {
            fs.rmSync(projectDir, { recursive: true });
        }
    }
    findOutput(value) {
        const inspectedStacks = this.app.node.findAll().filter(c => cdktf_1.TerraformStack.isStack(c));
        for (const stack of inspectedStacks) {
            const output = stack.node.findAll().filter(c => c instanceof cdktf_1.TerraformOutput && c.value === value)[0];
            // we don't really care if there are more outputs (possibly from different stacks)
            // that point to the same value. the first will suffice.
            if (output)
                return output;
        }
        // This can happen if either:
        // --------------------------
        //  1. User didn't define an output.
        //  2. User defined a complex literal output value (e.g { bucketName: bucket.bucket }).
        throw new Error(`Unable to find output defined for ${value} (Inspected stacks: ${inspectedStacks.map(s => s.node.id).join(',')})`);
    }
    isAddressable(object) {
        return object && typeof object === 'object' && !Array.isArray(object) && 'fqn' in object;
    }
}
exports.CdktfResolver = CdktfResolver;
_a = JSII_RTTI_SYMBOL_1;
CdktfResolver[_a] = { fqn: "@cdk8s/cdktf-resolver.CdktfResolver", version: "0.0.89" };
function copyDirectory(sourceDir, targetDir) {
    if (!fs.existsSync(targetDir)) {
        fs.mkdirSync(targetDir);
    }
    const files = fs.readdirSync(sourceDir);
    for (const file of files) {
        const sourceFilePath = path.join(sourceDir, file);
        const targetFilePath = path.join(targetDir, file);
        const stats = fs.statSync(sourceFilePath);
        if (stats.isDirectory()) {
            copyDirectory(sourceFilePath, targetFilePath);
        }
        else if (stats.isFile()) {
            fs.copyFileSync(sourceFilePath, targetFilePath);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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