# Release Process for repo-clean

This document outlines the complete process for releasing new versions of repo-clean to PyPI.

## 📋 Pre-Release Checklist

### Code Quality
- [ ] All tests pass locally: `pytest tests/`
- [ ] Code formatting is correct: `black src/`
- [ ] Type checking passes: `mypy src/`
- [ ] Linting passes: `flake8 src/`
- [ ] No known security vulnerabilities
- [ ] Documentation is up to date

### Version Management
- [ ] Update version in `src/__init__.py`
- [ ] Update version in `pyproject.toml` (if using modern build)
- [ ] Version follows semantic versioning (MAJOR.MINOR.PATCH)
- [ ] CHANGELOG.md updated with new features/fixes
- [ ] All new features documented

### Repository State
- [ ] All changes committed and pushed to main branch
- [ ] No uncommitted changes: `git status`
- [ ] Repository hygiene: `repo-clean scan` (eat your own dog food!)
- [ ] CI/CD pipeline passes all checks

## 🚀 Release Process

### 1. Create Release Tag

```bash
# Ensure you're on main branch
git checkout main
git pull origin main

# Create and push tag
git tag -a v1.0.0 -m "Release version 1.0.0"
git push origin v1.0.0
```

### 2. Create GitHub Release

1. Go to GitHub repository → Releases → "Create a new release"
2. Select the tag you just created
3. Use format: "v1.0.0 - Brief Description"
4. Include release notes with:
   - New features
   - Bug fixes
   - Breaking changes (if any)
   - Upgrade instructions

**Example Release Notes:**
```markdown
## 🎉 repo-clean v1.0.0

### ✨ New Features
- Complete .gitignore management system
- Multi-ecosystem code quality linting
- GitHub Actions integration templates
- Comprehensive safety controls

### 🛡️ Safety Improvements
- Enhanced backup/rollback functionality
- Preview mode for all operations
- Educational explanations for all issues

### 📚 Documentation
- Complete usage guides
- GitHub integration examples
- Best practices documentation

### 🔧 Technical
- Modern packaging with pyproject.toml
- Comprehensive test coverage
- Type hints throughout codebase

**Installation:** `pip install repo-clean`
**Documentation:** https://github.com/pvestal/repo-clean#readme
```

### 3. Automated Publishing

The GitHub Actions workflow will automatically:
- Run tests on multiple Python versions
- Build the package
- Publish to PyPI
- Create post-release tasks

## 🧪 Testing Release

### Test PyPI (Recommended First)

```bash
# Manual test publish to Test PyPI
python -m build
twine upload --repository testpypi dist/*

# Test installation from Test PyPI
pip install --index-url https://test.pypi.org/simple/ repo-clean
```

### Production PyPI

Publishing happens automatically via GitHub Actions when a release is created.

## 📦 Manual Publishing (Backup)

If automated publishing fails:

```bash
# Install build tools
pip install build twine

# Clean previous builds
rm -rf dist/ build/ *.egg-info/

# Build package
python -m build

# Check package
twine check dist/*

# Upload to PyPI
twine upload dist/*
```

## 🔍 Post-Release Verification

### Installation Testing
```bash
# Create fresh virtual environment
python -m venv test-env
source test-env/bin/activate  # or `test-env\Scripts\activate` on Windows

# Install from PyPI
pip install repo-clean

# Test basic functionality
repo-clean --help
repo-clean scan
repo-clean gitignore --analyze

# Cleanup
deactivate
rm -rf test-env
```

### Package Verification
- [ ] Check [PyPI page](https://pypi.org/project/repo-clean/) displays correctly
- [ ] README renders properly on PyPI
- [ ] All files included in package: `tar -tzf dist/*.tar.gz`
- [ ] Entry points work: `which repo-clean`

## 🔄 Version Numbering

Follow [Semantic Versioning](https://semver.org/):

- **MAJOR** (X.0.0): Breaking changes, incompatible API changes
- **MINOR** (1.X.0): New features, backwards compatible
- **PATCH** (1.0.X): Bug fixes, backwards compatible

### Examples:
- `1.0.0` → `1.0.1`: Bug fix release
- `1.0.1` → `1.1.0`: New gitignore feature added
- `1.1.0` → `2.0.0`: CLI interface changed (breaking)

## 🐛 Rollback Process

If a release has critical issues:

### PyPI Rollback (Limited)
```bash
# You can only "yank" releases, not delete them
# This hides them from pip install but doesn't remove them
pip install twine
twine --help  # Check for yank command in newer versions
```

### GitHub Release Rollback
1. Edit the GitHub release
2. Mark it as "pre-release"
3. Create hotfix branch
4. Fix critical issues
5. Create new patch release

### User Communication
```bash
# Create GitHub issue
title: "⚠️ Critical Issue in v1.0.0 - Use v1.0.1 Instead"
labels: critical, bug, release

# Update README with notice
echo "⚠️ **Important:** Please use v1.0.1 instead of v1.0.0 due to critical bug #123" >> README.md
```

## 📊 Release Metrics

Track these metrics for each release:

### Download Stats
- PyPI download statistics
- GitHub release download counts
- Documentation page views

### Quality Metrics
- Issue reports within 48 hours
- Community feedback/reviews
- Installation success rate

### Development Metrics
- Time from development to release
- Number of commits in release
- Test coverage percentage

## 🎯 Release Types

### Patch Release (1.0.X)
- **Timeline**: As needed for bugs
- **Testing**: Focused regression testing
- **Communication**: GitHub release notes

### Minor Release (1.X.0)
- **Timeline**: Monthly or when significant features ready
- **Testing**: Full test suite + new feature testing
- **Communication**: Release notes + documentation updates

### Major Release (X.0.0)
- **Timeline**: When breaking changes necessary
- **Testing**: Comprehensive testing + migration guides
- **Communication**: Announcement + migration documentation

## 🚨 Emergency Release Process

For critical security fixes:

1. **Immediate**: Create hotfix branch from latest release tag
2. **Fix**: Implement minimal fix for security issue
3. **Test**: Run focused security testing
4. **Release**: Create patch release within 24 hours
5. **Communicate**: Security advisory + update recommendations

```bash
# Emergency release workflow
git checkout v1.0.0
git checkout -b hotfix/security-fix
# ... make minimal security fix ...
git commit -m "🔒 Security fix for CVE-XXXX"
git tag v1.0.1
git push origin v1.0.1
# Create GitHub release immediately
```

## 📋 Release Template

```markdown
## repo-clean vX.X.X

### 🎯 Overview
Brief description of what this release focuses on.

### ✨ New Features
- Feature 1: Description
- Feature 2: Description

### 🐛 Bug Fixes
- Fix 1: Description
- Fix 2: Description

### 🔧 Improvements
- Improvement 1: Description
- Improvement 2: Description

### ⚠️ Breaking Changes
- Change 1: Description and migration path
- Change 2: Description and migration path

### 📚 Documentation
- Documentation updates
- New guides added

### 💻 Technical Changes
- Internal improvements
- Dependency updates

### 🙏 Contributors
Thank you to all contributors who made this release possible!

**Full Changelog**: https://github.com/pvestal/repo-clean/compare/v1.0.0...v1.1.0
```

---

**Remember: Better to delay a release for proper testing than to rush and cause user issues.**