# GitHub Integration Guide

repo-clean provides comprehensive GitHub integration for automated repository hygiene and code quality enforcement.

## 🚀 Quick Setup

### 1. GitHub Actions Workflow
Create `.github/workflows/repo-hygiene.yml`:

```yaml
name: Repository Hygiene & Code Quality
on:
  push:
    branches: [ main, develop ]
  pull_request:
    branches: [ main ]

jobs:
  hygiene-check:
    runs-on: ubuntu-latest

    steps:
    - name: Checkout code
      uses: actions/checkout@v4

    - name: Setup Python
      uses: actions/setup-python@v4
      with:
        python-version: '3.11'

    - name: Setup Node.js (for JS linting)
      uses: actions/setup-node@v3
      with:
        node-version: '18'

    - name: Install repo-clean
      run: pip install repo-clean

    - name: Install ecosystem linters
      run: |
        # Python linters
        pip install pylint black mypy flake8 isort bandit

        # JavaScript linters
        npm install -g eslint prettier @typescript-eslint/parser

    - name: Run repository hygiene scan
      run: repo-clean scan --format json --output hygiene-report.json

    - name: Run code quality linting
      run: repo-clean lint --format json --output linting-report.json

    - name: Upload hygiene report
      uses: actions/upload-artifact@v3
      if: always()
      with:
        name: hygiene-report
        path: |
          hygiene-report.json
          linting-report.json

    - name: Comment PR with results
      if: github.event_name == 'pull_request'
      uses: actions/github-script@v6
      with:
        script: |
          const fs = require('fs');

          // Read hygiene report
          const hygieneReport = JSON.parse(fs.readFileSync('hygiene-report.json', 'utf8'));
          const lintingReport = JSON.parse(fs.readFileSync('linting-report.json', 'utf8'));

          // Generate comment
          let comment = '## 🧹 Repository Hygiene Report\n\n';

          // Hygiene summary
          const totalHygieneIssues = Object.values(hygieneReport).flat().length;
          comment += `### 📊 Hygiene Issues: ${totalHygieneIssues}\n`;

          if (totalHygieneIssues > 0) {
            for (const [category, issues] of Object.entries(hygieneReport)) {
              if (issues.length > 0) {
                comment += `- **${category.replace('_', ' ')}**: ${issues.length} issues\n`;
              }
            }
          } else {
            comment += '✅ No hygiene issues found!\n';
          }

          // Linting summary
          comment += '\n### 🧪 Code Quality\n';
          let totalLintIssues = 0;
          let fixableIssues = 0;

          for (const [ecosystem, linters] of Object.entries(lintingReport)) {
            if (ecosystem === 'custom_quality') continue;

            for (const [linter, result] of Object.entries(linters)) {
              const issueCount = result.issues?.length || 0;
              totalLintIssues += issueCount;

              if (['prettier', 'black', 'gofmt', 'rustfmt'].includes(linter)) {
                fixableIssues += issueCount;
              }
            }
          }

          comment += `- **Total Issues**: ${totalLintIssues}\n`;
          comment += `- **Auto-fixable**: ${fixableIssues} (formatting only)\n`;
          comment += `- **Manual Review**: ${totalLintIssues - fixableIssues}\n\n`;

          if (totalHygieneIssues > 0 || totalLintIssues > 0) {
            comment += '### 💡 Next Steps\n';
            comment += '- Review the uploaded artifacts for detailed reports\n';
            if (fixableIssues > 0) {
              comment += '- Run `repo-clean lint --fix` to auto-fix formatting issues\n';
            }
            comment += '- Run `repo-clean clean --preview` to see proposed hygiene fixes\n';
          }

          // Post comment
          github.rest.issues.createComment({
            issue_number: context.issue.number,
            owner: context.repo.owner,
            repo: context.repo.repo,
            body: comment
          });
```

### 2. Pre-commit Hook Integration
Create `.github/hooks/pre-commit.yml`:

```yaml
repos:
  - repo: local
    hooks:
      - id: repo-clean-hygiene
        name: Repository hygiene check
        entry: repo-clean scan --exit-code
        language: system
        pass_filenames: false

      - id: repo-clean-lint
        name: Code quality linting
        entry: repo-clean lint --preview-fixes
        language: system
        pass_filenames: false
```

## 🔧 Advanced GitHub Integrations

### Repository Templates with repo-clean
Create a `.github/template-cleanup.yml` for repository templates:

```yaml
name: Template Cleanup
on:
  push:
    paths:
      - '.github/template-cleanup.yml'

jobs:
  cleanup:
    runs-on: ubuntu-latest
    steps:
    - uses: actions/checkout@v4

    - name: Install repo-clean
      run: pip install repo-clean

    - name: Clean template artifacts
      run: |
        repo-clean clean --backup-files --force
        repo-clean clean --bloat-directories --force

    - name: Generate .gitignore
      run: repo-clean gitignore --generate --ecosystem auto-detect

    - name: Commit cleanup
      run: |
        git config --local user.email "action@github.com"
        git config --local user.name "GitHub Action"
        git add .
        git commit -m "🧹 Template cleanup via repo-clean" || exit 0
        git push
```

### Issue Templates for Hygiene Reports
Create `.github/ISSUE_TEMPLATE/hygiene-report.md`:

```markdown
---
name: Repository Hygiene Report
about: Report repository hygiene issues found by repo-clean
title: '🧹 Hygiene Issues Found'
labels: hygiene, maintenance
assignees: ''
---

## Hygiene Report

**Generated by**: `repo-clean scan`
**Date**: <!-- Date -->
**Branch**: <!-- Branch name -->

### Issues Found

<!-- Paste repo-clean scan output here -->

### Proposed Fixes

- [ ] Remove backup files: `repo-clean clean --backup-files`
- [ ] Fix naming conventions: `repo-clean rename --interactive`
- [ ] Clean bloat directories: `repo-clean clean --bloat-directories`
- [ ] Add missing .gitignore patterns
- [ ] Fix code quality issues: `repo-clean lint --fix`

### Impact Assessment

- **Severity**: Low/Medium/High
- **Files affected**: <!-- Number -->
- **Size reduction**: <!-- MB saved -->

### Action Items

- [ ] Review proposed changes
- [ ] Apply safe fixes
- [ ] Update team guidelines
- [ ] Add to CI/CD pipeline
```

### Automated Hygiene Bot
Create `.github/workflows/hygiene-bot.yml`:

```yaml
name: Hygiene Bot
on:
  schedule:
    - cron: '0 2 * * 1'  # Weekly on Monday 2AM
  workflow_dispatch:

jobs:
  hygiene-maintenance:
    runs-on: ubuntu-latest
    permissions:
      contents: write
      pull-requests: write

    steps:
    - uses: actions/checkout@v4
      with:
        token: ${{ secrets.GITHUB_TOKEN }}

    - name: Setup repo-clean
      run: pip install repo-clean

    - name: Run hygiene scan
      id: scan
      run: |
        repo-clean scan --format json --output scan-results.json

        # Check if issues found
        ISSUE_COUNT=$(jq '[.[] | length] | add' scan-results.json)
        echo "issues_found=$ISSUE_COUNT" >> $GITHUB_OUTPUT

    - name: Create hygiene fix PR
      if: steps.scan.outputs.issues_found > 0
      run: |
        # Create branch
        git checkout -b automated-hygiene-fixes-$(date +%Y%m%d)

        # Apply safe fixes only
        repo-clean clean --backup-files --force || true
        repo-clean lint --fix --force || true

        # Check if changes made
        if git diff --quiet; then
          echo "No changes to commit"
          exit 0
        fi

        # Commit changes
        git config --local user.email "hygiene-bot@github.actions"
        git config --local user.name "Hygiene Bot"
        git add .
        git commit -m "🧹 Automated hygiene fixes

        - Remove backup files
        - Fix code formatting
        - Applied safe automatic fixes only

        Generated by repo-clean automated workflow"

        # Push branch
        git push origin HEAD

        # Create PR
        gh pr create \
          --title "🧹 Automated Repository Hygiene Fixes" \
          --body "$(cat <<EOF
        ## Automated Hygiene Maintenance

        This PR contains safe, automated fixes for repository hygiene issues:

        ### Changes Made
        - ✅ Removed backup files (.backup, .bak, .old)
        - ✅ Fixed code formatting (prettier, black, gofmt)
        - ✅ Applied safe linting fixes

        ### Manual Review Required
        Please review the scan results artifact for issues requiring manual attention.

        ### Safety
        - Only safe, non-logic changes applied
        - All changes are reversible
        - No structural code modifications

        **Generated by**: repo-clean hygiene bot
        **Scan Results**: See workflow artifacts
        EOF
        )" \
          --label "hygiene,automated"
      env:
        GITHUB_TOKEN: ${{ secrets.GITHUB_TOKEN }}
```

## 📊 Repository Insights & Badges

### Status Badges
Add to your README.md:

```markdown
[![Repository Hygiene](https://github.com/username/repo/actions/workflows/repo-hygiene.yml/badge.svg)](https://github.com/username/repo/actions/workflows/repo-hygiene.yml)

[![Code Quality](https://img.shields.io/badge/repo--clean-passing-green)](https://github.com/pvestal/repo-clean)
```

### Hygiene Dashboard Action
Create `.github/workflows/hygiene-dashboard.yml`:

```yaml
name: Hygiene Dashboard
on:
  push:
    branches: [ main ]
  schedule:
    - cron: '0 6 * * *'  # Daily at 6AM

jobs:
  dashboard:
    runs-on: ubuntu-latest
    steps:
    - uses: actions/checkout@v4

    - name: Generate hygiene dashboard
      run: |
        pip install repo-clean

        # Generate reports
        repo-clean scan --format json > hygiene.json
        repo-clean lint --format json > linting.json
        repo-clean report --format html > hygiene-dashboard.html

    - name: Deploy to GitHub Pages
      uses: peaceiris/actions-gh-pages@v3
      with:
        github_token: ${{ secrets.GITHUB_TOKEN }}
        publish_dir: .
        keep_files: true
        publish_branch: gh-pages
        destination_dir: hygiene-dashboard
```

## 🔐 Security & Compliance

### Security Scanning Integration
```yaml
name: Security & Hygiene
on: [push, pull_request]

jobs:
  security-hygiene:
    runs-on: ubuntu-latest
    steps:
    - uses: actions/checkout@v4

    - name: Security scan with repo-clean
      run: |
        pip install repo-clean bandit safety

        # Custom security patterns
        repo-clean lint --linters bandit

        # Check for secrets in backup files
        repo-clean scan --types backup_files | \
          grep -E '(password|key|token|secret)' || true
```

### License Compliance
```yaml
- name: License compliance check
  run: |
    # Check for proper LICENSE file
    repo-clean scan --check-documentation

    # Verify no proprietary files in open source
    repo-clean scan --types large_files | \
      grep -E '\.(exe|dll|so|dylib)$' && exit 1 || true
```

## 🔄 Workflow Templates

### Complete CI/CD Pipeline
```yaml
name: Complete Repository Health Pipeline
on:
  push:
  pull_request:
  schedule:
    - cron: '0 3 * * 1'

jobs:
  health-check:
    runs-on: ubuntu-latest
    outputs:
      hygiene-issues: ${{ steps.hygiene.outputs.issue-count }}
      lint-issues: ${{ steps.lint.outputs.issue-count }}

    steps:
    - uses: actions/checkout@v4

    - name: Setup environment
      uses: ./.github/actions/setup-repo-clean

    - name: Hygiene scan
      id: hygiene
      run: |
        repo-clean scan --format json --output hygiene.json
        ISSUES=$(jq '[.[] | length] | add' hygiene.json)
        echo "issue-count=$ISSUES" >> $GITHUB_OUTPUT

    - name: Code quality lint
      id: lint
      run: |
        repo-clean lint --format json --output lint.json
        # Count issues across all linters
        ISSUES=$(jq '[.. | .issues? | length] | add' lint.json)
        echo "issue-count=$ISSUES" >> $GITHUB_OUTPUT

  auto-fix:
    needs: health-check
    if: needs.health-check.outputs.hygiene-issues > 0 || needs.health-check.outputs.lint-issues > 0
    runs-on: ubuntu-latest

    steps:
    - uses: actions/checkout@v4

    - name: Apply safe fixes
      run: |
        pip install repo-clean

        # Apply only safe changes
        repo-clean clean --backup-files --force
        repo-clean lint --fix --force

    - name: Create fix PR
      if: github.event_name == 'schedule'
      uses: peter-evans/create-pull-request@v5
      with:
        title: "🧹 Automated hygiene & formatting fixes"
        body: |
          Automated fixes applied by repo-clean:
          - Removed backup files
          - Applied safe formatting fixes

          All changes are safe and reversible.
        branch: automated-fixes
        commit-message: "🧹 Apply automated hygiene fixes"

  quality-gate:
    needs: health-check
    runs-on: ubuntu-latest

    steps:
    - name: Quality gate
      run: |
        HYGIENE_ISSUES=${{ needs.health-check.outputs.hygiene-issues }}
        LINT_ISSUES=${{ needs.health-check.outputs.lint-issues }}

        echo "Hygiene issues: $HYGIENE_ISSUES"
        echo "Lint issues: $LINT_ISSUES"

        # Fail if too many issues (configurable thresholds)
        if [ "$HYGIENE_ISSUES" -gt 10 ]; then
          echo "❌ Too many hygiene issues: $HYGIENE_ISSUES > 10"
          exit 1
        fi

        if [ "$LINT_ISSUES" -gt 50 ]; then
          echo "❌ Too many lint issues: $LINT_ISSUES > 50"
          exit 1
        fi

        echo "✅ Quality gate passed"
```

## 📈 Metrics & Reporting

### Repository Health Metrics
```yaml
- name: Generate metrics
  run: |
    # Historical tracking
    repo-clean scan --format json > "reports/hygiene-$(date +%Y%m%d).json"
    repo-clean lint --format json > "reports/lint-$(date +%Y%m%d).json"

    # Trend analysis
    python scripts/analyze-trends.py reports/

- name: Update repository insights
  run: |
    # Update README with current stats
    HYGIENE_SCORE=$(repo-clean scan --score)
    LINT_SCORE=$(repo-clean lint --score)

    sed -i "s/Hygiene: [0-9]*%/Hygiene: ${HYGIENE_SCORE}%/" README.md
    sed -i "s/Quality: [0-9]*%/Quality: ${LINT_SCORE}%/" README.md
```

## 🎯 Integration Benefits

### For Development Teams
- **Automated quality enforcement**
- **Consistent coding standards**
- **Reduced manual review overhead**
- **Educational feedback in PRs**

### For Project Maintainers
- **Repository health visibility**
- **Automated maintenance PRs**
- **Quality metrics tracking**
- **Contributor guidance**

### For Enterprise
- **Compliance enforcement**
- **Security scanning integration**
- **Multi-repository governance**
- **Audit trail maintenance**

---

**repo-clean's GitHub integration transforms repository maintenance from manual chore to automated excellence.**