# GitIgnore Management Guide

repo-clean provides intelligent `.gitignore` file generation and management, helping you maintain proper repository hygiene across different programming ecosystems.

## 🎯 Overview

The gitignore module automatically:
- **Detects programming ecosystems** in your repository (Python, JavaScript, Go, Rust, Java)
- **Analyzes existing .gitignore coverage** with percentage scoring
- **Generates comprehensive patterns** from built-in templates and GitHub's official templates
- **Safely merges new patterns** with existing .gitignore files
- **Provides intelligent recommendations** for missing patterns

## 🚀 Quick Start

### Analyze Current .gitignore
```bash
# Check coverage and identify gaps
repo-clean gitignore --analyze
```

**Example Output:**
```
📊 .gitignore Analysis Results:
   • File exists: ✅
   • Size: 1,247 bytes (89 lines)
   • Patterns: 82
   • Coverage score: 75%
   • Missing ecosystems: javascript, rust

💡 Suggestions:
   • Missing patterns for: javascript, rust
   • Consider updating .gitignore with modern patterns
```

### Generate New .gitignore
```bash
# Create comprehensive .gitignore (auto-detect ecosystems)
repo-clean gitignore --generate

# Specify ecosystems manually
repo-clean gitignore --generate --ecosystems python javascript go

# Include custom patterns
repo-clean gitignore --generate --custom-patterns "*.local" "tmp/" "cache/"
```

### Update Existing .gitignore
```bash
# Merge new patterns with existing file (recommended)
repo-clean gitignore --update

# Preview what would be added
repo-clean gitignore --preview
```

## 🛠️ Command Reference

### Basic Commands

| Command | Description |
|---------|-------------|
| `--analyze` | Analyze existing .gitignore for coverage and gaps |
| `--generate` | Create new .gitignore (overwrites existing) |
| `--update` | Merge new patterns with existing .gitignore |
| `--preview` | Show what would be generated without making changes |

### Options

| Option | Description | Default |
|--------|-------------|---------|
| `--ecosystems` | Specify ecosystems: `python`, `javascript`, `go`, `rust`, `java`, `auto-detect` | `auto-detect` |
| `--no-github-templates` | Don't fetch official templates from GitHub API | `false` |
| `--backup` | Create backup of existing .gitignore | `true` |
| `--custom-patterns` | Add custom patterns to the generated file | None |

## 🔍 Ecosystem Detection

repo-clean automatically detects programming ecosystems based on:

### File Indicators
- **Python**: `requirements.txt`, `setup.py`, `pyproject.toml`, `Pipfile`, `.py` files
- **JavaScript**: `package.json`, `yarn.lock`, `package-lock.json`, `.js/.ts/.jsx/.tsx` files
- **Go**: `go.mod`, `go.sum`, `Gopkg.toml`, `.go` files
- **Rust**: `Cargo.toml`, `Cargo.lock`, `.rs` files
- **Java**: `pom.xml`, `build.gradle`, `gradle.properties`, `.java/.kt/.scala` files

### Smart Sampling
- Analyzes first 500 files to avoid performance issues in large repositories
- Prioritizes root-level indicator files for accurate detection
- Falls back to file extension analysis for comprehensive coverage

## 📝 Generated Patterns

### Common Patterns (All Projects)
```gitignore
# Operating System
.DS_Store
.DS_Store?
._*
Thumbs.db
ehthumbs.db

# Editors & IDEs
.vscode/
.idea/
*.iml
*.swp
*.swo
*~

# Temporary Files
*.tmp
*.temp
*.bak
*.old
```

### Python Ecosystem
```gitignore
# Python
__pycache__/
*.py[cod]
*.so
.Python
build/
dist/
*.egg-info/
.venv/
venv/
.env
.coverage
.pytest_cache/
.mypy_cache/
```

### JavaScript Ecosystem
```gitignore
# JavaScript/Node.js
node_modules/
npm-debug.log*
yarn-error.log*
.eslintcache
.parcel-cache
.next/
dist/
build/
coverage/
.env.local
.env.production.local
```

### Go Ecosystem
```gitignore
# Go
*.exe
*.test
*.out
vendor/
go.work
```

### Rust Ecosystem
```gitignore
# Rust
target/
Cargo.lock
**/*.rs.bk
*.pdb
```

### Java Ecosystem
```gitignore
# Java
*.class
*.jar
*.war
*.ear
.gradle/
build/
target/
*.log
```

## 🔄 Workflow Examples

### New Project Setup
```bash
# Start new project
mkdir my-project && cd my-project
git init

# Auto-generate .gitignore based on your code
repo-clean gitignore --generate

# Result: Intelligent .gitignore ready for your project
```

### Existing Project Audit
```bash
# Check current .gitignore quality
repo-clean gitignore --analyze

# See what's missing
repo-clean gitignore --preview

# Update with new patterns
repo-clean gitignore --update
```

### Multi-Ecosystem Project
```bash
# Full-stack web app (Python backend + React frontend)
repo-clean gitignore --generate --ecosystems python javascript

# Add custom deployment patterns
repo-clean gitignore --update --custom-patterns "deploy/" "*.env.staging"
```

### CI/CD Integration
```bash
# Check .gitignore coverage in CI
repo-clean gitignore --analyze

# Auto-update .gitignore weekly
repo-clean gitignore --update --backup
```

## 📊 Coverage Scoring

The coverage score (0-100%) indicates how well your .gitignore covers detected ecosystems:

- **90-100%**: Excellent - Comprehensive coverage
- **70-89%**: Good - Minor gaps, consider updating
- **50-69%**: Fair - Missing important patterns
- **0-49%**: Poor - Significant gaps, update recommended

### Scoring Algorithm
```
Coverage = (Covered Ecosystems / Detected Ecosystems) × 100

For each ecosystem:
- Covered if ≥30% of essential patterns are present
- Essential patterns: Core files like __pycache__, node_modules, target/
```

## 🛡️ Safety Features

### Backup Protection
```bash
# Automatic backup before updates (default: enabled)
repo-clean gitignore --update
# Creates: .gitignore.backup

# Disable backup (not recommended)
repo-clean gitignore --update --no-backup
```

### Merge Mode (Default)
```bash
# Safely merges new patterns with existing content
repo-clean gitignore --update

# Preserves:
# - Existing patterns
# - Custom comments
# - Manual additions
# - File structure
```

### Preview Mode
```bash
# See exactly what will be generated
repo-clean gitignore --preview

# Review before committing
repo-clean gitignore --preview --ecosystems python javascript > preview.txt
```

## 🌐 GitHub Templates Integration

repo-clean fetches official templates from GitHub's API for enhanced coverage:

### Available Templates
- **Python**: Comprehensive Python ecosystem patterns
- **Node**: JavaScript/TypeScript and Node.js patterns
- **Go**: Go language and tooling patterns
- **Rust**: Rust ecosystem and Cargo patterns
- **Java**: Java, Maven, and Gradle patterns

### Usage
```bash
# Include GitHub templates (default)
repo-clean gitignore --generate

# Use only built-in patterns (offline)
repo-clean gitignore --generate --no-github-templates
```

### API Fallback
- Falls back to built-in patterns if GitHub API is unavailable
- Caches templates for improved performance
- Handles rate limiting gracefully

## 🔧 Advanced Usage

### Custom Pattern Integration
```bash
# Add project-specific patterns
repo-clean gitignore --update --custom-patterns \
  "*.local" \
  "config/secrets.json" \
  "deployment/*.key" \
  "cache/" \
  "logs/*.log"
```

### Ecosystem Override
```bash
# Force specific ecosystems (ignore auto-detection)
repo-clean gitignore --generate --ecosystems python go rust

# Single ecosystem focus
repo-clean gitignore --generate --ecosystems javascript
```

### CI/CD Automation
```bash
#!/bin/bash
# Weekly .gitignore maintenance script

# Check coverage
COVERAGE=$(repo-clean gitignore --analyze | grep "Coverage score" | cut -d: -f2 | tr -d ' %')

if [ "$COVERAGE" -lt 80 ]; then
    echo "Low .gitignore coverage: ${COVERAGE}%"

    # Update and create PR
    repo-clean gitignore --update

    if git diff --quiet .gitignore; then
        echo "No updates needed"
    else
        git add .gitignore
        git commit -m "📝 Update .gitignore patterns (coverage: ${COVERAGE}%)"
        git push origin automated-gitignore-update
    fi
fi
```

## 🐛 Troubleshooting

### Common Issues

**1. No ecosystems detected**
```bash
# Problem: Empty repository or no code files
# Solution: Manually specify ecosystems
repo-clean gitignore --generate --ecosystems python javascript
```

**2. GitHub API rate limiting**
```bash
# Problem: Too many API requests
# Solution: Use offline mode
repo-clean gitignore --generate --no-github-templates
```

**3. Existing patterns not preserved**
```bash
# Problem: Used --generate instead of --update
# Solution: Use merge mode
repo-clean gitignore --update  # Preserves existing content
```

**4. Low coverage score despite complete .gitignore**
```bash
# Check what ecosystems are detected
repo-clean gitignore --analyze

# May detect ecosystems you don't actively use
# Score reflects coverage of detected ecosystems
```

### Debug Information

```bash
# Verbose ecosystem detection
repo-clean gitignore --preview --ecosystems auto-detect

# Manual ecosystem verification
ls -la | grep -E "\.(py|js|go|rs|java)$"
ls -la | grep -E "(package\.json|go\.mod|Cargo\.toml|pom\.xml)"
```

## 📚 Best Practices

### 1. Regular Maintenance
```bash
# Monthly .gitignore health check
repo-clean gitignore --analyze

# Update when adding new tools/frameworks
repo-clean gitignore --update
```

### 2. Team Consistency
```bash
# Standardize .gitignore across team repositories
repo-clean gitignore --generate --ecosystems python javascript
git add .gitignore
git commit -m "📝 Standardize .gitignore patterns"
```

### 3. Project Lifecycle
```bash
# New project: Generate comprehensive .gitignore
repo-clean gitignore --generate

# Mature project: Update carefully
repo-clean gitignore --preview  # Review first
repo-clean gitignore --update   # Then apply
```

### 4. Documentation
```bash
# Document custom patterns in .gitignore
echo "# Project-specific patterns" >> .gitignore
echo "config/local.env" >> .gitignore
echo "uploads/" >> .gitignore

# Then run update to organize
repo-clean gitignore --update
```

## 🎯 Integration with repo-clean Ecosystem

### Hygiene Scanning
```bash
# .gitignore gaps are detected in hygiene scans
repo-clean scan --types gitignore_gaps

# Full repository health includes .gitignore analysis
repo-clean report
```

### Cleanup Workflow
```bash
# Complete repository hygiene workflow
repo-clean scan                    # Find issues
repo-clean gitignore --update      # Fix .gitignore gaps
repo-clean clean --backup-files    # Remove ignored files
repo-clean lint --fix             # Fix code quality
repo-clean report                  # Verify improvements
```

---

**repo-clean's .gitignore management ensures your repository maintains proper hygiene standards across all programming ecosystems.**