"""
Repository cleaner for fixing hygiene issues
"""

from pathlib import Path
from typing import Dict, List, Optional
import re
from ..utils.safety import SafetyManager
from ..utils.errors import ErrorHandler, RepoCleanError, ErrorCategory, ErrorSeverity, ErrorContext
from .scanner import RepositoryScanner


class RepositoryCleaner:
    """Cleans repository hygiene issues safely"""

    def __init__(self, repository_path: Path, safety_manager: SafetyManager, error_handler: ErrorHandler):
        self.repository_path = repository_path
        self.safety_manager = safety_manager
        self.error_handler = error_handler
        self.scanner = RepositoryScanner(repository_path)

    def clean_repository(self, backup_files: bool = False, bloat_directories: bool = False,
                        interactive: bool = False, force: bool = False) -> Dict:
        """Clean repository with specified options"""
        results = {
            "files_cleaned": 0,
            "directories_removed": 0,
            "space_freed": 0,
            "backups_created": 0,
            "operations": []
        }

        try:
            # Scan for issues first
            scan_results = self.scanner.scan_repository()

            if backup_files and "backup_files" in scan_results:
                backup_result = self._clean_backup_files(
                    scan_results["backup_files"],
                    interactive=interactive,
                    force=force
                )
                results["files_cleaned"] += backup_result["files_cleaned"]
                results["space_freed"] += backup_result["space_freed"]
                results["backups_created"] += backup_result["backups_created"]
                results["operations"].extend(backup_result["operations"])

            if bloat_directories and "bloat_directories" in scan_results:
                bloat_result = self._clean_bloat_directories(
                    scan_results["bloat_directories"],
                    interactive=interactive,
                    force=force
                )
                results["directories_removed"] += bloat_result["directories_removed"]
                results["space_freed"] += bloat_result["space_freed"]
                results["backups_created"] += bloat_result["backups_created"]
                results["operations"].extend(bloat_result["operations"])

            return results

        except Exception as e:
            if isinstance(e, RepoCleanError):
                raise
            raise RepoCleanError(
                f"Repository cleaning failed: {e}",
                ErrorCategory.INTERNAL,
                ErrorSeverity.HIGH,
                ErrorContext(operation="clean_repository"),
                cause=e
            )

    def _clean_backup_files(self, backup_files: List[Dict], interactive: bool = False, force: bool = False) -> Dict:
        """Clean backup files safely"""
        results = {
            "files_cleaned": 0,
            "space_freed": 0,
            "backups_created": 0,
            "operations": []
        }

        for file_info in backup_files:
            file_path = self.repository_path / file_info["path"]

            if not file_path.exists():
                continue

            # Interactive confirmation
            if interactive and not force:
                file_size = file_path.stat().st_size
                response = input(f"Delete {file_path} ({self._format_size(file_size)})? [y/N/a/q]: ").lower()

                if response == 'q':
                    break
                elif response == 'a':
                    force = True  # Yes to all remaining
                elif response not in ['y', 'yes']:
                    continue

            try:
                # Get file size before deletion
                file_size = file_path.stat().st_size

                # Safely delete with backup
                success = self.safety_manager.safe_delete(
                    file_path,
                    reason=f"Backup file cleanup: {file_info.get('reason', 'Unknown')}"
                )

                if success:
                    results["files_cleaned"] += 1
                    results["space_freed"] += file_size
                    results["backups_created"] += 1
                    results["operations"].append({
                        "type": "delete",
                        "path": str(file_path),
                        "size": file_size,
                        "reason": file_info.get("reason", "Backup file")
                    })

            except Exception as e:
                self.error_handler.handle_error(RepoCleanError(
                    f"Failed to delete backup file {file_path}: {e}",
                    ErrorCategory.FILESYSTEM,
                    ErrorSeverity.MEDIUM,
                    ErrorContext(
                        operation="delete_backup_file",
                        file_path=str(file_path)
                    ),
                    cause=e
                ))
                continue

        return results

    def _clean_bloat_directories(self, bloat_dirs: List[Dict], interactive: bool = False, force: bool = False) -> Dict:
        """Clean bloat directories safely"""
        results = {
            "directories_removed": 0,
            "space_freed": 0,
            "backups_created": 0,
            "operations": []
        }

        for dir_info in bloat_dirs:
            dir_path = self.repository_path / dir_info["path"]

            if not dir_path.exists():
                continue

            # Interactive confirmation
            if interactive and not force:
                dir_size = dir_info.get("size", 0)
                response = input(f"Remove directory {dir_path} ({self._format_size(dir_size)})? [y/N/a/q]: ").lower()

                if response == 'q':
                    break
                elif response == 'a':
                    force = True  # Yes to all remaining
                elif response not in ['y', 'yes']:
                    continue

            try:
                # For directories, we'll create a backup archive
                dir_size = dir_info.get("size", 0)

                # Create backup archive of directory
                backup_path = self._backup_directory(dir_path, dir_info.get("reason", "Bloat directory"))

                # Remove directory
                import shutil
                shutil.rmtree(dir_path)

                results["directories_removed"] += 1
                results["space_freed"] += dir_size
                results["backups_created"] += 1
                results["operations"].append({
                    "type": "remove_directory",
                    "path": str(dir_path),
                    "size": dir_size,
                    "backup": str(backup_path),
                    "reason": dir_info.get("reason", "Bloat directory")
                })

            except Exception as e:
                self.error_handler.handle_error(RepoCleanError(
                    f"Failed to remove bloat directory {dir_path}: {e}",
                    ErrorCategory.FILESYSTEM,
                    ErrorSeverity.MEDIUM,
                    ErrorContext(
                        operation="remove_bloat_directory",
                        file_path=str(dir_path)
                    ),
                    cause=e
                ))
                continue

        return results

    def _backup_directory(self, dir_path: Path, reason: str = "") -> Path:
        """Create backup archive of directory before removal"""
        import shutil

        # Create unique backup filename
        relative_path = dir_path.relative_to(self.repository_path)
        backup_filename = f"{self.safety_manager.session_id}_{str(relative_path).replace('/', '_')}.tar.gz"
        backup_path = self.safety_manager.backup_dir / backup_filename

        try:
            # Create tar.gz archive
            shutil.make_archive(
                str(backup_path.with_suffix('')),
                'gztar',
                root_dir=dir_path.parent,
                base_dir=dir_path.name
            )

            return backup_path.with_suffix('.tar.gz')

        except Exception as e:
            raise RepoCleanError(
                f"Failed to backup directory {dir_path}: {e}",
                ErrorCategory.FILESYSTEM,
                ErrorSeverity.HIGH,
                ErrorContext(
                    operation="backup_directory",
                    file_path=str(dir_path)
                ),
                cause=e
            )

    def fix_naming_conventions(self, patterns: List[str], interactive: bool = False) -> Dict:
        """Fix naming convention issues"""
        results = {
            "files_renamed": 0,
            "backups_created": 0,
            "operations": []
        }

        try:
            # Scan for naming issues
            scan_results = self.scanner.scan_repository(issue_types=["naming_conventions"])

            if "naming_conventions" not in scan_results:
                return results

            naming_issues = scan_results["naming_conventions"]

            for issue in naming_issues:
                file_path = self.repository_path / issue["path"]

                if not file_path.exists():
                    continue

                # Generate suggested name
                suggested_name = self._suggest_better_name(file_path.name, patterns)
                suggested_path = file_path.parent / suggested_name

                # Interactive confirmation
                if interactive:
                    print(f"Rename: {file_path.name} → {suggested_name}")
                    response = input("Apply this rename? [y/N/s/q]: ").lower()

                    if response == 'q':
                        break
                    elif response == 's':
                        # Skip this file
                        continue
                    elif response not in ['y', 'yes']:
                        # Allow custom name
                        custom_name = input(f"Enter custom name (or press Enter to skip): ").strip()
                        if custom_name:
                            suggested_path = file_path.parent / custom_name
                        else:
                            continue

                # Check if destination exists
                if suggested_path.exists():
                    print(f"Warning: {suggested_path} already exists, skipping rename")
                    continue

                try:
                    # Safely rename with backup
                    success = self.safety_manager.safe_rename(
                        file_path,
                        suggested_path,
                        reason=f"Naming convention fix: {issue.get('reason', 'Unknown')}"
                    )

                    if success:
                        results["files_renamed"] += 1
                        results["backups_created"] += 1
                        results["operations"].append({
                            "type": "rename",
                            "from": str(file_path),
                            "to": str(suggested_path),
                            "reason": issue.get("reason", "Naming convention")
                        })

                except Exception as e:
                    self.error_handler.handle_error(RepoCleanError(
                        f"Failed to rename {file_path}: {e}",
                        ErrorCategory.FILESYSTEM,
                        ErrorSeverity.MEDIUM,
                        ErrorContext(
                            operation="rename_file",
                            file_path=str(file_path)
                        ),
                        cause=e
                    ))
                    continue

            return results

        except Exception as e:
            if isinstance(e, RepoCleanError):
                raise
            raise RepoCleanError(
                f"Naming convention fix failed: {e}",
                ErrorCategory.INTERNAL,
                ErrorSeverity.HIGH,
                ErrorContext(operation="fix_naming_conventions"),
                cause=e
            )

    def _suggest_better_name(self, filename: str, patterns: List[str]) -> str:
        """Suggest better naming for problematic files"""
        # Remove problematic prefixes
        prefixes_to_remove = [
            'ENHANCED_', 'WORKING_', 'FIXED_', 'FINAL_', 'NEW_', 'OLD_',
            'TEMP_', 'TEST_', 'COPY_', 'BACKUP_'
        ]

        # Remove problematic suffixes
        suffixes_to_remove = [
            '_COPY', '_OLD', '_NEW', '_BACKUP', '_WORKING', '_FIXED'
        ]

        cleaned_name = filename

        # Remove prefixes
        for prefix in prefixes_to_remove:
            if cleaned_name.startswith(prefix):
                cleaned_name = cleaned_name[len(prefix):]
                break

        # Remove suffixes (before file extension)
        name_parts = cleaned_name.rsplit('.', 1)
        base_name = name_parts[0]
        extension = '.' + name_parts[1] if len(name_parts) > 1 else ''

        for suffix in suffixes_to_remove:
            if base_name.endswith(suffix):
                base_name = base_name[:-len(suffix)]
                break

        # Ensure we have a meaningful name
        if not base_name or base_name.lower() in ['untitled', 'file', 'temp']:
            base_name = 'cleaned_file'

        return base_name + extension

    def _format_size(self, size: int) -> str:
        """Format file size in human readable format"""
        for unit in ['B', 'KB', 'MB', 'GB']:
            if size < 1024:
                return f"{size:.1f}{unit}"
            size /= 1024
        return f"{size:.1f}TB"

    def preview_changes(self, backup_files: bool = False, bloat_directories: bool = False) -> Dict:
        """Preview what changes would be made without actually performing them"""
        preview = {
            "total_files": 0,
            "total_directories": 0,
            "total_space": 0,
            "operations": []
        }

        try:
            # Scan for issues
            scan_results = self.scanner.scan_repository()

            if backup_files and "backup_files" in scan_results:
                for file_info in scan_results["backup_files"]:
                    file_path = self.repository_path / file_info["path"]
                    if file_path.exists():
                        file_size = file_path.stat().st_size
                        preview["total_files"] += 1
                        preview["total_space"] += file_size
                        preview["operations"].append({
                            "type": "delete",
                            "path": str(file_path),
                            "size": file_size,
                            "size_human": self._format_size(file_size),
                            "reason": file_info.get("reason", "Backup file")
                        })

            if bloat_directories and "bloat_directories" in scan_results:
                for dir_info in scan_results["bloat_directories"]:
                    dir_path = self.repository_path / dir_info["path"]
                    if dir_path.exists():
                        dir_size = dir_info.get("size", 0)
                        preview["total_directories"] += 1
                        preview["total_space"] += dir_size
                        preview["operations"].append({
                            "type": "remove_directory",
                            "path": str(dir_path),
                            "size": dir_size,
                            "size_human": self._format_size(dir_size),
                            "reason": dir_info.get("reason", "Bloat directory")
                        })

            preview["total_space_human"] = self._format_size(preview["total_space"])
            return preview

        except Exception as e:
            raise RepoCleanError(
                f"Preview generation failed: {e}",
                ErrorCategory.INTERNAL,
                ErrorSeverity.MEDIUM,
                ErrorContext(operation="preview_changes"),
                cause=e
            )