"""
GitIgnore generator for automatic .gitignore creation and management
"""

from pathlib import Path
from typing import Dict, List, Set, Optional
import requests
from ..utils.errors import ErrorHandler, RepoCleanError, ErrorCategory, ErrorSeverity, ErrorContext


class GitIgnoreGenerator:
    """Generates and manages .gitignore files intelligently"""

    def __init__(self, repository_path: Path, error_handler: ErrorHandler):
        self.repository_path = repository_path
        self.error_handler = error_handler

        # GitHub's gitignore templates API
        self.github_api_base = "https://api.github.com/gitignore/templates"

        # Common patterns by ecosystem
        self.ecosystem_patterns = {
            "python": [
                "# Python",
                "__pycache__/",
                "*.py[cod]",
                "*$py.class",
                "*.so",
                ".Python",
                "build/",
                "develop-eggs/",
                "dist/",
                "downloads/",
                "eggs/",
                ".eggs/",
                "lib/",
                "lib64/",
                "parts/",
                "sdist/",
                "var/",
                "wheels/",
                "*.egg-info/",
                ".installed.cfg",
                "*.egg",
                "MANIFEST",
                "*.manifest",
                "*.spec",
                "pip-log.txt",
                "pip-delete-this-directory.txt",
                ".tox/",
                ".coverage",
                ".coverage.*",
                ".cache",
                "nosetests.xml",
                "coverage.xml",
                "*.cover",
                ".hypothesis/",
                ".pytest_cache/",
                "*.mo",
                "*.pot",
                "*.log",
                "local_settings.py",
                "db.sqlite3",
                ".env",
                ".venv",
                "env/",
                "venv/",
                "ENV/",
                "env.bak/",
                "venv.bak/",
                ".spyderproject",
                ".spyproject",
                ".rope_project",
                ".mypy_cache/",
                ".dmypy.json",
                "dmypy.json",
                ".pyre/",
            ],
            "javascript": [
                "# JavaScript/Node.js",
                "node_modules/",
                "npm-debug.log*",
                "yarn-debug.log*",
                "yarn-error.log*",
                "lerna-debug.log*",
                ".pnpm-debug.log*",
                "report.[0-9]*.[0-9]*.[0-9]*.[0-9]*.json",
                "pids",
                "*.pid",
                "*.seed",
                "*.pid.lock",
                "lib-cov",
                "coverage",
                "*.lcov",
                ".nyc_output",
                ".grunt",
                "bower_components",
                ".lock-wscript",
                "build/Release",
                ".npm",
                ".eslintcache",
                ".stylelintcache",
                ".rpt2_cache/",
                ".rts2_cache_cjs/",
                ".rts2_cache_es/",
                ".rts2_cache_umd/",
                ".node_repl_history",
                "*.tgz",
                ".yarn-integrity",
                ".env",
                ".env.local",
                ".env.development.local",
                ".env.test.local",
                ".env.production.local",
                ".parcel-cache",
                ".next",
                "out",
                ".nuxt",
                "dist",
                ".cache/",
                ".vuepress/dist",
                ".serverless/",
                ".fusebox/",
                ".dynamodb/",
                ".tern-port",
                ".stores",
                ".svelte-kit",
            ],
            "go": [
                "# Go",
                "*.exe",
                "*.exe~",
                "*.dll",
                "*.so",
                "*.dylib",
                "*.test",
                "*.out",
                "go.work",
                "vendor/",
            ],
            "rust": [
                "# Rust",
                "target/",
                "Cargo.lock",
                "**/*.rs.bk",
                "*.pdb",
            ],
            "java": [
                "# Java",
                "*.class",
                "*.log",
                "*.ctxt",
                ".mtj.tmp/",
                "*.jar",
                "*.war",
                "*.nar",
                "*.ear",
                "*.zip",
                "*.tar.gz",
                "*.rar",
                "hs_err_pid*",
                "replay_pid*",
                ".gradle",
                "build/",
                "!gradle/wrapper/gradle-wrapper.jar",
                "!**/src/main/**/build/",
                "!**/src/test/**/build/",
                ".gradle",
                "**/build/",
                "!src/**/build/",
                "gradle-app.setting",
                "!gradle-wrapper.jar",
                ".gradletasknamecache",
                "target/",
                "pom.xml.tag",
                "pom.xml.releaseBackup",
                "pom.xml.versionsBackup",
                "pom.xml.next",
                "release.properties",
                "dependency-reduced-pom.xml",
                "buildNumber.properties",
                ".mvn/timing.properties",
                ".mvn/wrapper/maven-wrapper.jar",
            ],
            "common": [
                "# Common",
                ".DS_Store",
                ".DS_Store?",
                "._*",
                ".Spotlight-V100",
                ".Trashes",
                "ehthumbs.db",
                "Thumbs.db",
                "*.tmp",
                "*.temp",
                "*.swp",
                "*.swo",
                "*~",
                ".vscode/",
                ".idea/",
                "*.iml",
                "*.ipr",
                "*.iws",
                ".project",
                ".classpath",
                ".c9/",
                "*.launch",
                ".settings/",
                ".metadata",
                "bin/",
                "tmp/",
                "*.tmp",
                "*.bak",
                "*.swp",
                "*~.nib",
                "local.properties",
                ".loadpath",
                ".factorypath",
                ".buildpath",
                ".target",
            ]
        }

    def detect_ecosystems(self) -> List[str]:
        """Detect what programming ecosystems are present in the repository"""
        ecosystems = set()

        # Check for common files that indicate ecosystems
        ecosystem_indicators = {
            "python": ["requirements.txt", "setup.py", "pyproject.toml", "Pipfile", "environment.yml"],
            "javascript": ["package.json", "yarn.lock", "package-lock.json", ".nvmrc"],
            "go": ["go.mod", "go.sum", "Gopkg.toml", "glide.yaml"],
            "rust": ["Cargo.toml", "Cargo.lock"],
            "java": ["pom.xml", "build.gradle", "gradle.properties", "build.xml"],
        }

        # Check for ecosystem indicator files
        for ecosystem, indicators in ecosystem_indicators.items():
            for indicator in indicators:
                if (self.repository_path / indicator).exists():
                    ecosystems.add(ecosystem)
                    break

        # Check file extensions
        extension_mapping = {
            ".py": "python",
            ".js": "javascript",
            ".ts": "javascript",
            ".jsx": "javascript",
            ".tsx": "javascript",
            ".vue": "javascript",
            ".go": "go",
            ".rs": "rust",
            ".java": "java",
            ".kt": "java",
            ".scala": "java",
        }

        # Sample files to check (avoid scanning huge repos)
        sample_files = []
        for pattern in ["**/*"]:
            files = list(self.repository_path.glob(pattern))[:500]  # Limit to first 500 files
            sample_files.extend(files)

        for file_path in sample_files:
            if file_path.is_file() and file_path.suffix in extension_mapping:
                ecosystems.add(extension_mapping[file_path.suffix])

        return list(ecosystems)

    def get_github_template(self, template_name: str) -> Optional[str]:
        """Fetch a gitignore template from GitHub's API"""
        try:
            response = requests.get(f"{self.github_api_base}/{template_name}", timeout=10)
            if response.status_code == 200:
                return response.json().get("source", "")
            return None
        except Exception as e:
            self.error_handler.handle_error(RepoCleanError(
                f"Failed to fetch GitHub template {template_name}: {e}",
                ErrorCategory.NETWORK,
                ErrorSeverity.LOW,
                ErrorContext(
                    operation="fetch_github_template",
                    template_name=template_name
                ),
                cause=e
            ))
            return None

    def generate_gitignore(self, ecosystems: Optional[List[str]] = None,
                          include_github_templates: bool = True,
                          custom_patterns: Optional[List[str]] = None) -> str:
        """Generate a comprehensive .gitignore file"""

        if ecosystems is None:
            ecosystems = self.detect_ecosystems()

        gitignore_content = []

        # Add header
        gitignore_content.extend([
            "# Generated by repo-clean",
            f"# Detected ecosystems: {', '.join(ecosystems) if ecosystems else 'none'}",
            "",
        ])

        # Add common patterns first
        if "common" in self.ecosystem_patterns:
            gitignore_content.extend(self.ecosystem_patterns["common"])
            gitignore_content.append("")

        # Add ecosystem-specific patterns
        for ecosystem in ecosystems:
            if ecosystem in self.ecosystem_patterns:
                gitignore_content.extend(self.ecosystem_patterns[ecosystem])
                gitignore_content.append("")

        # Try to fetch GitHub templates
        if include_github_templates:
            github_template_mapping = {
                "python": "Python",
                "javascript": "Node",
                "go": "Go",
                "rust": "Rust",
                "java": "Java",
            }

            for ecosystem in ecosystems:
                if ecosystem in github_template_mapping:
                    template_name = github_template_mapping[ecosystem]
                    template_content = self.get_github_template(template_name)
                    if template_content:
                        gitignore_content.extend([
                            f"# GitHub {template_name} template",
                            template_content,
                            ""
                        ])

        # Add custom patterns
        if custom_patterns:
            gitignore_content.extend([
                "# Custom patterns",
                *custom_patterns,
                ""
            ])

        return "\n".join(gitignore_content)

    def analyze_existing_gitignore(self) -> Dict:
        """Analyze existing .gitignore file for gaps and improvements"""
        gitignore_path = self.repository_path / ".gitignore"

        analysis = {
            "exists": gitignore_path.exists(),
            "size": 0,
            "lines": 0,
            "patterns": [],
            "missing_ecosystems": [],
            "suggestions": [],
            "coverage_score": 0
        }

        if not analysis["exists"]:
            analysis["suggestions"].append("No .gitignore file found - should create one")
            detected_ecosystems = self.detect_ecosystems()
            analysis["missing_ecosystems"] = detected_ecosystems
            return analysis

        try:
            content = gitignore_path.read_text(encoding='utf-8')
            analysis["size"] = len(content)
            lines = content.splitlines()
            analysis["lines"] = len(lines)

            # Extract patterns (non-comment, non-empty lines)
            patterns = []
            for line in lines:
                line = line.strip()
                if line and not line.startswith('#'):
                    patterns.append(line)
            analysis["patterns"] = patterns

            # Check coverage for detected ecosystems
            detected_ecosystems = self.detect_ecosystems()
            covered_ecosystems = []

            for ecosystem in detected_ecosystems:
                if ecosystem in self.ecosystem_patterns:
                    ecosystem_patterns = self.ecosystem_patterns[ecosystem]
                    # Check if at least 30% of ecosystem patterns are covered
                    covered_count = 0
                    for pattern in ecosystem_patterns:
                        if not pattern.startswith('#'):  # Skip comment lines
                            if pattern in patterns or any(pattern in p for p in patterns):
                                covered_count += 1

                    coverage_ratio = covered_count / len([p for p in ecosystem_patterns if not p.startswith('#')])
                    if coverage_ratio >= 0.3:
                        covered_ecosystems.append(ecosystem)

            analysis["missing_ecosystems"] = [e for e in detected_ecosystems if e not in covered_ecosystems]

            # Calculate coverage score
            if detected_ecosystems:
                analysis["coverage_score"] = int((len(covered_ecosystems) / len(detected_ecosystems)) * 100)
            else:
                analysis["coverage_score"] = 100  # No ecosystems detected

            # Generate suggestions
            if analysis["missing_ecosystems"]:
                analysis["suggestions"].append(f"Missing patterns for: {', '.join(analysis['missing_ecosystems'])}")

            if analysis["coverage_score"] < 80:
                analysis["suggestions"].append("Consider updating .gitignore with modern patterns")

            if len(patterns) < 10:
                analysis["suggestions"].append("Very minimal .gitignore - consider expanding")

        except Exception as e:
            self.error_handler.handle_error(RepoCleanError(
                f"Failed to analyze .gitignore: {e}",
                ErrorCategory.FILESYSTEM,
                ErrorSeverity.MEDIUM,
                ErrorContext(
                    operation="analyze_gitignore",
                    file_path=str(gitignore_path)
                ),
                cause=e
            ))
            analysis["suggestions"].append("Error reading .gitignore file")

        return analysis

    def update_gitignore(self, backup: bool = True, merge_mode: bool = True) -> Dict:
        """Update or create .gitignore file"""
        gitignore_path = self.repository_path / ".gitignore"
        results = {
            "created": False,
            "updated": False,
            "backup_path": None,
            "lines_added": 0,
            "ecosystems_added": []
        }

        try:
            # Detect ecosystems
            ecosystems = self.detect_ecosystems()

            if not ecosystems:
                return {"error": "No programming ecosystems detected"}

            # Backup existing file if requested
            if backup and gitignore_path.exists():
                backup_path = gitignore_path.with_suffix('.gitignore.backup')
                backup_path.write_text(gitignore_path.read_text())
                results["backup_path"] = str(backup_path)

            if merge_mode and gitignore_path.exists():
                # Merge with existing
                existing_content = gitignore_path.read_text()
                existing_patterns = set()

                for line in existing_content.splitlines():
                    line = line.strip()
                    if line and not line.startswith('#'):
                        existing_patterns.add(line)

                # Generate new patterns
                new_content_lines = []
                new_patterns_added = 0

                # Keep existing content
                new_content_lines.append(existing_content.rstrip())
                new_content_lines.append("")
                new_content_lines.append("# Added by repo-clean")

                for ecosystem in ecosystems:
                    if ecosystem in self.ecosystem_patterns:
                        ecosystem_patterns = self.ecosystem_patterns[ecosystem]
                        added_for_ecosystem = False

                        for pattern in ecosystem_patterns:
                            if pattern.startswith('#'):
                                if not added_for_ecosystem:
                                    new_content_lines.append(pattern)
                                    added_for_ecosystem = True
                            else:
                                if pattern not in existing_patterns:
                                    new_content_lines.append(pattern)
                                    new_patterns_added += 1

                        if added_for_ecosystem:
                            results["ecosystems_added"].append(ecosystem)
                            new_content_lines.append("")

                if new_patterns_added > 0:
                    gitignore_path.write_text("\n".join(new_content_lines))
                    results["updated"] = True
                    results["lines_added"] = new_patterns_added

            else:
                # Create new or replace existing
                new_content = self.generate_gitignore(ecosystems)
                gitignore_path.write_text(new_content)
                results["created"] = not gitignore_path.exists()
                results["updated"] = gitignore_path.exists()
                results["ecosystems_added"] = ecosystems
                results["lines_added"] = len(new_content.splitlines())

            return results

        except Exception as e:
            raise RepoCleanError(
                f"Failed to update .gitignore: {e}",
                ErrorCategory.FILESYSTEM,
                ErrorSeverity.HIGH,
                ErrorContext(
                    operation="update_gitignore",
                    file_path=str(gitignore_path)
                ),
                cause=e
            )

    def list_available_templates(self) -> List[str]:
        """List available GitHub gitignore templates"""
        try:
            response = requests.get(self.github_api_base, timeout=10)
            if response.status_code == 200:
                return response.json()
            return []
        except Exception as e:
            self.error_handler.handle_error(RepoCleanError(
                f"Failed to fetch available templates: {e}",
                ErrorCategory.NETWORK,
                ErrorSeverity.LOW,
                ErrorContext(operation="list_templates"),
                cause=e
            ))
            return []

    def preview_gitignore(self, ecosystems: Optional[List[str]] = None) -> str:
        """Preview what a generated .gitignore would look like"""
        if ecosystems is None:
            ecosystems = self.detect_ecosystems()

        preview = self.generate_gitignore(ecosystems, include_github_templates=False)

        # Add analysis info
        analysis = self.analyze_existing_gitignore()
        preview_header = [
            "# Preview of generated .gitignore",
            f"# Detected ecosystems: {', '.join(ecosystems)}",
            f"# Current .gitignore exists: {analysis['exists']}",
            f"# Current coverage score: {analysis['coverage_score']}%",
            "",
        ]

        return "\n".join(preview_header) + preview