"""
Repository reporter for generating hygiene reports
"""

from pathlib import Path
import json
from datetime import datetime
from typing import Dict, List, Optional
from ..utils.explanations import ExplanationEngine, ExplanationLevel
from .scanner import RepositoryScanner
from .linter import RepositoryLinter


class RepositoryReporter:
    """Generates repository hygiene reports"""

    def __init__(self, repository_path: Path, explanation_engine: ExplanationEngine):
        self.repository_path = repository_path
        self.explanation_engine = explanation_engine
        self.scanner = RepositoryScanner(repository_path)
        self.linter = RepositoryLinter(repository_path)

    def generate_report(self, format: str = "text", include_linting: bool = True) -> str:
        """Generate comprehensive repository report"""
        # Gather all data
        scan_results = self.scanner.scan_repository()

        linting_results = {}
        if include_linting:
            try:
                linting_results = self.linter.lint_repository()
            except Exception:
                # If linting fails, continue without it
                linting_results = {}

        report_data = {
            "metadata": self._generate_metadata(),
            "hygiene": scan_results,
            "linting": linting_results,
            "summary": self._generate_summary(scan_results, linting_results)
        }

        # Generate report in requested format
        if format == "json":
            return self._generate_json_report(report_data)
        elif format == "html":
            return self._generate_html_report(report_data)
        else:
            return self._generate_text_report(report_data)

    def _generate_metadata(self) -> Dict:
        """Generate report metadata"""
        return {
            "repository_path": str(self.repository_path),
            "report_generated": datetime.now().isoformat(),
            "repo_clean_version": "2.0.0",
            "is_git_repository": (self.repository_path / ".git").exists(),
            "repository_size": self._calculate_repository_size()
        }

    def _calculate_repository_size(self) -> Dict:
        """Calculate repository size statistics"""
        total_size = 0
        file_count = 0
        directory_count = 0

        try:
            for item in self.repository_path.rglob("*"):
                if item.is_file():
                    total_size += item.stat().st_size
                    file_count += 1
                elif item.is_dir():
                    directory_count += 1
        except Exception:
            pass

        return {
            "total_size": total_size,
            "total_size_human": self._format_size(total_size),
            "file_count": file_count,
            "directory_count": directory_count
        }

    def _generate_summary(self, hygiene_results: Dict, linting_results: Dict) -> Dict:
        """Generate summary statistics"""
        # Count hygiene issues
        total_hygiene_issues = 0
        hygiene_breakdown = {}

        for category, issues in hygiene_results.items():
            issue_count = len(issues) if isinstance(issues, list) else 0
            total_hygiene_issues += issue_count
            hygiene_breakdown[category] = issue_count

        # Count linting issues
        total_lint_issues = 0
        fixable_issues = 0
        lint_breakdown = {}

        for ecosystem, linters in linting_results.items():
            if ecosystem == "custom_quality":
                # Handle custom quality checks
                for check_name, check_result in linters.items():
                    issue_count = len(check_result.get("issues", []))
                    total_lint_issues += issue_count
                    lint_breakdown[f"custom_{check_name}"] = issue_count
            elif isinstance(linters, dict):
                for linter_name, linter_result in linters.items():
                    if isinstance(linter_result, dict) and "issues" in linter_result:
                        issue_count = len(linter_result["issues"])
                        total_lint_issues += issue_count
                        lint_breakdown[f"{ecosystem}_{linter_name}"] = issue_count

                        # Count fixable issues
                        if linter_name in ["prettier", "black", "gofmt", "rustfmt"]:
                            fixable_issues += issue_count

        # Calculate overall health score
        health_score = self._calculate_health_score(
            total_hygiene_issues, total_lint_issues, hygiene_results, linting_results
        )

        return {
            "total_issues": total_hygiene_issues + total_lint_issues,
            "hygiene_issues": total_hygiene_issues,
            "lint_issues": total_lint_issues,
            "fixable_issues": fixable_issues,
            "health_score": health_score,
            "hygiene_breakdown": hygiene_breakdown,
            "lint_breakdown": lint_breakdown,
            "recommendations": self._generate_recommendations(
                hygiene_results, linting_results, total_hygiene_issues, total_lint_issues
            )
        }

    def _calculate_health_score(self, hygiene_issues: int, lint_issues: int,
                               hygiene_results: Dict, linting_results: Dict) -> int:
        """Calculate overall repository health score (0-100)"""
        base_score = 100

        # Deduct points for hygiene issues
        base_score -= min(hygiene_issues * 2, 40)  # Max 40 points for hygiene

        # Deduct points for linting issues (less severe)
        base_score -= min(lint_issues * 0.5, 30)  # Max 30 points for linting

        # Bonus points for good practices
        if (self.repository_path / "README.md").exists():
            base_score += 5
        if (self.repository_path / "LICENSE").exists():
            base_score += 5
        if (self.repository_path / ".gitignore").exists():
            base_score += 5

        return max(0, min(100, int(base_score)))

    def _generate_recommendations(self, hygiene_results: Dict, linting_results: Dict,
                                 hygiene_issues: int, lint_issues: int) -> List[str]:
        """Generate actionable recommendations"""
        recommendations = []

        # Hygiene recommendations
        if "backup_files" in hygiene_results and hygiene_results["backup_files"]:
            recommendations.append("🧹 Run 'repo-clean clean --backup-files' to remove backup file clutter")

        if "bloat_directories" in hygiene_results and hygiene_results["bloat_directories"]:
            recommendations.append("📦 Run 'repo-clean clean --bloat-directories' to remove build artifacts")

        if "naming_conventions" in hygiene_results and hygiene_results["naming_conventions"]:
            recommendations.append("🏷️ Run 'repo-clean rename --interactive' to fix naming conventions")

        # Linting recommendations
        fixable_count = 0
        for ecosystem, linters in linting_results.items():
            if isinstance(linters, dict):
                for linter_name, linter_result in linters.items():
                    if linter_name in ["prettier", "black", "gofmt", "rustfmt"]:
                        fixable_count += len(linter_result.get("issues", []))

        if fixable_count > 0:
            recommendations.append(f"🔧 Run 'repo-clean lint --fix' to auto-fix {fixable_count} formatting issues")

        # General recommendations
        if hygiene_issues > 20:
            recommendations.append("📈 Consider adding repo-clean to your CI/CD pipeline for automated hygiene")

        if not (self.repository_path / ".gitignore").exists():
            recommendations.append("📄 Add a .gitignore file to prevent future hygiene issues")

        if not (self.repository_path / "README.md").exists():
            recommendations.append("📚 Add a README.md file to document your project")

        return recommendations

    def _generate_text_report(self, report_data: Dict) -> str:
        """Generate human-readable text report"""
        lines = []

        # Header
        lines.append("🧹 Repository Health Report")
        lines.append("=" * 50)
        lines.append("")

        # Metadata
        metadata = report_data["metadata"]
        lines.append(f"📁 Repository: {metadata['repository_path']}")
        lines.append(f"📊 Size: {metadata['repository_size']['total_size_human']} "
                    f"({metadata['repository_size']['file_count']} files)")
        lines.append(f"🕐 Generated: {metadata['report_generated']}")
        lines.append("")

        # Summary
        summary = report_data["summary"]
        health_score = summary["health_score"]
        health_emoji = "🟢" if health_score >= 80 else "🟡" if health_score >= 60 else "🔴"

        lines.append(f"## Overall Health: {health_emoji} {health_score}/100")
        lines.append("")
        lines.append(f"📋 Total Issues: {summary['total_issues']}")
        lines.append(f"   • Hygiene Issues: {summary['hygiene_issues']}")
        lines.append(f"   • Code Quality Issues: {summary['lint_issues']}")
        if summary['fixable_issues'] > 0:
            lines.append(f"   • Auto-fixable: {summary['fixable_issues']} 🔧")
        lines.append("")

        # Hygiene Issues Detail
        if summary['hygiene_issues'] > 0:
            lines.append("## 🧹 Hygiene Issues")
            lines.append("")

            for category, issues in report_data["hygiene"].items():
                if not issues:
                    continue

                category_name = category.replace("_", " ").title()
                lines.append(f"### {category_name} ({len(issues)} found)")

                # Show first few examples
                for i, issue in enumerate(issues[:3]):
                    path = issue.get("path", "unknown")
                    reason = issue.get("reason", "No reason provided")
                    size = issue.get("size")

                    if size:
                        size_str = f" ({self._format_size(size)})"
                    else:
                        size_str = ""

                    lines.append(f"   • {path}{size_str}")
                    lines.append(f"     └─ {reason}")

                if len(issues) > 3:
                    lines.append(f"   ... and {len(issues) - 3} more")

                lines.append("")

        # Linting Issues Detail
        if summary['lint_issues'] > 0:
            lines.append("## 🧪 Code Quality Issues")
            lines.append("")

            for ecosystem, linters in report_data["linting"].items():
                if ecosystem == "custom_quality":
                    continue

                if not isinstance(linters, dict):
                    continue

                ecosystem_issues = sum(len(result.get("issues", []))
                                     for result in linters.values()
                                     if isinstance(result, dict))

                if ecosystem_issues == 0:
                    continue

                lines.append(f"### {ecosystem.title()} ({ecosystem_issues} issues)")

                for linter_name, linter_result in linters.items():
                    if not isinstance(linter_result, dict):
                        continue

                    issues = linter_result.get("issues", [])
                    if not issues:
                        lines.append(f"   ✅ {linter_name}: No issues")
                    else:
                        fixable = " 🔧" if linter_name in ["prettier", "black", "gofmt", "rustfmt"] else ""
                        lines.append(f"   ⚠️  {linter_name}: {len(issues)} issues{fixable}")

                lines.append("")

        # Recommendations
        if summary["recommendations"]:
            lines.append("## 💡 Recommendations")
            lines.append("")
            for rec in summary["recommendations"]:
                lines.append(f"• {rec}")
            lines.append("")

        # Footer
        lines.append("---")
        lines.append("Generated by repo-clean - https://github.com/pvestal/repo-clean")

        return "\n".join(lines)

    def _generate_json_report(self, report_data: Dict) -> str:
        """Generate machine-readable JSON report"""
        return json.dumps(report_data, indent=2, default=str)

    def _generate_html_report(self, report_data: Dict) -> str:
        """Generate HTML report"""
        metadata = report_data["metadata"]
        summary = report_data["summary"]

        # Health score styling
        health_score = summary["health_score"]
        if health_score >= 80:
            health_color = "#10b981"  # green
            health_emoji = "🟢"
        elif health_score >= 60:
            health_color = "#f59e0b"  # yellow
            health_emoji = "🟡"
        else:
            health_color = "#ef4444"  # red
            health_emoji = "🔴"

        html = f"""
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Repository Health Report</title>
    <style>
        body {{
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f8fafc;
        }}
        .container {{
            background: white;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }}
        .header {{
            border-bottom: 2px solid #e5e7eb;
            padding-bottom: 20px;
            margin-bottom: 30px;
        }}
        .health-score {{
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 1.5em;
            font-weight: bold;
            color: {health_color};
        }}
        .stats-grid {{
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }}
        .stat-card {{
            background: #f8fafc;
            padding: 20px;
            border-radius: 8px;
            border-left: 4px solid #3b82f6;
        }}
        .stat-number {{
            font-size: 2em;
            font-weight: bold;
            color: #1f2937;
        }}
        .stat-label {{
            color: #6b7280;
            font-size: 0.9em;
        }}
        .issue-section {{
            margin: 30px 0;
            padding: 20px;
            background: #fef7f7;
            border-radius: 8px;
            border-left: 4px solid #ef4444;
        }}
        .issue-section.clean {{
            background: #f0fdf4;
            border-left-color: #10b981;
        }}
        .issue-list {{
            list-style: none;
            padding: 0;
        }}
        .issue-item {{
            padding: 8px 0;
            border-bottom: 1px solid #e5e7eb;
        }}
        .issue-path {{
            font-family: monospace;
            background: #f3f4f6;
            padding: 2px 6px;
            border-radius: 4px;
        }}
        .recommendations {{
            background: #eff6ff;
            border: 1px solid #bfdbfe;
            border-radius: 8px;
            padding: 20px;
            margin: 20px 0;
        }}
        .recommendation {{
            margin: 10px 0;
            padding: 8px;
            background: white;
            border-radius: 4px;
        }}
        .footer {{
            margin-top: 40px;
            padding-top: 20px;
            border-top: 1px solid #e5e7eb;
            text-align: center;
            color: #6b7280;
        }}
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🧹 Repository Health Report</h1>
            <p><strong>Repository:</strong> {metadata['repository_path']}</p>
            <p><strong>Generated:</strong> {metadata['report_generated']}</p>
            <p><strong>Size:</strong> {metadata['repository_size']['total_size_human']}
               ({metadata['repository_size']['file_count']} files)</p>
        </div>

        <div class="health-score">
            {health_emoji} Overall Health: {health_score}/100
        </div>

        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-number">{summary['total_issues']}</div>
                <div class="stat-label">Total Issues</div>
            </div>
            <div class="stat-card">
                <div class="stat-number">{summary['hygiene_issues']}</div>
                <div class="stat-label">Hygiene Issues</div>
            </div>
            <div class="stat-card">
                <div class="stat-number">{summary['lint_issues']}</div>
                <div class="stat-label">Code Quality Issues</div>
            </div>
            <div class="stat-card">
                <div class="stat-number">{summary['fixable_issues']}</div>
                <div class="stat-label">Auto-fixable Issues 🔧</div>
            </div>
        </div>
"""

        # Add hygiene issues section
        if summary['hygiene_issues'] > 0:
            html += """
        <div class="issue-section">
            <h2>🧹 Hygiene Issues</h2>
"""
            for category, issues in report_data["hygiene"].items():
                if issues:
                    category_name = category.replace("_", " ").title()
                    html += f"<h3>{category_name} ({len(issues)} found)</h3>"
                    html += '<ul class="issue-list">'
                    for issue in issues[:5]:  # Show first 5
                        path = issue.get("path", "unknown")
                        reason = issue.get("reason", "")
                        html += f'<li class="issue-item"><span class="issue-path">{path}</span><br><small>{reason}</small></li>'
                    if len(issues) > 5:
                        html += f'<li class="issue-item"><em>... and {len(issues) - 5} more</em></li>'
                    html += '</ul>'
            html += "</div>"
        else:
            html += '<div class="issue-section clean"><h2>✅ No Hygiene Issues Found</h2></div>'

        # Add recommendations
        if summary["recommendations"]:
            html += '<div class="recommendations"><h2>💡 Recommendations</h2>'
            for rec in summary["recommendations"]:
                html += f'<div class="recommendation">{rec}</div>'
            html += '</div>'

        # Footer
        html += """
        <div class="footer">
            <p>Generated by <a href="https://github.com/pvestal/repo-clean">repo-clean</a></p>
        </div>
    </div>
</body>
</html>
"""
        return html

    def _format_size(self, size: int) -> str:
        """Format file size in human readable format"""
        for unit in ['B', 'KB', 'MB', 'GB']:
            if size < 1024:
                return f"{size:.1f}{unit}"
            size /= 1024
        return f"{size:.1f}TB"