"""
Basic tests for repo-clean main functionality
"""

import pytest
import tempfile
import os
from pathlib import Path
from unittest.mock import patch
from src.main import RepoCleanCLI


def test_cli_creation():
    """Test that CLI can be created successfully"""
    cli = RepoCleanCLI()
    assert cli is not None
    assert hasattr(cli, 'error_handler')
    assert hasattr(cli, 'explanation_engine')


def test_help_command():
    """Test that help command works"""
    cli = RepoCleanCLI()
    with pytest.raises(SystemExit) as exc:
        cli.main(['--help'])
    assert exc.value.code == 0


def test_scan_empty_repository():
    """Test scanning an empty repository"""
    with tempfile.TemporaryDirectory() as tmpdir:
        # Initialize as git repo
        os.chdir(tmpdir)
        os.system('git init')

        cli = RepoCleanCLI()
        result = cli.main(['scan', '--repository', tmpdir])

        # Should succeed with no issues found
        assert result == 0


def test_gitignore_analyze_no_file():
    """Test gitignore analyze when no .gitignore exists"""
    with tempfile.TemporaryDirectory() as tmpdir:
        # Create a Python file to trigger ecosystem detection
        test_file = Path(tmpdir) / "test.py"
        test_file.write_text("print('hello')")

        cli = RepoCleanCLI()
        result = cli.main(['gitignore', '--analyze', '--repository', tmpdir])

        # Should succeed and detect missing .gitignore
        assert result == 0


def test_gitignore_preview():
    """Test gitignore preview functionality"""
    with tempfile.TemporaryDirectory() as tmpdir:
        # Create a Python file to trigger ecosystem detection
        test_file = Path(tmpdir) / "setup.py"
        test_file.write_text("from setuptools import setup")

        cli = RepoCleanCLI()
        result = cli.main(['gitignore', '--preview', '--repository', tmpdir])

        # Should succeed and show preview
        assert result == 0


def test_invalid_command():
    """Test invalid command handling"""
    cli = RepoCleanCLI()
    result = cli.main(['invalid-command'])

    # Should fail with error code
    assert result != 0


def test_version_in_init():
    """Test that version information is properly set"""
    import src
    assert hasattr(src, '__version__')
    assert hasattr(src, '__author__')
    assert hasattr(src, '__description__')
    assert src.__version__ == "1.0.0"


@patch('sys.argv', ['repo-clean', '--help'])
def test_main_function():
    """Test the main function entry point"""
    from src.main import main

    with pytest.raises(SystemExit) as exc:
        main()
    assert exc.value.code == 0


def test_cli_with_verbose():
    """Test CLI with verbose flag"""
    with tempfile.TemporaryDirectory() as tmpdir:
        cli = RepoCleanCLI()
        result = cli.main(['--verbose', 'scan', '--repository', tmpdir])

        # Should succeed
        assert result == 0


def test_repository_path_validation():
    """Test repository path validation"""
    cli = RepoCleanCLI()

    # Test with non-existent path
    result = cli.main(['scan', '--repository', '/nonexistent/path/12345'])

    # Should handle gracefully
    assert result != 0