"""6.4.0 Migration etude de prix

Revision ID: 2409eb7e97a4
Revises: 0b1f034ec704
Create Date: 2022-01-13 11:07:11.157903

"""

# revision identifiers, used by Alembic.
revision = "2409eb7e97a4"
down_revision = "0b1f034ec704"

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import mysql


def update_database_structure():
    # ### commands auto generated by Alembic - please adjust! ###
    op.drop_constraint("fk_price_study_owner_id", "price_study", type_="foreignkey")
    op.drop_constraint("fk_price_study_project_id", "price_study", type_="foreignkey")
    op.drop_constraint("fk_price_study_company_id", "price_study", type_="foreignkey")
    op.drop_column("price_study", "company_id")
    op.drop_column("price_study", "project_id")
    op.drop_column("price_study", "owner_id")

    op.drop_column("price_study", "notes")
    op.drop_column("price_study", "name")

    op.add_column(
        "price_study",
        sa.Column(
            "general_overhead",
            sa.Numeric(precision=6, scale=5, asdecimal=False),
            nullable=True,
        ),
    )
    op.add_column(
        "price_study",
        sa.Column(
            "mask_hours",
            sa.Boolean(),
            nullable=True,
        ),
    )
    op.add_column(
        "price_study",
        sa.Column(
            "force_ht",
            sa.Boolean(),
            nullable=True,
        ),
    )
    op.alter_column(
        "base_price_study_product",
        "type_",
        existing_type=mysql.VARCHAR(collation="utf8mb4_unicode_ci", length=30),
        nullable=False,
    )

    op.add_column(
        "base_price_study_product",
        sa.Column("task_line_id", sa.Integer(), nullable=True),
    )
    op.drop_constraint(
        "fk_price_study_chapter_task_line_group_id",
        "price_study_chapter",
        type_="foreignkey",
    )
    op.add_column("price_study", sa.Column("task_id", sa.Integer(), nullable=True))
    op.add_column("price_study_work_item", sa.Column("order", sa.Integer()))
    op.add_column(
        "base_price_study_product",
        sa.Column("chapter_id", sa.Integer(), nullable=False),
    )
    # ### end Alembic commands ###


def _clean_price_studies(session):
    # On veut pouvoir mettre la clé à null pour les factures avancement qui ont une
    # étude
    op.drop_constraint("fk_task_price_study_id", "task", type_="foreignkey")
    op.create_foreign_key(
        op.f("fk_task_price_study_id"),
        "task",
        "price_study",
        ["price_study_id"],
        ["id"],
        ondelete="SET NULL",
    )
    # On duplique avant d'ajouter le mode qui est nullable False
    from zope.sqlalchemy import mark_changed

    # Supprime les orphelins
    session.execute(
        """delete from price_study where id not in (
            select distinct(price_study_id) from task
        )"""
    )
    # Supprime les études sans produits
    session.execute(
        """
        delete from price_study where id not in (
            select study_id from base_price_study_product
        );
        """
    )
    # supprime les études attachées à des factures à l'avancement
    session.execute(
        """delete from price_study where id in (
                select price_study_id from task join invoice on task.id=invoice.id 
                where invoice.invoicing_mode='progress' and price_study_id is not null
            )
        """
    )
    # Supprime les études rattachées à des documents validés
    session.execute(
        """delete from price_study where id in (
                select price_study_id from task where task.status='valid'
            )
        """
    )
    # Supprime les études des documents brouillons rattachés à des devis validés
    session.execute(
        """delete from price_study where id in (
                select price_study_id from task join invoice on task.id=invoice.id 
                where invoice.estimation_id is not null
            )
        """
    )
    # Supprime les études rattachés à des documents v
    mark_changed(session)
    session.flush()


def _set_price_study_default_values(session):
    session.execute(sa.text("update price_study set force_ht=0"))
    session.execute(sa.text("update price_study set mask_hours=1"))


def _set_default_tva(session):
    from zope.sqlalchemy import mark_changed
    from caerp.models.tva import Tva

    tva = Tva.get_default()
    if tva is None:
        tva = Tva.query().first()
    session.execute(
        sa.text(
            """update base_price_study_product set tva_id={} 
            where tva_id is null""".format(
                tva.id
            )
        )
    )
    mark_changed(session)


def duplicate_products(session, price_study_id, newps_id):
    columns = """
        `type_`,
        `margin_rate`,
        `ht`,
        `description`,
        `product_id`,
        `tva_id`,
        `unity`,
        `quantity`,
        `total_ht`,
        `order`,
        `created_at`,
        `updated_at`
    """
    # On utilise le chapter_id comme "cache" pour l'ancien id (nécessaire pour que le returning fonctionne)
    res = session.execute(
        sa.text(
            f"""insert into base_price_study_product ({columns}, study_id, chapter_id)
                    select {columns}, {newps_id}, `id` from base_price_study_product
                    where type_="price_study_product" and study_id={price_study_id}
                    returning id, chapter_id
    """
        )
    )
    for new_id, old_id in res.fetchall():
        session.execute(
            sa.text(
                f"""
            insert into price_study_product (`id`, `base_sale_product_id`, `supplier_ht`)
            select {new_id}, `base_sale_product_id`, supplier_ht from price_study_product
            where id={old_id}
            """
            )
        )


def duplicate_works(session, price_study_id, newps_id):
    columns = """
        `type_`,
        `margin_rate`,
        `ht`,
        `description`,
        `product_id`,
        `tva_id`,
        `unity`,
        `quantity`,
        `total_ht`,
        `order`,
        `created_at`,
        `updated_at`
    """
    # On utilise le chapter_id comme "cache" pour l'ancien id (nécessaire pour que le returning fonctionne)
    res = session.execute(
        sa.text(
            f"""insert into base_price_study_product ({columns}, study_id, chapter_id)
                    select {columns}, {newps_id}, `id` from base_price_study_product
                    where type_="price_study_work" and study_id={price_study_id}
                    returning id, chapter_id
    """
        )
    )
    for new_id, old_id in res.fetchall():
        session.execute(
            sa.text(
                f"""
            insert into price_study_work (`id`, `title`, `display_details`)
            select {new_id}, title, display_details from price_study_work
            where id={old_id}
            """
            )
        )

        columns = """
            `type_`,
            `supplier_ht`,
            `ht`,
            `unity`,
            `work_unit_quantity`,
            `total_quantity`,
            `description`,
            `total_ht`,
            `work_unit_ht`,
            `quantity_inherited`,
            `base_sale_product_id`
        """

        session.execute(
            sa.text(
                f"""
            insert into price_study_work_item ({columns}, price_study_work_id)
            select {columns}, {new_id} from price_study_work_item
            where price_study_work_id={old_id}
            """
            )
        )


def duplicate_price_study(session, price_study_id, task_id):
    from zope.sqlalchemy import mark_changed

    columns = f"""`ht`, `created_at` , `updated_at`"""

    res = session.execute(
        sa.text(
            f"""INSERT INTO price_study ({columns}, force_ht, task_id, general_overhead)
        select {columns}, 1, {task_id}, 0 from price_study
        where id={price_study_id} returning id"""
        )
    )
    new_id = res.scalar()
    duplicate_products(session, price_study_id, new_id)
    duplicate_works(session, price_study_id, new_id)

    # On supprime l'ancienne clé étangère
    session.execute(f"""update task set price_study_id = NULL where id={task_id}""")
    mark_changed(session)


def duplicate_double_price_studies(session):
    # On duplique les études qui sont rattachées à des devis et des factures
    escape = []
    for result in session.execute(
        sa.text(
            """
            select id, task.price_study_id, sub.cmpt from task join (
                select price_study_id, count(price_study_id) as cmpt 
                from task group by price_study_id having count(price_study_id)>1 
            ) as sub 
            on sub.price_study_id =task.price_study_id;
            """
        )
    ):
        task_id = result["id"]
        price_study_id = result["price_study_id"]
        # On ne traite qu'à partir de la deuxième étude (on ne veut pas dupliquer deux
        # fois la même)
        if price_study_id in escape:
            duplicate_price_study(session, price_study_id, task_id)
        else:
            escape.append(price_study_id)


def migrate_datas():
    from alembic.context import get_bind
    from zope.sqlalchemy import mark_changed
    from caerp_base.models.base import DBSESSION

    session = DBSESSION()
    _clean_price_studies(session)
    _set_price_study_default_values(session)
    duplicate_double_price_studies(session)

    for table in "price_study_product", "price_study_work_item":
        op.add_column(table, sa.Column("mode", sa.String(length=20), nullable=False))

    # On update les études de prix qui sont rattachés à un devis mais qu'on a
    # pas traité ci-dessus
    for task in session.execute(
        sa.text(
            """select task.id, price_study_id from task join node on node.id=task.id 
            where price_study_id is not null and node.type_='estimation'"""
        )
    ):
        session.execute(
            "update price_study set task_id={} where id={}".format(
                task["id"], task["price_study_id"]
            )
        )
    mark_changed(session)
    session.flush()

    # On update les études prix qui ne sont rattachées qu'à des factures
    for task in session.execute(
        sa.text(
            "select id, price_study_id from task "
            "where price_study_id is not null and id not in ("
            "select task.id from task join node on node.id=task.id "
            "where price_study_id is not null and node.type_='estimation'"
            ")"
        )
    ):
        session.execute(
            "update price_study set task_id={} where id={}".format(
                task["id"], task["price_study_id"]
            )
        )
    mark_changed(session)
    session.flush()

    # Pour chaque étude de prix, créer un chapitre
    for price_study in session.execute("select id from price_study;"):
        session.execute(
            "insert into price_study_chapter ( `price_study_id`) values ({})".format(
                price_study["id"]
            )
        )
    mark_changed(session)
    session.flush()

    for chapter in session.execute(
        "select id, price_study_id from price_study_chapter;"
    ):
        session.execute(
            "update base_price_study_product set chapter_id={}"
            " where study_id={}".format(
                chapter["id"],
                chapter["price_study_id"],
            )
        )
    mark_changed(session)
    session.flush()
    for table in "price_study_product", "price_study_work_item":
        session.execute(f"update {table} set mode='ht';")
        session.execute(
            f"update {table} set mode='supplier_ht' where supplier_ht is not null"
            " and supplier_ht > 0;"
        )

    session.execute(
        """
    update price_study_work_item as wi, (
        select id, RANK() OVER(PARTITION by price_study_work_id order by id asc) as 'rank' 
        from price_study_work_item
    ) as ranks set wi.order=ranks.rank where wi.id=ranks.id;
    """
    )
    # On set le coef de marge des éléménts de l'EDP
    session.execute(
        """
        update base_price_study_product
        set margin_rate= 1 - (1-ifnull(margin_rate,0)) / (1+general_overhead) 
        where  general_overhead>0;
        """
    )
    # On droppe le general overhead au niveau du work item
    session.execute(
        """
    update price_study_work_item set mode='ht' 
    where _general_overhead > 0 or _margin_rate > 0
    """
    )
    mark_changed(session)
    session.flush()

    # On supprime tous les éléments des Task
    session.execute(
        """
        delete task_line_group from task_line_group inner join task on task_line_group.task_id=task.id inner join price_study on task.id=price_study.task_id;
        """
    )
    mark_changed(session)
    session.flush()
    _set_default_tva(session)
    from caerp.models.price_study import PriceStudy

    class Dummy:
        dbsession = session

    request = Dummy()
    for price_study in PriceStudy.query().filter(PriceStudy.task != None):
        if price_study.task.status != "valid":
            price_study.sync_with_task(request)

    op.drop_constraint("fk_task_price_study_id", "task", type_="foreignkey")
    op.drop_constraint(
        "fk_base_price_study_product_study_id",
        "base_price_study_product",
        type_="foreignkey",
    )
    op.drop_column("base_price_study_product", "study_id")
    op.create_foreign_key(
        op.f("fk_base_price_study_product_chapter_id"),
        "base_price_study_product",
        "price_study_chapter",
        ["chapter_id"],
        ["id"],
        ondelete="CASCADE",
    )
    op.create_foreign_key(
        op.f("fk_base_price_study_product_task_line_id"),
        "base_price_study_product",
        "task_line",
        ["task_line_id"],
        ["id"],
        ondelete="SET NULL",
    )
    op.create_foreign_key(
        op.f("fk_price_study_chapter_task_line_group_id"),
        "price_study_chapter",
        "task_line_group",
        ["task_line_group_id"],
        ["id"],
        ondelete="set null",
    )
    op.create_foreign_key(
        op.f("fk_price_study_task_id"),
        "price_study",
        "task",
        ["task_id"],
        ["id"],
        ondelete="cascade",
    )

    op.drop_column("task", "price_study_id")


def upgrade():
    update_database_structure()
    migrate_datas()


def downgrade():
    # ### commands auto generated by Alembic - please adjust! ###
    print("Cannot be downgraded")
