# Copyright 2023 The Langfun Authors
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""LM cache base."""

import abc
import dataclasses
import datetime
from typing import Annotated, Any, Callable
import langfun.core as lf


@dataclasses.dataclass(frozen=True)
class LMCacheEntry:
  """LM cache entry."""
  result: lf.LMSamplingResult
  expire: datetime.datetime | None = None


class LMCacheBase(lf.LMCache):
  """The common LMCache base."""

  key: Annotated[
      Callable[[lf.LanguageModel, lf.Message], Any] | None,
      (
          'A callable ojbect used for computing the key (hashable structure) '
          'from the language model used and input prompt. If None, a default '
          'key will be used, which is sensitive to the sampling options and '
          'input prompt.'
      )
  ] = None

  ttl: Annotated[
      int | None,
      (
          'Time-to-live in seconds.'
      )
  ] = None

  def _on_bound(self):
    super()._on_bound()
    self._key = self.key or default_key

  def get(self,
          lm: lf.LanguageModel,
          prompt: lf.Message) -> lf.LMSamplingResult | None:
    """Gets the cached result of a prompt generated by a language model."""
    entry = self._get(lm.model_id, self._key(lm, prompt))
    if entry is None:
      return None
    if entry.expire is not None and entry.expire < datetime.datetime.now():
      return None
    return entry.result

  def put(self,
          lm: lf.LanguageModel,
          prompt: lf.Message,
          result: lf.LMSamplingResult) -> None:
    """Puts the result of a prompt generated by a language model in cache."""
    expire = None
    if self.ttl:
      expire = datetime.datetime.now() + datetime.timedelta(seconds=self.ttl)
    entry = LMCacheEntry(result, expire)
    self._put(lm.model_id, self._key(lm, prompt), entry)

  @abc.abstractmethod
  def _get(self, model_id: str, key: str) -> LMCacheEntry | None:
    """Returns a LM cache entry associated with the key."""

  @abc.abstractmethod
  def _put(self, model_id: str, key: str, entry: LMCacheEntry) -> None:
    """Puts a LM cache entry associated with the key."""


def default_key(lm: lf.LanguageModel, prompt: lf.Message) -> Any:
  """Default key for LM cache."""
  return (prompt.text, lm.sampling_options.cache_key())
