"""Radiance ground.

Ground is usually used as part of the sky definition.
"""
import honeybee.typing as typing
import ladybug.futil as futil


class Ground(object):
    """Radiance ground.

    Ground definition relies on skyfunc and must be used with one of the Radiance
    skies generated by gendaylit, gensky, etc. You can adjust the ground reflection
    using -g option in gensky / gendaylit. The default value in gensky is %20.

    .. code-block:: shell

        skyfunc glow ground_glow
        0
        0
        4 1 1 1 0
        ground_glow source ground
        0
        0
        4 0 0 -1 180

        oconv some.sky ground.rad scene.rad > scene.oct

    Note:
        For more information see Chapter `6.3.4  The Ground "Glow": An "Upside-Down"
        Sky` in Rendering with Radiance. The chapter is also accessible online at the
    link below.
    https://www.radiance-online.org/community/workshops/2003-berkeley/presentations/Mardaljevic/rwr_ch6.pdf

    Properties:
        * r_emittance
        * g_emittance
        * b_emittance
        * modifier
    """

    def __init__(self, modifier='skyfunc'):
        """Create ground.

        Args:
            modifier: Optional input to change the modifier from skyfunc.

        """
        self.modifier = modifier
        self._r_emittance = 1.0
        self._g_emittance = 1.0
        self._b_emittance = 1.0

    @property
    def r_emittance(self):
        """Ground emittance values for red channel (Default: 1.0)."""
        return self._r_emittance

    @r_emittance.setter
    def r_emittance(self, value):
        self._r_emittance = typing.float_in_range(value, 0, 1, 'r_emittance')

    @property
    def g_emittance(self):
        """Ground emittance values for green channel (Default: 1.0)."""
        return self._g_emittance

    @g_emittance.setter
    def g_emittance(self, value):
        self._g_emittance = typing.float_in_range(value, 0, 1, 'g_emittance')

    @property
    def b_emittance(self):
        """Ground emittance values for blue channel (Default: 1.0)."""
        return self._b_emittance

    @b_emittance.setter
    def b_emittance(self, value):
        self._b_emittance = typing.float_in_range(value, 0, 1, 'b_emittance')

    @property
    def modifier(self):
        "Ground modifier."
        return self._modifier

    @modifier.setter
    def modifier(self, value):
        self._modifier = str(value)

    @classmethod
    def from_dict(cls, input_dict):
        """Create ground from_dict.

        Args:
            input_dict: A python dictionary in the following format

        .. code-block:: python

                {
                'type': 'Ground',
                'r_emittance': r_emittance,
                'g_emittance': g_emittance,
                'b_emittance': b_emittance,
                'modifier': modifier
                }
        """
        assert 'type' in input_dict, \
            'Input dict is missing type. Not a valid ground dictionary.'
        assert input_dict['type'] == 'Ground', \
            'Input type must be Ground not %s' % input_dict['type']
        ground = cls()
        ground.r_emittance = input_dict['r_emittance']
        ground.g_emittance = input_dict['g_emittance']
        ground.b_emittance = input_dict['b_emittance']
        ground.modifier = input_dict['modifier']
        return ground

    def to_file(self, folder='.', name=None, mkdir=False):
        """Write ground to a .ground Radiance file.

        Returns:
            Full path to the newly created file.
        """
        content = self.to_radiance() + '\n'
        name = typing.valid_string(name) if name else 'ground.rad'
        return futil.write_to_file_by_name(folder, name, content, mkdir)

    def to_radiance(self):
        """Get ground as a Radiance input string."""
        ground = \
            '%s glow ground_glow\n0\n0\n4 %.3f %.3f %.3f 0\n' \
            'ground_glow source ground\n0\n0\n4 0 0 -1 180' % (
                self._modifier, self.r_emittance, self.g_emittance, self.b_emittance
            )

        return ground

    def to_dict(self):
        """Translate ground to a dictionary."""
        return {
            'type': 'Ground',
            'r_emittance': self.r_emittance,
            'g_emittance': self.g_emittance,
            'b_emittance': self.b_emittance,
            'modifier': self.modifier
        }

    def __eq__(self, value):
        if type(value) != type(self):
            return False
        if (value.modifier, value.r_emittance, value.g_emittance, value.b_emittance) != \
                (self.modifier, self.r_emittance, self.g_emittance, self.b_emittance):
            return False
        return True

    def __ne__(self, value):
        return not self.__eq__(value)

    def __repr__(self):
        return self.to_radiance()
