# 🇬🇷 Greek

This is an overview of all the datasets used in the Greek part of EuroEval. The
datasets are grouped by their task - see the [task overview](/tasks) for more
information about what these constitute.

## Sentiment Classification

### Greek-SA

This dataset was published in [this paper](https://doi.org/10.1145/2801948.2802010) and
consists of data from Twitter/X.

The original dataset contains 5,936 / 383 / 767 samples for the training,
validation, and test splits, respectively. We use 1,024 / 256 / 2,048 samples for
our training, validation and test splits, respectively. The test split is created using
additional samples from the training set.

Here are a few examples from the training split:

```json
{
    "text": "Εκτός του ότι ήταν προβληματικό το τηλέφωνό από την πρώτη μέρα όταν μπόρεσα να μιλήσω με το service γιατί το τηλέφωνό δεν το σηκώνουν μου είπαν πως δεν γίνεται αντικατάσταση αλλά service ....σε τηλέφωνό του κουτιού....και όταν το έστειλα να το δουν....αυτα!!!!! Τοχούν 15 μέρες ! Με έχουν αφήσει δίχως τηλέφωνο 15 μέρες και ούτε ένα τηλέφωνο. Ούτε να μου πούνε κάτι. Τσαμπίκα κρατάνε Όμηρο και μένα και τα χρήματα μου. Απαράδεκτοι. Λυπάμαι γιατί cubot τηλέφωνα αγοράζαμε πάντα. Μην πέσεις στην ανάγκη τους. Μακριά!",
    "label": "negative"
}
```

```json
{
    "text": "Το τηλεφωνακι ειναι φανταστικο ειδικα για τα  λεφτα αυτα! Εξαιρετικη εμπειρια χειρισμου κ πλοηγησης, καποιες απειροελαχιστες καθυστερησεις ΔΕΝ χαλουν την τελικη εμπειρια. Πρωινες φωτογραφιες πολυ καλες, βραδυνες πολυ μετριες αλλα οχι εντελως χαλια. Το ηχειο ακουγεται παντα, δεν καλυπτεται κι ειναι αξιοπρεπες για κινητο. Οθονη, το δυνατο χαρτι, εξαιρετικη, μπαταρια, με βγαζει 48 συνεχομενες ωρες με λιγο gaming πολυ social texting κ γενικως ψαχουλεμα αρκετο. Καποιο αρνητικο, θα μπορουσα να ελεγα η ταχυτητα του επεξεργαστη , μετα τις βραδυνες φωτο, αλλα με 200+ euros ηξερα τι επαιρνα κ ειμαι πολυ ευχαριστημενος.",
    "label": "positive"
}
```

```json
{
    "text": "Τόση αυτολύπηση στο ΠΑΣΟΚ, είναι πια να τους λυπάσαι. #dendiavasetomnimonio",
    "label": "negative"
}
```

When evaluating generative models, we use the following setup (see the
[methodology](/methodology) for more information on how these are used):

- Number of few-shot examples: 12
- Prefix prompt:

  ```text
  Τα ακόλουθα είναι έγγραφα και το συναίσθημά τους, το οποίο μπορεί να είναι 'θετικό', 'ουδέτερο' ή 'αρνητικό'.
  ```

- Base prompt template:

  ```text
  Έγγραφο: {text}
  Συναίσθημα: {label}
  ```

- Instruction-tuned prompt template:

  ```text
  Έγγραφο: {text}

  Ταξινομήστε το συναίσθημα στο έγγραφο. Απαντήστε με 'θετικό', 'ουδέτερο', ή 'αρνητικό', και τίποτα άλλο.
  ```

- Label mapping:
  - `positive` ➡️ `θετικό`
  - `negative` ➡️ `αρνητικό`

You can evaluate this dataset directly as follows:

```bash
euroeval --model <model-id> --dataset greek-sa
```

## Named Entity Recognition

### elNER

This dataset was published in
[this paper](https://doi.org/10.1145/3411408.3411437).

The original dataset consists of 16,798 / 1,901 / 2,103 samples for the
training, validation, and test splits, respectively. We use 1,024 / 256 / 2,048
samples for our training, validation and test splits, respectively. The new splits
are subsets of the original splits.

Here are a few examples from the training split:

```json
{
    "tokens": ["Επιστήμονες", "στη", "Βρετανία", "μετέτρεψαν", "αρσενικά", "ποντίκια", "σε", "θηλυκά", "κάνοντας", "παρεμβάσεις", "στο", "γονιδίωμά", "τους", "και", "αφαιρώντας", "ορισμένα", "τμήματα", "του", "DNA", "τους", "."],
    "labels": ["O", "O", "B-LOC", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O"]
}
```

```json
{
    "tokens": ["Στο", "μεταξύ", ",", "μία", "άλλη", "δέσμη", "κατευθυντήριων", "γραμμών", "της", "ΕΚΤ", ",", "που", "αφορά", "τα", "δάνεια", "που", "θα", "γίνουν", "«", "κόκκινα", "»", "στο", "μέλλον", ",", "συνάντησε", "αντιδράσεις", "από", "αρκετά", "μέλη", "της", "Ευρωβουλής", ",", "ιδιαίτερα", "από", "την", "Ιταλία", ",", "καθώς", "και", "από", "λομπίστες", "."],
    "labels": ["O", "O", "O", "O", "O", "O", "O", "O", "O", "B-ORG", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "B-ORG", "O", "O", "O", "O", "B-LOC", "O", "O", "O", "O", "O", "O"]
}
```

```json
{
    "tokens": ["Είναι", "σημαντικό", "η", "Ελλάδα", "να", "συνεχίσει", "σε", "αυτή", "την", "πορεία", "για", "να", "ανακτήσει", "σταθερή", "πρόσβαση", "στις", "αγορές", "...", "Παρά", "την", "ενισχυμένη", "εμπιστοσύνη", "των", "αγορών", ",", "η", "ελληνική", "οικονομία", "αντιμετωπίζει", "ένα", "δύσκολο", "οικονομικό", "και", "χρηματοοικονομικό", "περιβάλλον", "."],
    "labels": ["O", "O", "O", "B-LOC", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "B-MISC", "O", "O", "O", "O", "O", "O"]
}
```

When evaluating generative models, we use the following setup (see the
[methodology](/methodology) for more information on how these are used):

- Number of few-shot examples: 8
- Prefix prompt:

  ```text
  Ακολουθούν προτάσεις και λεξικά JSON με τις ονομαστικές οντότητες που εμφανίζονται στην δεδομένη πρόταση.
  ```

- Base prompt template:

  ```text
  Πρόταση: {text}
  Ονομαστικές οντότητες: {label}
  ```

- Instruction-tuned prompt template:

  ```text
  Πρόταση: {text}

  Αναγνωρίστε τις ονομαστικές οντότητες στην πρόταση. Θα πρέπει να παράγετε αυτό ως λεξικό JSON με κλειδιά 'πρόσωπο', 'τοποθεσία', 'οργανισμός' και 'διάφορα'. Οι τιμές πρέπει να είναι λίστες των ονομαστικών οντοτήτων αυτού του τύπου, ακριβώς όπως εμφανίζονται στην πρόταση.
  ```

- Label mapping:
  - `B-PER` ➡️ `πρόσωπο`
  - `I-PER` ➡️ `πρόσωπο`
  - `B-LOC` ➡️ `τοποθεσία`
  - `I-LOC` ➡️ `τοποθεσία`
  - `B-ORG` ➡️ `οργανισμός`
  - `I-ORG` ➡️ `οργανισμός`
  - `B-MISC` ➡️ `διάφορα`
  - `I-MISC` ➡️ `διάφορα`

You can evaluate this dataset directly as follows:

```bash
euroeval --model <model-id> --dataset elner
```

## Linguistic Acceptability

### ScaLA-el

This dataset was published in [this paper](https://aclanthology.org/2023.nodalida-1.20/)
and was automatically created from the [Greek Universal Dependencies
treebank](https://github.com/UniversalDependencies/UD_Greek-GUD) by assuming that the
documents in the treebank are correct, and corrupting the samples to create
grammatically incorrect samples. The corruptions were done by either removing a word
from a sentence, or by swapping two neighbouring words in a sentence. To ensure that
this does indeed break the grammaticality of the sentence, a set of rules were used on
the part-of-speech tags of the words in the sentence.

The original full dataset consists of 1,024 / 256 / 2,048 samples for training,
validation and testing, respectively (so 3,328 samples used in total). These splits are
used as-is in the framework.

Here are a few examples from the training split:

```json
{
    "text": "Πίσω σ το γραφείο, προσπαθώ να βάλω σε τάξη τις σκέψεις μου.",
    "label": "correct"
}
```

```json
{
    "text": "Πρώτα έκανε την κουτσουκέλα της και μετά άρχιζε τις διαχύσεις.",
    "label": "correct"
}
```

```json
{
    "text": "Αν αποφασίσει ότι πρέπει να συνεχίσουμε την έρευνα, θα ζητήσουμε γραπτή εντολή.",
    "label": "correct"
}
```

When evaluating generative models, we use the following setup (see the
[methodology](/methodology) for more information on how these are used):

- Number of few-shot examples: 12
- Prefix prompt:

  ```text
  Οι ακόλουθες είναι προτάσεις και εάν είναι γραμματικά σωστές.
  ```

- Base prompt template:

  ```text
  Πρόταση: {text}
  Γραμματικά σωστή: {label}
  ```

- Instruction-tuned prompt template:

  ```text
  Πρόταση: {text}

  Προσδιορίστε εάν η πρόταση είναι γραμματικά σωστή ή όχι. Απαντήστε με 'ναι' αν είναι σωστή, ή 'όχι' αν δεν είναι. Απαντήστε μόνο με αυτή τη λέξη, και τίποτα άλλο.
  ```

- Label mapping:
  - `correct` ➡️ `ναι`
  - `incorrect` ➡️ `όχι`

You can evaluate this dataset directly as follows:

```bash
euroeval --model <model-id> --dataset scala-el
```

## Reading Comprehension

### MultiWikiQA-el

This dataset was published in [this paper](https://doi.org/10.48550/arXiv.2509.04111)
and contains Wikipedia articles with LLM-generated questions and answers in 300+
languages.

The original full dataset consists of 5,000 samples in a single split. We use a 1,024 /
256 / 2,048 split for training, validation and testing, respectively, sampled randomly.

Here are a few examples from the training split:

```json
{
    "context": "Για συνώνυμους οικισμούς στην Ελλάδα δείτε το λήμμα: Σαρακήνα (αποσαφήνιση)\n\nΗ Σαρακήνα είναι ορεινό χωριό της περιφερειακής ενότητας Γρεβενών σε υψόμετρο 750 μέτρα.\n\nΓεωγραφία - Ιστορία \nΗ Σαρακήνα βρίσκεται προς στα ανατολικά όρια με την περιφερειακή ενότητα Κοζάνης και στην περιοχή των Βεντζίων. Απέχει 29 χλμ. Α.-ΝΑ. από τα Γρεβενά και 28,5 χλμ. ΒΔ. από τη Δεσκάτη. Νότια του χωριού περνάει ο χείμαρρος Γκιώνης που καταλήγει νοτιότερα στον ποταμό Αλιάκμονα. Σύμφωνα με την τοπική παράδοση το χωριό βρισκόταν πάντα στην ίδια τοποθεσία και είχε το ίδιο όνομα. Την περίοδο της τουρκοκρατίας ήταν κεφαλοχώρι, είχε στενή σχέση με την Ιερά Μονή Ζαβόρδας, από την οποία απέχει 25 χλμ. βορειοδυτικά και σε αυτό έμεινε για μεγάλο διάστημα ο Όσιος Νικάνορας. Πολλοί κάτοικοί του δούλεψαν για λογαριασμό της Μονής Ζάβορδας, όπου και προστατεύονταν από τους Τούρκους. Στον Κώδικα της Ζάβορδας αναφέρονται πολλοί κάτοικοί της ως αφιερωτές  της Ιεράς Μονής Αγίου Νικάνορος ή Ζάβορδας τόσο την περίοδο 1534 έως 1692 (Α΄ Γραφή) όσο και 1692 και μετά (Β΄Γραφή).\n\nΗ Σαρακήνα είναι ο τόπος καταγωγής του δάσκαλου - λαογράφου Κώστα Καραπατάκη (1906 - 2000).\n\nΑξιοθέατα\nΗ τρίκλιτη θολωτή βασιλική εκκλησία του Αγίου Νικολάου έχει χαρακτηριστεί από το 1995 ως \"μνημείο χρήζον ειδικής κρατικής προστασίας. Πρόκειται για ενοριακό ναό μεγάλων διαστάσεων, στον τύπο της θολωτής βασιλικής, που κτίστηκε το 1858. Eσωτερικά το μνημείο είναι κατάγραφο με τοιχογραφίες που χρονολογούνται στο δεύτερο μισό του 19ου αι.\".\n\nΔιοικητικά \nΤο χωριό αναφέρεται επίσημα 1918 στο ΦΕΚ 260Α - 31/12/1918 με το όνομα Σαρακίνα να ορίζεται έδρα τη ομώνυμης κοινότητας που ανήκε στο νομό Κοζάνης. Το όνομα διορθώνεται σε Σαρακήνα το 1940 και το 1964 με το ΦΕΚ 185Α - 30/10/1964 αποσπάται στο νομό Γρεβενών. Το χωριό σύμφωνα με το σχέδιο Καλλικράτης, μαζί με το Νεοχώρι και το Δίπορο αποτελούν την τοπική κοινότητα Σαρακήνας που ανήκει στη δημοτική ενότητα Βεντζίου του Δήμου Γρεβενών και σύμφωνα με την απογραφή 2011ως κοινότητα έχει πληθυσμό 378 κατοίκους, ενώ ως οικισμός 177.\n\nΕξωτερικοί σύνδεσμοι\nΝαός Αγίου Νικολάου Σαρακήνας  από τον ιστότοπο Οδυσσεύς του Υπουργείου Πολιτισμού\n\nΠαραπομπές\n\nΔήμος Γρεβενών\nΣαρακήνα', 'question': 'Που βρίσκεται η Σαρακήνα;",
    "answers": {
        "answer_start": [134],
        "text": ["Γρεβενών"]
    }
}
```

```json
{
    "context": "Η Ρεάλ Κλουμπ Θέλτα ντε Βίγο, Σ.Α.Ν. «Β» (Real Club Celta de Vigo, S.A.D. \"B\") είναι ισπανικός ποδοσφαιρικός σύλλογος με έδρα το Βίγο της Ποντεβέρδα, στην αυτόνομη κοινότητα της Γαλικίας. Ιδρύθηκε το 1927, είναι η δεύτερη ομάδα της Θέλτα Βίγο και συμμετέχει σήμερα στη Σεγούντα Ντιβισιόν Β. Παίζει τους εντός έδρας αγώνες της στο Μουνιθιπάλ ντε Μπαρέιρο, χωρητικότητας 4.500 θέσεων.\n\nΙστορία \nΤο 1927, ιδρύθηκε η Σπορτ Κλουμπ Τουρίστα (Sport Club Turista), που μετονομάστηκε σε Κλουμπ Τουρίστα (Club Turista) εννέα χρόνια αργότερα. Το 1988, προβλεπόταν μία συγχώνευση με τη Γκραν Πένια ΦΚ, αλλά τελικά η Τουρίστα απορροφήθηκε από τη Θέλτα Βίγο και μετονομάστηκε Θέλτα Τουρίστα (Celta Turista). Η Γκραν Πένια συνέχισε να αγωνίζεται τις επόμενες δεκαετίες ως ανεξάρτητος σύλλογος.\n\nΣτην πρώτη σεζόν επαγγελματικού ποδοσφαίρου, η Θέλτα Τουρίστα αγωνίστηκε στην Περιφερειακή Κατηγορία, τερματίζοντας στην πρώτη θέση με 57 πόντους. Αγωνίστηκε για πρώτη φορά στη Σεγούντα Ντιβισιόν Β τη σεζόν 1992-93, όπου υποβιβάστηκε την επόμενη σεζόν. Το 1996, για να συμμορφωθεί με τους νέους κανονισμούς της Βασιλικής Ισπανικής Ομοσπονδίας Ποδοσφαίρου, ο σύλλογος άλλαξε την ονομασία του σε Θέλτα Βίγο Β (Celta de Vigo Β).\n\nΙστορικό ονομασιών \n\n Σπορτ Κλουμπ Τουρίστα (1927-1936) \n Κλουμπ Τουρίστα (1936-1988) \n Θέλτα Τουρίστα (1988-1996) \n Θέλτα Βίγο Β (1996-)\n\nΔιακρίσεις \n\n Τερθέρα Ντιβισιόν: 1957-58, 1999-2000, 2000-01 \n Κύπελλο Βασιλικής Ισπανικής Ομοσπονδίας Ποδοσφαίρου: 2001-02\n\nΕξωτερικοί σύνδεσμοι \n\n Επίσημος ιστότοπος Θέλτα Βίγο \n Προφίλ ομάδας στο Futbolme  \n\nΙδρύσεις ποδοσφαιρικών ομάδων το 1927\nΠοδοσφαιρικές ομάδες Ισπανίας\nΘέλτα Βίγο\nΓαλικία",
    "question": "Πότε άλλαξε όνομα η Θέλτα Τουρίστα σε Θέλτα Βίγο Β;",
    "answers": {
        "answer_start": [1036],
        "text": ["1996"]
    }
}
```

```json
{
    "context": "Ο Ντανίλο Α΄ Τσέπτσεβιτς Πέτροβιτς-Νιέγκος (σερβικά: Данило I Шћепчевић Петровић Његош, 1670 – 11 Ιανουαρίου 1735), πιο γνωστός ως Βλάντικα (επίσκοπος) Ντανίλο, από τον Οίκο των Πέτροβιτς-Νιέγκος ήταν Σέρβος Ορθόδοξος Μητροπολίτης του Τσετίνγιε την περίοδο 1697-1735 και Πρίγκιπας-Επίσκοπος του Μαυροβουνίου. Αυτοαποκαλείτο ως "vojevodič srpskoj zemlji" (Δούκας των σερβικών εδαφών).\n\nΒιογραφία\n\nΟ Ντανίλο Τσέπτσεβιτς γεννήθηκε περίπου το 1670, στο Νιεγκούσι. Θεωρείται ως ο ιδρυτής της δυναστείας των Πέτροβιτς-Νιέγκος στο Μαυροβούνιο το 1696. Αφού πρώτα διηύθυνε αμυντικές επιχειρήσεις και οχυρωματικές εργασίες, ενώ, έστω σε περιορισμένο βαθμό, κατάφερε να θέσει ένα τέρμα στις αντιπαλότητες μεταξύ των διαφορετικών μεγάλων οικογενειών του Μαυροβουνίου, ο Ντανίλο ξεκίνησε πολεμικές επιχειρήσεις κατά των Οθωμανών το 1711. Στη διάρκεια της βασιλείας του, έκαναν, για πρώτη φορά, την εμφάνισή τους διπλωματικές σχέσεις και πολιτικοί δεσμοί μεταξύ της Ρωσίας και του Μαυροβουνίου.\n\nΈνας Ρώσος ιστορικός, ο Πάβελ Ροβίνσκι, έγραψε με εντυπωσιακό βαθμό αντικειμενικότητας αναφορικά με τις σχέσεις μεταξύ Μαυροβουνίου και Ρωσίας, χωρίς να τον απασχολούν τα συμφέροντα της κυβέρνησης της χώρας του. Ο Ροβίνσκι έβγαλε ως συμπέρασμα ότι ήταν η απειλή της Αυστρίας και της Τουρκίας (και κατά καιρούς κι αυτή της Δημοκρατίας της Βενετίας) που έστρεψαν το Μαυροβούνιο προς τη Ρωσία. Χωρίς να έχουν πού να αποταθούν στον σκληρό αγώνα που έδιναν για την επιβίωσή τους, οι ηγέτες της σερβικής περιοχής του Μαυροβουνίου στράφηκαν προς το παρελθόν τους, στις μυθικές τους καταβολές και την πηγή όλων των σλαβικών φύλων, όχι μόνο επειδή αποτελούσε μια Μεγάλη Δύναμη, αλλά και για το γεγονός ότι αποτελούσε ισχυρό αντίβαρο απέναντι στους Τούρκους και τους Αυστριακούς.\n\nΤο 1715, ο Ντανίλο επισκέφτηκε τον Τσάρο Πέτρο Α΄ στην Αγία Πετρούπολη και εξασφάλισε τη συμμαχία του ενάντια στους Οθωμανούς—ένα ταξίδι το οποίο έγινε παράδοση για τους διαδόχους του στην ηγεσία του Μαυροβουνίου και σε όλες τις υπόλοιπες σερβικές περιοχές των Βαλκανίων. Στη συνέχεια, επανέκτησε τον έλεγχο της Ζέτα, η οποία βρισκόταν υπό οθωμανική κατοχή, άνοιξε, εκ νέου, το μοναστήρι στο Τσετίνγιε, και ανήγειρε οχυρώσεις γύρω από το Μοναστήρι του Πόντοστρογκ-Πόντμανιε στην Μπούντβα το οποίο ανακατασκευάστηκε το 1630 και χρησίμευε ως θερινή κατοικία για την ηγέτιδα οικογένεια του Μαυροβουνίου. Σε κείμενο που σώζεται σε χειρόγραφο, δώρο του προς το Σερβικό Πατριαρχείο του Πετς, το 1732, ο Ντανίλο συστήνεται με υπερηφάνεια ως \"Ντανίλ Νιέγκος, Επίσκοπος του Τσετίνγιε, ηγεμόνας της σερβικής γης.\"\n\nΤον Ντανίλο διαδέχτηκαν δύο μέλη της οικογένειάς του και στενοί συγγενείς, αρχικά ο ξάδερφός του Σάββας Β΄ Πέτροβιτς-Νιέγκος και στη συνέχεια ο ανιψιός του Βασίλειος Πέτροβιτς-Νιέγκος, ο οποίος για περισσότερες από δύο δεκαετίες κατάφερε να εκτοπίσει από την εξουσία τον αντιδημοφιλή Σάββα και να γίνει η σημαντικότερη ηγετική αρχή του Μαυροβουνίου και εκπρόσωπός του εκτός συνόρων. Η επιλογή του Σάββα Β΄ Πέτροβιτς-Νιέγκος από τον Ντανίλο είχε κυρίως να κάνει με τους οικογενειακούς δεσμούς και το γεγονός ότι συγκαταλεγόταν μεταξύ των υποστηρικτών του τελευταίου, καθώς η οικογένεια του Σάββα προερχόταν από τη γενέτειρα των Πέτροβιτς, Νιεγκούσι. Όπως κι ο Ντανίλο, ο Σάββας έγινε μοναχός, υπηρετώντας στο Μοναστήρι του Μάνιε στις ακτές του Μαυροβουνίου, όπου χρίστηκε αρχιερέας το 1719 από τον Σέρβο Πατριάρχη του Πετς, Μωυσή (1712–1726). Από τη στιγμή που χρίστηκε κι έπειτα, ο Ντανίλο άρχισε σταδιακά να μυεί τον νεαρό Σάββα στην πολιτική ζωή, δίνοντας του ρόλο συγκυβερνήτη ως προετοιμασία ενόψει του μελλοντικού του ρόλου. Ωστόσο, λίγα πράγματα από τη μετέπειτα πολιτική δραστηριότητα του Σάββα δείχνουν ότι είχε οποιοδήποτε κέρδος από τα "πολιτικά γυμνάσια" στα οποία τον υπέβαλλε ο Ντανίλο, με εξαίρεση το γεγονός ότι εξακολούθησε να διατηρεί μια στάση status quo, ενώ, ταυτόχρονα, άφηνε τους τοπικούς φυλάρχους ελεύθερους να αυτοδιαχειρίζονται τις περιοχές τους.\n\nΕξωτερικοί σύνδεσμοι\n Ο Βλάντικα Ντανίλο στο montenet.org - στα αγγλικά\n Διατηρημένες Επιστολές του Βλάντικα Ντανίλο - στα σερβικά\n Επίσημη ιστοσελίδα του Βασιλικού Οίκου του Μαυροβουνίου.\n The Njegoskij Fund Public Project : Private family archives-based digital documentary fund focused on history and culture of Royal Montenegro.\n Άρθρο στη μαυροβουνίτικη γλώσσα σχετικά με τον Μητροπολίτη Ντανίλο Πέτροβιτς-Νιέγκος\n\nΠαραπομπές\n\nΒλαντικα Ντανιλο\nΟίκος των Πέτροβιτς-Νιέγκος', 'question': 'Σε ποια τοποθεσία έχτισε οχυρά ο Ντανίλο;",
    "answers": {
        "answer_start": [2180],
        "text": ["γύρω από το Μοναστήρι του Πόντοστρογκ-Πόντμανιε στην Μπούντβα"]
    }
}
```

When evaluating generative models, we use the following setup (see the
[methodology](/methodology) for more information on how these are used):

- Number of few-shot examples: 4
- Prefix prompt:

  ```text
  Ακολουθούν κείμενα με σχετικές ερωτήσεις και απαντήσεις.
  ```

- Base prompt template:

  ```text
  Κείμενο: {text}
  Ερώτηση: {question}
  Απάντηση (έως 3 λέξεις):
  ```

- Instruction-tuned prompt template:

  ```text
  Κείμενο: {text}

  Απαντήστε στην παρακάτω ερώτηση που σχετίζεται με το κείμενο παραπάνω χρησιμοποιώντας το πολύ 3 λέξεις.

  Ερώτηση: {question}
  ```

You can evaluate this dataset directly as follows:

```bash
euroeval --model <model-id> --dataset multi-wiki-qa-el
```

## Knowledge

### Global-MMLU-el

Global-MMLU is a machine translated version of the English [MMLU
dataset](https://openreview.net/forum?id=d7KBjmI3GmQ) and features questions within 57
different topics, such as elementary mathematics, US history and law. The translation to
Ukrainian was done by the [Cohere Labs Community](https://cohere.com/research).

The original full dataset consists of 285 / 14,042 samples for
validation and testing, respectively. We use a 1,024 / 256 / 2,048 split for training,
validation and testing, respectively (so 3,328 samples used in total).
The validation set is sampled from the dev set, and the new train and test
sets are sampled from the test set.

Here are a few examples from the training split:

```json
{
    "text": "Συμπεριλαμβανομένου του πυθμένα, πόσες πλευρές υπάρχουν σε μια πυραμίδα με τετράγωνο;\nΕπιλογές:\na. τρία\nb. τέσσερις\nc. πέντε\nd. έξι",
    "label": "c",
}
```

```json
{
    "text": "Κάθε χώρα με ακτές έχει αποκλειστική οικονομική ζώνη (ΑΟΖ) σύμφωνα με τη Σύμβαση του ΟΗΕ για το Δίκαιο της Θάλασσας. Ποια από τις παρακάτω προτάσεις σχετικά με το Δίκαιο της Θάλασσας και την ΑΟΖ ΔΕΝ είναι σωστή;\nΕπιλογές:\na. Η ΑΟΖ εκτείνεται έως και 200 \u200b\u200bναυτικά μίλια (370 χλμ.) από την ακτογραμμή.\nb. Οι χώρες έχουν το δικαίωμα να εκμεταλλεύονται τους φυσικούς πόρους στην ΑΟΖ τους.\nc. Οι χώρες μπορούν να εκμεταλλευτούν πόρους στην υφαλοκρηπίδα που εκτείνονται έως και 350 ναυτικά μίλια (560 km) πέρα \u200b\u200bαπό την ΑΟΖ τους.\nd. Μια χώρα δεν έχει το δικαίωμα να εκμεταλλεύεται τους έμβιους πόρους στην ΑΟΖ της.",
    "label": "d",
}
```

```json
{
    "text": "Αυτή η ερώτηση αναφέρεται στις ακόλουθες πληροφορίες. \"Χθες, 7 Δεκεμβρίου 1941 - μια ημερομηνία που θα ζήσει με δυστυχία - οι Ηνωμένες Πολιτείες της Αμερικής δέχθηκαν ξαφνική και εσκεμμένη επίθεση από ναυτικές και αεροπορικές δυνάμεις της Αυτοκρατορίας της Ιαπωνίας. Οι Ηνωμένες Πολιτείες ήταν σε ειρήνη με αυτό το έθνος και, Πράγματι, μία ώρα αφότου οι ιαπωνικές αεροπορικές μοίρες είχαν αρχίσει να βομβαρδίζουν… ο Ιάπωνας πρεσβευτής στις Ηνωμένες Πολιτείες και ο συνάδελφός του παρέδωσαν στον Υπουργό. Επίσημη απάντηση του Κράτους σε πρόσφατο αμερικανικό μήνυμα Ενώ αυτή η απάντηση ανέφερε ότι φαινόταν άχρηστο να συνεχιστούν οι υπάρχουσες διπλωματικές διαπραγματεύσεις, δεν περιείχε καμία απειλή ή υπαινιγμό για πόλεμο ή ένοπλη επίθεση….Η επίθεση χθες…προκάλεσε σοβαρές ζημιές στο αμερικανικό ναυτικό. Και στρατιωτικές δυνάμεις έχουν χαθεί. Επιπλέον, έχουν αναφερθεί τορπιλισμένα αμερικανικά πλοία στην ανοιχτή θάλασσα… Ως Αρχηγός του Στρατού και του Ναυτικού, έχω δώσει εντολή να ληφθούν όλα τα μέτρα για την άμυνά μας. \" Πρόεδρος Franklin D. Roosevelt, ραδιοφωνική ομιλία, 8 Δεκεμβρίου 1941 Ποιο από τα παρακάτω ήταν η πιο άμεση συνέπεια των γεγονότων που περιγράφονται στο παραπάνω απόσπασμα;\nΕπιλογές:\na. Τα εμπορικά εμπάργκο με την Ιαπωνία επεκτείνονται μέχρι τη δεκαετία του 1980\nb. Η Μάχη της Νορμανδίας\nc. Ο εγκλεισμός των Ιαπωνοαμερικανών\nd. Ο βομβαρδισμός στο Ναγκασάκι",
    "label": "c",
}
```

When evaluating generative models, we use the following setup (see the
[methodology](/methodology) for more information on how these are used):

- Number of few-shot examples: 5
- Prefix prompt:

  ```text
  Ακολουθούν ερωτήσεις πολλαπλών επιλογών (με απαντήσεις).
  ```

- Base prompt template:

  ```text
  Ερώτηση: {text}
  Απάντηση: {label}
  ```

- Instruction-tuned prompt template:

  ```text
  Ερώτηση: {text}

  Απαντήστε στην παραπάνω ερώτηση χρησιμοποιώντας 'a', 'b', 'c' ή 'd', και τίποτα άλλο.
  ```

You can evaluate this dataset directly as follows:

```bash
euroeval --model <model-id> --dataset global-mmlu-el
```

## Common-sense Reasoning

### Winogrande-el

This dataset was published in [this paper](https://doi.org/10.48550/arXiv.2506.19468)
and is a translated and filtered version of the English [Winogrande
dataset](https://doi.org/10.1145/3474381).

The original full dataset consists of 47 / 1,210 samples for training and testing, and
we use 128 of the test samples for validation, resulting in a 47 / 128 / 1,085 split for
training, validation and testing, respectively.

Here are a few examples from the training split:

```json
{
    "text": "Δεν μπορούσα να ελέγξω την υγρασία όπως έλεγχα τη βροχή, γιατί το _ έμπαινε παντού. Σε τι αναφέρεται το κενό _;\nΕπιλογές:\na. υγρασία\nb. βροχή",
    "label": "a"
}
```

```json
{
    "text": "Η Τζέσικα πίστευε ότι το Sandstorm ήταν το καλύτερο τραγούδι που γράφτηκε ποτέ, αλλά η Πατρίσια το μισούσε. _ αγόρασε ένα εισιτήριο για τη συναυλία τζαζ. Σε τι αναφέρεται το κενό _;\nΕπιλογές:\na. Τζέσικα\nb. Πατρίσια", "label": "b"
}
```

```json
{
    "text": "Ο θερμοστάτης έδειξε ότι ήταν είκοσι βαθμούς πιο κρύο κάτω από ό,τι ήταν πάνω, οπότε ο Byron έμεινε στο _ επειδή κρύωνε. Σε τι αναφέρεται το κενό _;\nΕπιλογές:\na. κάτω\nb. πάνω",
    "label": "b"
}
```

When evaluating generative models, we use the following setup (see the
[methodology](/methodology) for more information on how these are used):

- Number of few-shot examples: 5
- Prefix prompt:

  ```text
  Ακολουθούν ερωτήσεις πολλαπλής επιλογής (με απαντήσεις).
  ```

- Base prompt template:

  ```text
  Ερώτηση: {text}
  Επιλογές:
  a. {option_a}
  b. {option_b}
  Απάντηση: {label}
  ```

- Instruction-tuned prompt template:

  ```text
  Ερώτηση: {text}
  Επιλογές:
  a. {option_a}
  b. {option_b}

  Απαντήστε στην παραπάνω ερώτηση χρησιμοποιώντας 'a' ή 'b', και τίποτα άλλο.
  ```

You can evaluate this dataset directly as follows:

```bash
euroeval --model <model-id> --dataset winogrande-el
```

## Summarisation

### Greek-Wikipedia

This dataset was published in [this paper](https://doi.org/10.1145/3688671.3688769)
and consists of Greek Wikipedia articles and their summaries assigned by human editors.

The original full dataset consists of 83,432 / 5,000 / 5,000 samples for training,
validation and testing, respectively. We use a 1,024 / 256 / 2,048 split for training,
validation and testing, respectively (so 3,328 samples used in total). All new splits
are subsets of the original splits.

Here are a few examples from the training split:

```json
{
    "text": "Λυκούργος (νομοθέτης)\n\n Ο Λυκούργος θεωρείται ο νομοθέτης της αρχαίας Σπάρτης, παρότι δεν είναι γνωστό αν αυτή η νομοθεσία είναι έργο ενός ή πολλών προσώπων και μίας ή πολλών περιόδων. Άγνωστο επίσης παραμένει αν ο Λυκούργος ήταν ιστορικό ή μυθικό πρόσωπο. Τη ζωή και το νομοθετικό έργο του Λυκούργου την καλύπτει η ομίχλη των μύθων και των θρύλων της αρχαιότητας και μερικοί ιστορικοί των νεότερων χρόνων υποστήριξαν ότι αυτός δεν ήταν ιστορικό πρόσωπο αλλά συμβολική μορφή. Είναι χαρακτηριστικός ο μύθος που θέλει να του έχουν δοθεί οι νόμοι από το μαντείο των Δελφών. Σύμφωνα με την επικρατέστερη παράδοση, ο Λυκούργος έζησε γύρω στο 800 π.Χ. και υπήρξε κηδεμόνας του ανιψιού του βασιλιά της Σπάρτης, Χαρίλαου. Ταξίδεψε σε πολλά μέρη της γης, μελετώντας την οργάνωση του κράτους, στην Κρήτη, την Αίγυπτο, την Ιωνία, τη Λιβύη, την Ιβηρική Χερσόνησο και στην Ινδία. Επιστρέφοντας στη Σπάρτη φρόντισε να μεταρρυθμίσει το σπαρτιατικό πολίτευμα. Στον Λυκούργο αποδίδεται η περίφημη Μεγάλη Ρήτρα, που αναφέρεται στα κύρια στοιχεία του σπαρτιατικού πολιτεύματος. Δικό του έργο θεωρείται η διαίρεση της χώρας σε αναπαλλοτρίωτους ίσους κλήρους (που τους μοίρασε στους ελεύθερους πολίτες), η εισαγωγή βαριών και σιδερένιων νομισμάτων αντί των χρυσών και ασημένιων (για να αποθαρρύνεται η συσσώρευση πλούτου), η σιδερένια πειθαρχία των πολιτών, η ανάθεση της ανατροφής των παιδιών στην πολιτεία, η υποχρέωση της υποταγής των νεότερων στους γεροντότερους καθώς και η θέσπιση της γερουσίας, με τη συμμετοχή των δύο βασιλιάδων. Η διαμόρφωση των νόμων ήταν δικό της έργο, ενώ η τελική τους έγκριση ανήκε στην Απέλλα. Αυτοί οι θεσμοί στάθηκαν ως οι θεμελιώδεις αρχές του σπαρτιατικού πολιτεύματος. Σύμφωνα πάντα με την παράδοση, όταν ολοκληρώθηκε το έργο του αποσύρθηκε στην Κίρρα Φωκίδας ή στην Κρήτη, όπου αφέθηκε να πεθάνει από ασιτία. Εν τω μεταξύ, είχε υποχρεώσει τους συμπολίτες του να ορκιστούν ότι θα τηρούσαν τους νόμους του ως την επιστροφή του. Ο μύθος λέει ότι έφυγε από τη Σπάρτη για πάντα, ώστε να μην αλλάζουν τη νομοθεσία του οι συμπολίτες του, που είχαν ορκιστεί να μην το κάνουν παρά μόνο αν κάποτε επέστρεφε. Σχετικά με τη χρονολόγηση του βίου του υπάρχουν ιδιαίτερα πολλές αντιφάσεις. Ο Αριστοτέλης τον τοποθετεί ως σύγχρονο του Ιφίτου, ο οποίος έζησε γύρω στα 884 π.Χ. Με τη συνδρομή του Ιφίτου ο Λυκούργος εγκαθίδρυσε την ιερή εκεχειρία της Ολυμπίας, σύμφωνα με την οποία απαγορεύονταν όλες οι πολεμικές συγκρούσεις κατά τη διάρκεια των Ολυμπιακών Αγώνων και προστάτευε τους Ηλείους από οποιαδήποτε ενδεχόμενη εχθρική επίθεση (Müller, Dor. 1.7.7.) Ο Ξενοφών διαφοροποιείται σημαντικά από τον Αριστοτέλη, τοποθετώντας τον περισσότερο από 200 χρόνια νωρίτερα, δηλαδή στην εποχή των Ηρακλειδών (Xen. Rep. Lac. 10.8.) Ο Τίμαιος, προκειμένου πιθανόν να υπερκεράσει αυτή τη δυσχέρεια, υπέθεσε ότι υπήρχαν δύο πρόσωπα με το όνομα "Λυκούργος". (Plut. Lyc. 1.) Από αυτές τις αποκλίσεις φαίνεται ότι το όνομα του Λυκούργου δεν εμφανίζεται στον κατάλογο των Σπαρτιατών βασιλέων, ένα από τα παλαιότερα "έγγραφα" στην ελληνική ιστορία (Müller, Dor. 1.7.3.). Ως εκ τούτου, είναι κατανοητό πως ο Ηρόδοτος μπορούσε (1.65) να αποκαλεί "Λυκούργο" τον κηδεμόνα του ανεψιού του Λαβώτα, ενώ ο Σιμωνίδης (Aelian, Ael. VH 9.41) τον αναφέρει ως γιο του Πρύτανι, αδελφού του Ευνόμου, Ο Διονύσιος (2.49) τον θέλει θείο του Ευνόμου και οι κοινότερες αναφορές (Plut. Lyc. 2; Arist. Pol. 2.7. 1; Ephor. apud Strab. x. p. 482) τον θέλουν γιο του Ευνόμου και κηδεμόνα του ανεψιού του Χαριλάου. Ιωάννα Ζούλα, Ο δημιουργός της σπαρτιατικής πολιτείας, εφ. Το Βήμα, 9/11/2003",
    "target_text": "Ο Λυκούργος θεωρείται ο νομοθέτης της αρχαίας Σπάρτης που καθιέρωσε τον στρατιωτικό προσανατολισμό της σπαρτιατικής κοινωνίας σύμφωνα με χρησμό του Απόλλωνα από το μαντείο των Δελφών. Όλες οι μεταρρυθμίσεις του απευθύνονταν στις τρεις αρετές των Σπαρτιατών: ισότητα (μεταξύ των πολιτών), στρατιωτική ικανότητα και λιτότητα. Αναφέρεται από τους αρχαίους ιστορικούς και φιλόσοφους Ηρόδοτο, Ξενοφώντα, Πλάτωνα, Πολύβιο, Πλούταρχο και Επίκτητο. Πολλοί αρχαίοι ιστορικοί πίστευαν ότι ήταν υπεύθυνος για τις κοινοβουλευτικές και μιλιταριστικές μεταρρυθμίσεις που μεταμόρφωσαν τη σπαρτιατική κοινωνία και κυρίως τη Μεγάλη Ρήτρα."
}
```

```json
{
    "text": "Ο θρίαμβος της θέλησης\n\n Η ταινία καταγράφει την συγκέντρωση του Ναζιστικού κόμματος, που έλαβε μέρος από τις 5 έως τις 10 Σεπτεμβρίου 1934 στην Νυρεμβέργη. Το συνέδριο αυτό ονομάστηκε αργότερα Συνέδριο της Ενότητας και της Ισχύος, Συνέδριο της Δύναμης και Νίκη της Θέλησης (γερμ. Reichsparteitag der Einheit und Stärke, Reichsparteitag der Macht καιTriumph des Willens). Λίγους μήνες πριν από το συνέδριο, την λεγόμενη νύχτα των μεγάλων μαχαιριών, ο Χίτλερ είχε δώσει εντολή μαζικής δολοφονίας όλων των πολιτικών του αντιπάλων, ώστε να είναι η μοναδική πολιτική δύναμη στη χώρα. Στην ταινία παρουσιάζονται σκηνές από την προετοιμασία και την διεξαγωγή του συνεδρίου και των παρελάσεων. Στο συνέδριο εμφανίζονται όλοι οι σημαντικοί παράγοντες του Εθνικοσοσιαλισμού σε ομιλίες τους. Στην αρχή του έργου εμφανίζονται επιγραμματικοί πίνακες με διάφορες χρονικές αναφορές, από την απαρχή του πρώτου παγκοσμίου πολέμου μέχρι την ημέρα του συνεδρίου. Ακολουθεί η κινηματογράφηση της πρώτης ημέρας του συνεδρίου. Κοντινά πλάνα μέσα στον ουρανό δείχνουν τα σύννεφα. Το αεροπλάνο του Χίτλερ προσγειώνεται, ενώ ενθουσιώδη πλήθη καλωσορίζουν τον Χίτλερ, που οδηγείται μέσα στο αυτοκίνητό του. Το βράδυ γίνεται μεγαλειώδης νυχτερινή πομπή με δαυλούς. Οι θαμώνες του στρατοπέδου της Νυρεμβέργης ξυπνούν και προετοιμάζονται. Πλένονται, γυμνάζονται, παίζουν σαν μικρά παιδιά. Το συνέδριο αρχίζει. Ο Ρούντολφ Ες κάνει την εναρκτήρια ομιλία. Ακολουθούν ομιλίες των Γιόζεφ Γκαίμπελς, Άλφρεντ Ρόζενμπεργκ και άλλων στελεχών του κόμματος. Το Τμήμα Υπηρεσίας Εργασίας του Ράιχ παρελαύνει. Ο Χίτλερ ομιλεί. Η ημέρα κλείνει με την νυχτερινή παρέλαση της Ταγμάτων Εφόδου (Ες-Α) με δαυλούς. Η ημέρα ξεκινάει με παρέλαση της Χιτλερικής νεολαίας. Ο Μπάλντουρ φον Σίραχ αναγγέλλει την ομιλία του Χίτλερ. Ο Χίτλερ απευθύνεται στη νεολαία και της εμπνέει θάρρος και τόλμη. Ακολουθεί παρέλαση του ιππικού και των μοτοσικλετιστών του γερμανικού στρατού. Το βράδυ ο Χίτλερ κάνει νυχτερινή ομιλία, στην οποία εκθέτει τους λόγους, για τους οποίους το εθνικοσοσιαλιστικό κόμμα έμεινε αβλαβές, παρά τις πρόσφατες εξελίξεις. Την τέταρτη ημέρα, ο Χίτλερ, μπροστά σε 150 χιλιάδες άνδρες της Ες-Α και των Ες-Ες, οδοιπορεί στο μνημειώδες πεδίο της Νυρεμβέργης και καταθέτει στεφάνι στο ηρώο των πεσόντων του Α' Παγκοσμίου Πολέμου. Ακολουθεί ομιλία του Χίτλερ, ενώ ο Λούτσε αποτίει φόρο τιμής στον Χίτλερ. Ο Χίτλερ αγιάζει τις σημαίες των αγωνιστών. Παρέλαση των Ναζί με όλες τις μονάδες. Ο Χίτλερ ξανά ομιλεί στα πλήθη. Ο Ες κλείνει την συνέλευση αποθεώνοντας τον Χίτλερ. Το έργο απέσπασε πολλά βραβεία στη Γερμανία, στις ΗΠΑ, Γαλλία, Σουηδία και σε άλλες χώρες. Το 1937 η Ρίφενσταλ τιμήθηκε με χρυσό μετάλλιο στην παγκόσμια έκθεση του Παρισιού το 1937. Sabine Hake: German National Cinema. Reprinted edition. Routledge, London u. a. 2003, ISBN 0-415-08902-6. Martin Loiperdinger: Der Parteitagsfilm „Triumph des Willens“ von Leni Riefenstahl. Rituale der Mobilmachung. Leske + Budrich, Opladen 1987, ISBN 3-8100-0598-3 (Forschungstexte Wirtschafts- und Sozialwissenschaften 22). Rainer Rother: Führerkult als Film „Triumph des Willens“. In: Gerd Biegel, Wulf Otte (Hrsg.): Ein Volk dankt seinem (Ver)führer. Die Reichserntedankfeste auf dem Bückeberg 1933–1937. Vorträge zur Ausstellung. Braunschweigisches Landesmuseum, Braunschweig 2002, ISBN 3-927939-58-7, S. 109–116 (Veröffentlichungen des Braunschweigischen Landesmuseums 102). Susan Sontag: Faszinierender Faschismus. In: Susan Sontag: Im Zeichen des Saturn. Essays. Hanser, München u. a. 2003, ISBN 3-446-20424-5, S. 97–125. Jürgen Trimborn: Riefenstahl. Eine deutsche Karriere. Aufbau-Verlag, Berlin 2002, ISBN 3-351-02536-X. Πολυμέσα σχετικά με το θέμα Triumph of the Will στο Wikimedia Commons Παρουσίαση με σύγχρονο σχολιασμό και φωτογραφικό υλικό) Παρουσίαση από το Filmzentrale.com Αρχειοθετήθηκε 2012-07-03 στο Wayback Machine. δείτε το έργο",
    "target_text": "Ο θρίαμβος της θέλησης (γερμ. Triumph des Willens, αγγλ. Triumph of the Will) είναι κινηματογραφικό έργο σε σκηνοθεσία Λένι Ρίφενσταλ που καταγράφει σε στιγμιότυπα το Κομματικό συνέδριο της Νυρεμβέργης του 1934."
}
```

```json
{
    "text": "Γεώργιος ντε Λενκάστρε του Αβέιρο\n\n Ήταν ο πρωτότοκος γιος τού Ιωάννη ντε Λενκάστρε 1ου δούκα τού Αβέιρο και της Ζουλιάνα ντε Λάρα, κόρης τού 3ου μαρκησίου της Βίλλα Ρεάλ. Χρησιμοποιούσε τον τίτλο τού μαρκκησίου τού Τόρρες Νόβας, ως το 1571 που διαδέχθηκε τον πατέρα του ως 2ος δούκας τού Αβέιρο. Ήταν στενός σύμβουλος τού 4ου εξαδέλφου του Σεβαστιανού της Πορτογαλίας και τον συνόδευσε στο ταξίδι του στον θείο εκείνου Φίλιππο Β΄ των Αψβούργων βασιλιά της Ισπανίας. Επίσης τον συνόδευσε και στην εκστρατεία του στο Μαρόκο: στην καταστρεπτική μάχη τού Αλκάσερ Κουιμπίρ ο Γεώργιος ντε Λενκάστρε σκοτώθηκε μαζί με τον βασιλιά, το 1578. Νυμφεύτηκε τη Μανταλένα Γκιρόν, αδελφή τού 1ου δούκα της Οσσούνα από την Ισπανία και είχε τέκνο: Ζουλιάνα ντε Λενκάστρε 1560-1636, διεκδικήτρια στο δουκάτο τού Αβέιρο, παντρεύτηκε τον εξάδελφο τού πατέρα της Άλβαρο ντε Λενκάστρε επίσης διεκδικητή στο δουκάτο τού Αβέιρο. Έτσι συμβιβάστηκαν οι δύο αντίδικοι. ”Nobreza de Portugal e do Brasil” – Vol. II, page 343. Published by Zairol Lda., Lisbon 1989. Genealogy of George of Lencastre, 2nd Duke of Aveiro, in Portuguese",
    "target_text": "Ο Γεώργιος ντε Λενκάστρε, πορτογαλ. Jorge de Lencastre (1548 - 1578) από τον Οίκο των Αβις ήταν 2ος δούκας τού Αβέιρο (1571-78) και μαρκήσιος τού Τόρρες Νόβας (1548-71)."
}
```

When evaluating generative models, we use the following setup (see the
[methodology](/methodology) for more information on how these are used):

- Number of few-shot examples: 1
- Prefix prompt:

  ```text
  Ακολουθούν άρθρα με τις αντίστοιχες περιλήψεις.
  ```

- Base prompt template:

  ```text
  Ειδησεογραφικό άρθρο: {text}
  Περίληψη: {target_text}
  ```

- Instruction-tuned prompt template:

  ```text
  Ειδησεογραφικό άρθρο: {text}

  Γράψτε μια περίληψη του παραπάνω άρθρου.
  ```

You can evaluate this dataset directly as follows:

```bash
euroeval --model <model-id> --dataset greek-wikipedia
```
