import os
import whisper
import re
import pandas as pd
from moviepy.video.io.VideoFileClip import VideoFileClip
from moviepy.audio.io.AudioFileClip import AudioFileClip
from spacy.lang.zh import Chinese
from spacy.lang.en import English
import spacy
from langdetect import detect, detect_langs, LangDetectException
from pydub import AudioSegment


def parse_subtitle(file_path, vtt_file=None):
    """
    Parses various subtitle formats (.ass, .sub, .srt, .txt, .vtt) into a DataFrame.
    If vtt_file is provided, it will be used directly as the content.
    """
    if vtt_file is None:
        try:
            with open(file_path, "r", encoding="utf-8-sig", errors="replace") as file:
                lines = file.readlines()
        except FileNotFoundError:
            return pd.DataFrame(columns=["Timestamps", "Content"])
        except ImportError:
            print("pysrt library not found. Falling back to less robust parsing.")
    else:
        lines = vtt_file.splitlines()

    timestamps = []
    contents = []
    current_content = []
    if file_path.lower().endswith(".txt") or (
        vtt_file is not None and file_path.lower().endswith(".txt")
    ):
        contents = lines
        timestamps = [""] * len(contents)
    else:
        i = 0
        while i < len(lines):
            line = lines[i].strip()
            # Check for timestamp line
            if "-->" in line or re.match(
                r"\d{2}:\d{2}:\d{2}[,\.]\d{3} --> \d{2}:\d{2}:\d{2}[,\.]\d{3}", line
            ):
                timestamps.append(line)
                i += 1
                current_content = []
                # Skip any empty lines and collect text until a new timestamp is detected.
                while i < len(lines) and not re.match(
                    r"\d{2}:\d{2}:\d{2}[,\.]\d{3} --> \d{2}:\d{2}:\d{2}[,\.]\d{3}",
                    lines[i].strip(),
                ):
                    stripped = lines[i].strip()
                    if stripped:  # only add non-empty text lines
                        current_content.append(stripped)
                    i += 1
                contents.append(" ".join(current_content))
            # Handle other subtitle formats (Dialogue or similar)
            elif "Dialogue:" in line or re.match(r"{\d+}{\d+}.*", line):
                timestamps.append(line)
                i += 1
                current_content = []
                while i < len(lines) and not lines[i].strip().isdigit():
                    stripped = lines[i].strip()
                    if stripped:
                        current_content.append(stripped)
                    i += 1
                contents.append(" ".join(current_content))
            else:
                i += 1

    return pd.DataFrame({"Timestamps": timestamps, "Content": contents})


def transcribe(file_path, language=None, output_dir=None, model_size="large-v3"):
    """
    Transcribes an audio file to a WebVTT file with proper timestamps.

    Args:
        file_path (str): Path to the audio file
        language (str, optional): Language code for transcription
        output_dir (str, optional): Directory to save the VTT file
        model_size (str, optional): Whisper model size (tiny, base, small, medium, large-v1, large-v2, large-v3)
    """
    model = whisper.load_model(f"{model_size}", device="cpu")
    result = model.transcribe(
        file_path, fp16=False, verbose=True, language=language if language else None
    )
    detected_language = result.get(
        "language", language if language else "unknown")

    # Create VTT content with proper timestamps
    vtt_content = ["WEBVTT\n"]
    for segment in result["segments"]:
        # ...existing timestamp formatting...
        hours = int(segment["start"] // 3600)
        minutes = int((segment["start"] % 3600) // 60)
        start_seconds = segment["start"] % 60
        end_hours = int(segment["end"] // 3600)
        end_minutes = int((segment["end"] % 3600) // 60)
        end_seconds = segment["end"] % 60

        start_time = f"{hours:02d}:{minutes:02d}:{start_seconds:06.3f}"
        end_time = f"{end_hours:02d}:{end_minutes:02d}:{end_seconds:06.3f}"
        text = segment["text"].strip()
        vtt_content.append(f"\n{start_time} --> {end_time}\n{text}")

    # Use provided output_dir or default to the base file's directory
    if output_dir is None:
        output_dir = os.path.dirname(os.path.abspath(file_path))
    else:
        os.makedirs(output_dir, exist_ok=True)
    base_name = os.path.splitext(os.path.basename(file_path))[0]
    out_file = os.path.join(output_dir, base_name + ".vtt")
    with open(out_file, "w", encoding="utf-8") as f:
        f.write(" ".join(vtt_content))

    return out_file, detected_language


def segment(file_path, sentence_count=20, cite_timestamps=False):
    """Segments a text file into paragraphs by grouping every N sentences.
    
    Args:
        file_path (str): Path to the input file
        sentence_count (int): Number of sentences per paragraph
        cite_timestamps (bool): Whether to include timestamps as headers in output
    """
    try:
        # Check if cite_timestamps is requested for non-timestamp files
        if cite_timestamps and not file_path.lower().endswith(('.vtt', '.srt', '.ass', '.ssa', '.sub', '.smi')):
            raise ValueError("Error: --cite-timestamps can only be used with files that have timestamps (VTT, SRT, ASS, SSA, SUB, SMI)")
        
        # Handle docx files
        if file_path.lower().endswith('.docx'):
            from docx import Document
            doc = Document(file_path)
            text = "\n".join(para.text for para in doc.paragraphs if para.text.strip())
        # Handle markdown files
        elif file_path.lower().endswith(('.md', '.markdown', '.txt')):
            with open(file_path, 'r', encoding='utf-8') as f:
                text = f.read()
        else:
            # Handle subtitle files
            vtt_df = parse_subtitle(file_path)
            if cite_timestamps:
                # Check for missing or corrupted timestamps
                if 'Timestamps' not in vtt_df.columns or vtt_df['Timestamps'].isnull().any() or vtt_df['Timestamps'].eq('').any():
                    raise ValueError("Error: Timestamps missing or corrupted for cite_timestamps option.")
                
                return _segment_with_timestamps(vtt_df, sentence_count)
            else:
                text = "。".join(vtt_df["Content"])

        # Directly use basic language classes
        if any(char in text for char in "，。？！"):
            nlp = Chinese()
        else:
            nlp = English()

        # Add the sentencizer component to the pipeline
        if "sentencizer" not in nlp.pipe_names:
            nlp.add_pipe("sentencizer")

        doc = nlp(text)

        paragraphs = []
        current_paragraph = []
        current_count = 0
        for sent in doc.sents:
            # Add Chinese comma if needed
            sent_text = sent.text.strip()
            if not any(sent_text.endswith(p) for p in "，。？！,.!?"):
                sent_text += "，"
            current_paragraph.append(sent_text)
            current_count += 1
            if current_count >= sentence_count:
                paragraphs.append("".join(current_paragraph))
                current_paragraph = []
                current_count = 0

        if current_paragraph:
            paragraphs.append("".join(current_paragraph))

        return "\n\n".join(paragraphs)
        
    except Exception as e:
        print(f"Error in segment: {e}")
        # Return the original text if processing fails
        if file_path.lower().endswith(('.docx', '.md', '.markdown', '.txt')):
            try:
                if file_path.lower().endswith('.docx'):
                    from docx import Document
                    doc = Document(file_path)
                    return "\n".join(para.text for para in doc.paragraphs if para.text.strip())
                else:
                    with open(file_path, 'r', encoding='utf-8') as f:
                        return f.read()
            except:
                return "Error reading file"
        else:
            try:
                vtt_df = parse_subtitle(file_path)
                return "。".join(vtt_df["Content"])
            except:
                return "Error reading subtitle file"


def _segment_with_timestamps(vtt_df, sentence_count):
    """Helper function to segment subtitle content with timestamps."""
    # Convert timestamp format to HH:MM:SS
    def format_timestamp(timestamp_str):
        # Extract start time from timestamp (format: "HH:MM:SS.mmm --> HH:MM:SS.mmm")
        if "-->" in timestamp_str:
            parts = timestamp_str.split("-->")
            start_time = parts[0].strip()
            end_time = parts[1].strip()
            # Convert to seconds format (remove milliseconds)
            start_time = start_time.split('.')[0] if '.' in start_time else start_time.split(',')[0]
            end_time = end_time.split('.')[0] if '.' in end_time else end_time.split(',')[0]
            return start_time, end_time
        return timestamp_str, timestamp_str
    
    # Handle cases where we have empty or corrupted data
    if vtt_df.empty or 'Content' not in vtt_df.columns or 'Timestamps' not in vtt_df.columns:
        return "Error: No valid content or timestamps found"
    
    # Group content by sentences
    all_content = []
    all_timestamps = []
    
    for idx, row in vtt_df.iterrows():
        content = row['Content'].strip() if pd.notna(row['Content']) else ''
        timestamp = row['Timestamps'] if pd.notna(row['Timestamps']) else ''
        
        if content and timestamp:
            # Split content into sentences for proper grouping
            sentences = re.split(r'[。！？.!?]', content)
            sentences = [s.strip() for s in sentences if s.strip()]
            
            # If no sentences found, treat whole content as one sentence
            if not sentences:
                sentences = [content]
            
            for sentence in sentences:
                if sentence:
                    all_content.append(sentence)
                    all_timestamps.append(timestamp)
    
    # If no content found, return error
    if not all_content:
        return "Error: No valid content found for processing"
    
    # Group into chunks based on sentence_count
    output_chunks = []
    current_chunk = []
    chunk_start_time = None
    chunk_end_time = None
    
    for i, (content, timestamp) in enumerate(zip(all_content, all_timestamps)):
        if len(current_chunk) == 0:
            # Start new chunk
            chunk_start_time, _ = format_timestamp(timestamp)
        
        current_chunk.append(content)
        _, chunk_end_time = format_timestamp(timestamp)
        
        # Check if we should close this chunk
        if len(current_chunk) >= sentence_count or i == len(all_content) - 1:
            # Format chunk content
            chunk_content = "。".join(current_chunk)
            if not chunk_content.endswith(('。', '！', '？', '.', '!', '?')):
                chunk_content += "。"
            
            # Create timestamp header
            timestamp_header = f"### **{chunk_start_time} - {chunk_end_time}**"
            
            # Add to output
            output_chunks.append(f"{timestamp_header}\n\n{chunk_content}")
            
            # Reset for next chunk
            current_chunk = []
            chunk_start_time = None
    
    return "\n\n".join(output_chunks)


def download_audio(url, output_dir=None, timestamp=None, output_wav=None):
    """
    Download audio from a URL and convert it to WAV format.

    Args:
        url (str): URL of the video/audio to download
        output_dir (str, optional): Directory to save the downloaded file
        timestamp (tuple, optional): (start_seconds, end_seconds) for extraction
        output_wav (str, optional): Custom filename for the output WAV file

    Returns:
        str: Path to the downloaded WAV file
    """
    import yt_dlp

    if output_dir is None:
        output_dir = os.getcwd()

    # If output_wav is provided, use it as the output filename (without extension)
    if output_wav:
        output_wav = os.path.splitext(output_wav)[0]  # Remove extension if present
        outtmpl = os.path.join(output_dir, f"{output_wav}.%(ext)s")
    else:
        outtmpl = os.path.join(output_dir, "%(title)s.%(ext)s")

    ydl_opts = {
        "format": "bestaudio/best",
        "postprocessors": [
            {
                "key": "FFmpegExtractAudio",
                "preferredcodec": "wav",
            }
        ],
        "outtmpl": outtmpl,
        "quiet": False,
        "no_warnings": True,
    }

    try:
        with yt_dlp.YoutubeDL(ydl_opts) as ydl:
            info = ydl.extract_info(url, download=True)
            if output_wav:
                output_file = os.path.join(output_dir, f"{output_wav}.wav")
            else:
                output_file = ydl.prepare_filename(info).rsplit(".", 1)[0] + ".wav"

            if timestamp:
                # Extract the specified segment
                return extract_audio_segment(output_file, timestamp, output_dir)
            return output_file
    except Exception as e:
        raise Exception(f"Error downloading audio: {str(e)}")


def language_detect(file_path, detected_lang=None):
    """
    Detects the language of a text file using langdetect.
    Returns language code (e.g., 'zh', 'en', etc.).
    """
    try:
        df = parse_subtitle(file_path)
        sample_content = " ".join(df["Content"].head(20))
        if not sample_content.strip():
            # Fallback if file content is empty or only whitespace
            return "en"
        languages = detect_langs(sample_content)
        if languages:
            detected = languages[0].lang
            return "zh" if detected.startswith("zh") else detected
    except Exception as e:
        print(f"Language detection error: {e}")
    return "en"


def parse_timestamp(start_time=None, end_time=None):
    """Parse start and end times in HH:MM:SS format to seconds tuple."""
    if not start_time or not end_time:
        return None
    try:
        def time_to_seconds(time_str):
            h, m, s = map(int, time_str.split(':'))
            return h * 3600 + m * 60 + s

        return (time_to_seconds(start_time), time_to_seconds(end_time))
    except:
        raise ValueError("Invalid time format. Use HH:MM:SS")


def extract_audio_segment(audio_path, timestamp=None, output_dir=None, output_wav=""):
    """
    Extract full audio or segment using moviepy.

    Args:
        audio_path (str): Path to input audio/video file
        timestamp (dict, optional): Dictionary with 'start' and 'end' times in HH:MM:SS format
        output_dir (str, optional): Output directory for the extracted audio
        output_wav (str, optional): Custom filename for the output WAV file
    """
    if output_dir is None:
        output_dir = os.path.dirname(audio_path)

    base_name = os.path.splitext(os.path.basename(audio_path))[0]

    try:
        # Use VideoFileClip first to handle both video and audio files
        try:
            clip = VideoFileClip(audio_path)
            audio = clip.audio
        except:
            # If not video, try loading as audio
            audio = AudioFileClip(audio_path)

        if timestamp:
            # Convert HH:MM:SS to seconds
            start = sum(x * int(t) for x, t in zip([3600, 60, 1], timestamp['start'].split(':')))
            end = sum(x * int(t) for x, t in zip([3600, 60, 1], timestamp['end'].split(':')))

            # Extract segment
            audio = audio.subclipped(start, end)
            if output_wav:
                # Remove .wav extension if present in output_wav
                output_wav = os.path.splitext(output_wav)[0]
                output_path = os.path.join(output_dir, f"{output_wav}.wav")
            else:
                # Use default timestamp-based filename
                output_path = os.path.join(output_dir, f"{base_name}_{timestamp['start']}-{timestamp['end']}.wav")
        else:
            output_path = os.path.join(output_dir, f"{base_name}.wav")

        # Write WAV file
        audio.write_audiofile(output_path, codec='pcm_s16le')

        # Clean up
        audio.close()
        if 'clip' in locals():
            clip.close()

        return output_path

    except Exception as e:
        raise Exception(f"Error extracting audio: {e}")
