#!/usr/bin/env python3
"""
MLOps-Enhanced Training Pipeline for DAGnostics
Integrates all MLOps components into a comprehensive training system
"""

import sys

sys.path.insert(0, "src")

import json
import logging
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Optional

# Training components
from dagnostics.training.fine_tuner import SLMFineTuner
from mlops.data_validator import DataQualityReport, DataValidator

# MLOps components
from mlops.experiment_tracker import (
    ExperimentConfig,
    ExperimentTracker,
    create_experiment_config_from_args,
)
from mlops.hyperparameter_tuner import (
    HyperparameterSpace,
    HyperparameterTuner,
    OptimizationConfig,
)
from mlops.model_registry import ModelRegistry

logger = logging.getLogger(__name__)


class MLOpsTrainingPipeline:
    """
    Production-grade MLOps training pipeline
    Integrates experiment tracking, data validation, hyperparameter tuning
    """

    def __init__(
        self,
        mlflow_tracking_uri: str = "sqlite:///mlops/experiments.db",
        enable_wandb: bool = False,
        wandb_project: str = "dagnostics-mlops",
        enable_hyperparameter_tuning: bool = False,
    ):
        self.mlflow_tracking_uri = mlflow_tracking_uri
        self.enable_wandb = enable_wandb
        self.wandb_project = wandb_project
        self.enable_hyperparameter_tuning = enable_hyperparameter_tuning

        # Initialize components
        self.data_validator = DataValidator()
        self.experiment_tracker = ExperimentTracker(
            mlflow_tracking_uri=mlflow_tracking_uri,
            wandb_project=wandb_project,
            enable_wandb=enable_wandb,
        )
        self.model_registry = ModelRegistry(mlflow_tracking_uri=mlflow_tracking_uri)

        # Create MLOps directories
        for dir_name in [
            "mlops",
            "mlops/experiments",
            "mlops/data_reports",
            "mlops/models",
        ]:
            Path(dir_name).mkdir(exist_ok=True)

        logger.info("🚀 MLOps Training Pipeline initialized")

    def run_full_pipeline(
        self,
        model_name: str = "microsoft/DialoGPT-small",
        train_dataset_path: str = "data/training/train_dataset.jsonl",
        validation_dataset_path: str = "data/training/validation_dataset.jsonl",
        learning_rate: float = 5e-6,
        batch_size: int = 2,
        epochs: int = 3,
        max_length: int = 512,
        use_quantization: bool = False,
        force_cpu: bool = True,
        model_output_name: str = "dagnostics-mlops-model",
        experiment_name: str = "dagnostics-production-training",
        run_hyperparameter_optimization: bool = False,
        **kwargs,
    ) -> Dict[str, Any]:
        """
        Run complete MLOps training pipeline
        """

        logger.info("🚀 Starting MLOps Training Pipeline")
        logger.info("=" * 60)

        pipeline_start_time = datetime.now()

        try:
            # Stage 1: Data Validation
            logger.info("📊 Stage 1: Data Validation & Quality Assessment")
            validation_reports = self._run_data_validation(
                train_dataset_path, validation_dataset_path
            )

            # Stage 2: Hyperparameter Optimization (Optional)
            if run_hyperparameter_optimization or self.enable_hyperparameter_tuning:
                logger.info("🎯 Stage 2: Hyperparameter Optimization")
                optimization_results = self._run_hyperparameter_optimization(
                    train_dataset_path, validation_dataset_path, model_name, force_cpu
                )
                # Update parameters with optimized values
                if optimization_results:
                    optimized_params = optimization_results["best_params"]
                    learning_rate = optimized_params.get("learning_rate", learning_rate)
                    batch_size = optimized_params.get("batch_size", batch_size)
                    epochs = optimized_params.get("epochs", epochs)
                    max_length = optimized_params.get("max_length", max_length)

                    logger.info(f"🎛️  Using optimized parameters: {optimized_params}")
            else:
                optimization_results = None

            # Stage 3: Model Training with Experiment Tracking
            logger.info("🤖 Stage 3: Model Training with MLOps Integration")
            training_results = self._run_tracked_training(
                model_name=model_name,
                train_dataset_path=train_dataset_path,
                validation_dataset_path=validation_dataset_path,
                learning_rate=learning_rate,
                batch_size=batch_size,
                epochs=epochs,
                max_length=max_length,
                use_quantization=use_quantization,
                force_cpu=force_cpu,
                model_output_name=model_output_name,
                experiment_name=experiment_name,
                validation_reports=validation_reports,
                optimization_results=optimization_results,
                **kwargs,
            )

            # Stage 4: Model Validation & Quality Assurance
            logger.info("✅ Stage 4: Model Validation & Quality Assurance")
            model_validation_results = self._run_model_validation(
                training_results["model_path"], validation_dataset_path
            )

            # Stage 5: Model Registry & Versioning
            logger.info("🏛️  Stage 5: Model Registry & Versioning")
            registry_results = self._register_model(
                model_path=training_results["model_path"],
                model_name=model_output_name,
                training_results=training_results,
                validation_reports=validation_reports,
                optimization_results=optimization_results,
                model_validation_results=model_validation_results,
                training_params={
                    "model_name": model_name,
                    "learning_rate": learning_rate,
                    "batch_size": batch_size,
                    "epochs": epochs,
                    "force_cpu": force_cpu,
                },
            )

            # Stage 6: Pipeline Summary & Reporting
            pipeline_duration = (datetime.now() - pipeline_start_time).total_seconds()

            pipeline_results = {
                "pipeline_status": "success",
                "pipeline_duration_seconds": pipeline_duration,
                "data_validation": validation_reports,
                "hyperparameter_optimization": optimization_results,
                "training_results": training_results,
                "model_validation": model_validation_results,
                "model_registry": registry_results,
                "final_model_path": training_results["model_path"],
                "experiment_run_id": training_results.get("run_id"),
                "pipeline_timestamp": pipeline_start_time.isoformat(),
            }

            # Save pipeline results
            self._save_pipeline_results(pipeline_results)

            logger.info("🎉 MLOps Training Pipeline Completed Successfully!")
            logger.info(f"⏱️  Total Duration: {pipeline_duration:.1f} seconds")
            logger.info(f"🤖 Final Model: {training_results['model_path']}")

            return pipeline_results

        except Exception as e:
            logger.error(f"❌ MLOps Pipeline Failed: {e}")

            # Save failure information
            failure_results = {
                "pipeline_status": "failed",
                "error_message": str(e),
                "pipeline_duration_seconds": (
                    datetime.now() - pipeline_start_time
                ).total_seconds(),
                "failure_timestamp": datetime.now().isoformat(),
            }

            self._save_pipeline_results(failure_results)
            raise

    def _run_data_validation(
        self, train_dataset_path: str, validation_dataset_path: str
    ) -> Dict[str, DataQualityReport]:
        """Run comprehensive data validation"""

        try:
            # Validate datasets
            validation_reports = self.data_validator.validate_training_pipeline(
                train_path=train_dataset_path, val_path=validation_dataset_path
            )

            # Check if data quality is sufficient for training
            train_quality = validation_reports.get("train")
            if train_quality and train_quality.quality_score < 0.3:
                raise ValueError(
                    f"Training data quality too low: {train_quality.quality_score:.2f} < 0.3. "
                    f"Issues: {train_quality.issues}"
                )

            # Log validation summary
            for dataset_type, report in validation_reports.items():
                logger.info(
                    f"📊 {dataset_type.title()} Data Quality: "
                    f"Score={report.quality_score:.2f}, "
                    f"Samples={report.total_samples}, "
                    f"Issues={len(report.issues)}"
                )

            return validation_reports

        except Exception as e:
            logger.error(f"❌ Data validation failed: {e}")
            raise

    def _run_hyperparameter_optimization(
        self,
        train_dataset_path: str,
        validation_dataset_path: str,
        model_name: str,
        force_cpu: bool,
    ) -> Optional[Dict[str, Any]]:
        """Run hyperparameter optimization"""

        try:
            # Define hyperparameter space
            hyperparameter_space = HyperparameterSpace(
                learning_rate_min=1e-6,
                learning_rate_max=1e-3,
                batch_sizes=[1, 2, 4] if force_cpu else [4, 8, 16],
                epochs_min=1,
                epochs_max=5 if force_cpu else 8,
                max_length_options=[256, 512, 1024],
                optimizer_choices=["adamw_torch", "adafactor"],
            )

            # Optimization configuration
            optimization_config = OptimizationConfig(
                study_name=f"dagnostics-hpo-{datetime.now().strftime('%Y%m%d-%H%M%S')}",
                n_trials=5 if force_cpu else 15,  # Fewer trials for CPU
                timeout=3600,  # 1 hour
                metric_name="val_loss",
                direction="minimize",
                max_epochs_per_trial=3 if force_cpu else 5,
                max_time_per_trial=1800 if force_cpu else 3600,
            )

            # Define training function for optimization
            def optimization_training_function(params: Dict[str, Any]) -> float:
                """Training function for hyperparameter optimization"""

                try:
                    # Create fine-tuner
                    fine_tuner = SLMFineTuner(
                        model_name=model_name,
                        use_quantization=False,  # Disable for optimization
                        force_cpu=force_cpu,
                    )

                    # Run training with suggested parameters
                    model_path = fine_tuner.train_model(
                        train_dataset_path=train_dataset_path,
                        validation_dataset_path=validation_dataset_path,
                        num_epochs=params["epochs"],
                        learning_rate=params["learning_rate"],
                        batch_size=params["batch_size"],
                        model_output_name=f"hpo-trial-{params['trial_number']}",
                    )

                    # Evaluate model (simplified)
                    if Path(validation_dataset_path).exists():
                        eval_results = fine_tuner.evaluate_model(
                            model_path, validation_dataset_path
                        )
                        return eval_results["perplexity"]  # Lower is better
                    else:
                        # Fallback: use a dummy metric
                        return 5.0  # Reasonable perplexity value

                except Exception as e:
                    logger.error(f"❌ Optimization trial failed: {e}")
                    return float("inf")  # Return worst possible value

            # Run optimization
            tuner = HyperparameterTuner(
                config=optimization_config,
                hyperparameter_space=hyperparameter_space,
                training_function=optimization_training_function,
            )

            optimization_results = tuner.optimize()

            # Generate optimization plots
            tuner.plot_optimization_history()

            logger.info(f"🏆 Hyperparameter optimization completed")
            logger.info(f"📊 Best parameters: {optimization_results['best_params']}")

            return optimization_results

        except Exception as e:
            logger.error(f"❌ Hyperparameter optimization failed: {e}")
            return None

    def _run_tracked_training(
        self,
        validation_reports: Dict[str, DataQualityReport],
        optimization_results: Optional[Dict[str, Any]] = None,
        **training_kwargs,
    ) -> Dict[str, Any]:
        """Run training with full experiment tracking"""

        # Create experiment configuration
        experiment_config = create_experiment_config_from_args(**training_kwargs)
        experiment_config.tags.update(
            {
                "mlops_pipeline": "true",
                "data_quality_score": f"{validation_reports.get('train', DataQualityReport).quality_score:.2f}",
                "hyperparameter_optimized": "true" if optimization_results else "false",
            }
        )

        # Add dataset information
        train_report = validation_reports.get("train")
        if train_report:
            experiment_config.dataset_size = train_report.total_samples

        try:
            with self.experiment_tracker.start_experiment(experiment_config) as tracker:

                # Log data validation results
                tracker.log_dataset_info(
                    {
                        "train_samples": (
                            train_report.total_samples if train_report else 0
                        ),
                        "train_quality_score": (
                            train_report.quality_score if train_report else 0
                        ),
                        "train_issues": len(train_report.issues) if train_report else 0,
                        "validation_samples": (
                            validation_reports.get(
                                "validation", DataQualityReport
                            ).total_samples
                            if "validation" in validation_reports
                            else 0
                        ),
                    }
                )

                # Log optimization results if available
                if optimization_results:
                    tracker.log_params(
                        {
                            f"hpo_{k}": v
                            for k, v in optimization_results["best_params"].items()
                        }
                    )
                    tracker.log_metrics(
                        {
                            "hpo_best_value": optimization_results["best_value"],
                            "hpo_n_trials": optimization_results["optimization_report"][
                                "results"
                            ]["n_trials"],
                        }
                    )

                # Create and configure fine-tuner
                fine_tuner = SLMFineTuner(
                    model_name=training_kwargs["model_name"],
                    use_quantization=training_kwargs.get("use_quantization", False),
                    force_cpu=training_kwargs.get("force_cpu", True),
                )

                # Enhanced training with progress tracking
                model_path = self._run_enhanced_training(
                    fine_tuner, tracker, training_kwargs
                )

                # Log final model
                if Path(model_path).exists():
                    tracker.log_artifacts({"model_directory": model_path})

                training_results = {
                    "model_path": model_path,
                    "run_id": tracker.current_run.info.run_id,
                    "experiment_id": tracker.current_experiment.experiment_id,
                    "final_metrics": {
                        # Add final training metrics here
                    },
                }

                logger.info(f"✅ Tracked training completed: {model_path}")

                return training_results

        except Exception as e:
            logger.error(f"❌ Tracked training failed: {e}")
            raise

    def _run_enhanced_training(
        self,
        fine_tuner: SLMFineTuner,
        tracker: ExperimentTracker,
        training_kwargs: Dict[str, Any],
    ) -> str:
        """Run training with enhanced progress tracking"""

        # Override the training method to add progress tracking
        original_train_model = fine_tuner.train_model

        def tracked_train_model(*args, **kwargs):
            # Log training start
            tracker.log_metrics({"training_started": 1})

            # Run original training
            result = original_train_model(*args, **kwargs)

            # Log training completion
            tracker.log_metrics({"training_completed": 1})

            return result

        # Temporarily replace the method
        fine_tuner.train_model = tracked_train_model

        try:
            model_path = fine_tuner.train_model(
                train_dataset_path=training_kwargs["train_dataset_path"],
                validation_dataset_path=training_kwargs["validation_dataset_path"],
                num_epochs=training_kwargs["epochs"],
                learning_rate=training_kwargs["learning_rate"],
                batch_size=training_kwargs["batch_size"],
                model_output_name=training_kwargs["model_output_name"],
            )

            return model_path

        finally:
            # Restore original method
            fine_tuner.train_model = original_train_model

    def _run_model_validation(
        self, model_path: str, validation_dataset_path: str
    ) -> Dict[str, Any]:
        """Run comprehensive model validation"""

        try:
            validation_results = {
                "model_exists": Path(model_path).exists(),
                "model_size_mb": 0,
                "validation_metrics": {},
            }

            if validation_results["model_exists"]:
                # Calculate model size
                model_size = sum(
                    f.stat().st_size for f in Path(model_path).rglob("*") if f.is_file()
                )
                validation_results["model_size_mb"] = model_size / (1024 * 1024)

                # Run evaluation if validation dataset exists
                if Path(validation_dataset_path).exists():
                    fine_tuner = SLMFineTuner()
                    eval_results = fine_tuner.evaluate_model(
                        model_path, validation_dataset_path
                    )
                    validation_results["validation_metrics"] = eval_results

            logger.info(
                f"✅ Model validation completed: {validation_results['model_size_mb']:.1f}MB"
            )

            return validation_results

        except Exception as e:
            logger.error(f"❌ Model validation failed: {e}")
            return {"validation_error": str(e)}

    def _register_model(
        self,
        model_path: str,
        model_name: str,
        training_results: Dict[str, Any],
        validation_reports: Dict[str, Any],
        optimization_results: Optional[Dict[str, Any]],
        model_validation_results: Dict[str, Any],
        training_params: Dict[str, Any],
    ) -> Dict[str, Any]:
        """Register model in model registry with comprehensive metadata"""

        try:
            # Prepare metrics from various sources
            metrics = {}

            # Add validation metrics if available
            if "validation_metrics" in model_validation_results:
                metrics.update(model_validation_results["validation_metrics"])

            # Add optimization metrics if available
            if optimization_results:
                metrics["hpo_best_value"] = optimization_results["best_value"]
                metrics["hpo_trials"] = optimization_results["optimization_report"][
                    "results"
                ]["n_trials"]

            # Add data quality metrics
            if "train" in validation_reports:
                train_report = validation_reports["train"]
                metrics["data_quality_score"] = train_report.quality_score
                metrics["training_samples"] = train_report.total_samples

            # Prepare training information
            training_info = {
                "train_dataset_path": training_params.get("train_dataset_path", ""),
                "parameters": training_params,
                "duration_seconds": 0,  # Will be updated with actual duration
            }

            # Prepare tags
            tags = {
                "mlops_pipeline": "true",
                "optimization_used": "true" if optimization_results else "false",
                "data_quality": f"{metrics.get('data_quality_score', 0):.2f}",
                "training_samples": str(metrics.get("training_samples", 0)),
            }

            # Register model
            model_metadata = self.model_registry.register_model(
                model_path=model_path,
                model_name=model_name,
                model_type="fine-tuned",
                base_model=training_params.get(
                    "model_name", "microsoft/DialoGPT-small"
                ),
                description=f"MLOps trained model - {model_name}",
                metrics=metrics,
                training_info=training_info,
                experiment_id=training_results.get("experiment_id"),
                run_id=training_results.get("run_id"),
                tags=tags,
            )

            logger.info(
                f"🏛️  Model registered: {model_metadata.model_id} v{model_metadata.version}"
            )

            return {
                "model_id": model_metadata.model_id,
                "version": model_metadata.version,
                "registry_status": "success",
                "model_hash": model_metadata.model_hash,
                "model_size_mb": model_metadata.model_size_mb,
            }

        except Exception as e:
            logger.error(f"❌ Model registration failed: {e}")
            return {"registry_status": "failed", "error": str(e)}

    def _save_pipeline_results(self, results: Dict[str, Any]):
        """Save complete pipeline results"""

        try:
            results_dir = Path("mlops/pipeline_results")
            results_dir.mkdir(exist_ok=True)

            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            results_file = results_dir / f"pipeline_results_{timestamp}.json"

            with open(results_file, "w") as f:
                json.dump(results, f, indent=2, default=str)

            logger.info(f"💾 Pipeline results saved: {results_file}")

        except Exception as e:
            logger.error(f"❌ Failed to save pipeline results: {e}")


def run_mlops_training(
    train_dataset_path: str = "data/training/train_dataset.jsonl",
    validation_dataset_path: str = "data/training/validation_dataset.jsonl",
    model_name: str = "microsoft/DialoGPT-small",
    learning_rate: float = 5e-6,
    batch_size: int = 2,
    epochs: int = 3,
    force_cpu: bool = True,
    enable_hyperparameter_tuning: bool = False,
    enable_wandb: bool = False,
    **kwargs,
) -> Dict[str, Any]:
    """
    Convenient function to run MLOps training pipeline
    """

    # Setup logging
    logging.basicConfig(
        level=logging.INFO,
        format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
        handlers=[
            logging.StreamHandler(),
            logging.FileHandler("mlops/training_pipeline.log"),
        ],
    )

    # Initialize pipeline
    pipeline = MLOpsTrainingPipeline(
        enable_wandb=enable_wandb,
        enable_hyperparameter_tuning=enable_hyperparameter_tuning,
    )

    # Run pipeline
    return pipeline.run_full_pipeline(
        model_name=model_name,
        train_dataset_path=train_dataset_path,
        validation_dataset_path=validation_dataset_path,
        learning_rate=learning_rate,
        batch_size=batch_size,
        epochs=epochs,
        force_cpu=force_cpu,
        run_hyperparameter_optimization=enable_hyperparameter_tuning,
        **kwargs,
    )


if __name__ == "__main__":
    # Example: Run MLOps training pipeline
    results = run_mlops_training(
        train_dataset_path="data/training/train_dataset.jsonl",
        validation_dataset_path="data/training/validation_dataset.jsonl",
        epochs=3,
        learning_rate=5e-6,
        batch_size=2,
        force_cpu=True,
        enable_hyperparameter_tuning=False,  # Set to True for HPO
        enable_wandb=False,  # Set to True for W&B integration
    )

    print("🎉 MLOps Training Pipeline completed!")
    print(f"📊 Results: {results['pipeline_status']}")
    print(f"🤖 Model: {results['final_model_path']}")
