# -*- coding: utf-8 -*-
# ------------------------------------------------------------------------------
#
#   Copyright 2023 Valory AG
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.
#
# ------------------------------------------------------------------------------

"""
This script allows pushing a file directly to IPFS.

Usage:

python push_to_ipfs.py <file_path>
"""

from typing import Tuple

import multibase
import multicodec
from aea.helpers.cid import to_v1
from aea_cli_ipfs.ipfs_utils import IPFSTool


def push_to_ipfs(file_path: str) -> Tuple[str, str]:
    """
    Push a file to IPFS.

    :param file_path: Path of the file to be pushed to IPFS.
    :type file_path: str

    :return: A tuple containing v1_file_hash and v1_file_hash_hex.
    :rtype: Tuple[str, str]
    """
    response = IPFSTool().client.add(
        file_path, pin=True, recursive=True, wrap_with_directory=False
    )
    v1_file_hash = to_v1(response["Hash"])
    cid_bytes = multibase.decode(v1_file_hash)
    multihash_bytes = multicodec.remove_prefix(cid_bytes)
    v1_file_hash_hex = "f01" + multihash_bytes.hex()
    return v1_file_hash, v1_file_hash_hex


def main(file_path: str) -> None:
    """
    Push a file to IPFS and print the v1_file_hash and v1_file_hash_hex.

    :param file_path: Path of the file to be pushed to IPFS.
    :type file_path: str
    """
    v1_file_hash, v1_file_hash_hex = push_to_ipfs(file_path)
    print("IPFS file hash v1: {}".format(v1_file_hash))
    print("IPFS file hash v1 hex: {}".format(v1_file_hash_hex))
