"""The validation module contains functions to validate the serialized output of distributions.

This ensures that the Generative Metadata Format (GMF) files are interoperable and well formed.
"""

from __future__ import annotations

from copy import deepcopy
from importlib.metadata import entry_points

import jsonschema

from metasyn.distribution.na import NADistribution
from metasyn.registry import DistributionRegistry

SCHEMA_BASE = {
    "$schema": "https://json-schema.org/draft/2020-12/schema",
    "$id": "http://sodascience.github.io/generative_metadata_format/core/1.1/generative_metadata_format",  # noqa: E501
    "type": "object",
    "properties": {
        "n_rows": {"type": "number"},
        "n_columns": {"type": "number"},
        "provenance": {
            "type": "object",
            "properties": {
                "created by": {"type": "object"},
                "creation time": {"type": "string"}
            },
            "required": ["created by", "creation time"]
        },
        "vars": {
            "type": "array",
            "items": {
                "type": "object",
                "properties": {
                    "name": {"type": "string"},
                    "description": {"type": "string"},
                    "type": {"enum": ["discrete", "continuous", "string", "categorical", "date",
                                      "datetime", "time"]},
                    "dtype": {"type": "string"},
                    "prop_missing": {"type": "number"},
                    "distribution": {
                        "$ref": "#/$defs/all_dist_def"
                    }
                }
            },
            "required": ["name", "type", "dtype", "provenance", "prop_missing", "distribution"]
        }
    },
    "required": ["n_rows", "n_columns", "vars"],
}


def validate_gmf_dict(gmf_dict: dict):
    """Validate a JSON dictionary of a metaframe as it would be written to a GMF file.

    Make sure that you have used the _jsonify function to convert numpy arrays to
    lists, etc.

    Arguments
    ---------
    gmf_dict:
        Dictionary containing the metasyn output for a metaframe.
    """
    packages = [entry.name for entry in entry_points(group="metasyn.distribution_registry")]
    schema = create_schema(packages)
    jsonschema.validate(gmf_dict, schema)


def create_schema(packages: list[str]) -> dict:
    """Create JSON Schema to validate a GMF file.

    Arguments
    ---------
    packages:
        List of packages to create the schema with.

    Returns
    -------
    schema:
        Schema containing all the distributions in the distribution packages.
    """
    defs: list[dict] = []
    for fitter in DistributionRegistry.parse(packages).fitters:
        defs.append(fitter.distribution.schema())
    defs.append(NADistribution.schema())

    schema = deepcopy(SCHEMA_BASE)
    schema.update({"$defs": {"all_dist_def": {"anyOf": defs}}})
    return schema
