import httpx
from typing import List, Optional, Dict, Any

from .schemas import EvaluationResponse
from .exceptions import DeepRailsAPIError

class DeepRails:
    """
    Python SDK client for the DeepRails API.
    """
    def __init__(self, token: str, base_url: str = "https://api.deeprails.com"):
        """
        Initializes the DeepRails client.

        Args:
            token: Your DeepRails API key (starts with 'sk_').
            base_url: The base URL of the DeepRails API.
        """
        if not token:
            raise ValueError("A valid DeepRails API token is required.")
        
        self._base_url = base_url
        self._headers = {
            "Authorization": f"Bearer {token}",
            "Content-Type": "application/json",
            "User-Agent": "deeprails-python-sdk/0.2.0"
        }
        self._client = httpx.Client(base_url=self._base_url, headers=self._headers, timeout=30.0)

    def _request(self, method: str, endpoint: str, **kwargs) -> httpx.Response:
        """Helper method to make requests and handle API errors."""
        try:
            response = self._client.request(method, endpoint, **kwargs)
            response.raise_for_status()
            return response
        except httpx.HTTPStatusError as e:
            error_detail = "No detail provided."
            try:
                error_detail = e.response.json().get("detail", error_detail)
            except Exception:
                error_detail = e.response.text
            raise DeepRailsAPIError(status_code=e.response.status_code, error_detail=error_detail) from e
        except httpx.RequestError as e:
            raise DeepRailsAPIError(status_code=500, error_detail=f"Request failed: {e}") from e

    def create_evaluation(
        self,
        *,
        model_input: Dict[str, Any],
        model_output: str,
        model_used: Optional[str] = None,
        run_mode: Optional[str] = "smart",  # Set default to "smart"
        guardrail_metrics: Optional[List[str]] = None,
        nametag: Optional[str] = None,
        webhook: Optional[str] = None
    ) -> EvaluationResponse:
        """
        Creates a new evaluation and immediately processes it.

        Args:
            model_input: A dictionary containing the inputs for the model.
                         Must contain a "user_prompt" key.
            model_output: The response generated by the model you are evaluating.
            model_used: The name or identifier of the model being evaluated.
            run_mode: The evaluation mode (e.g., "smart", "dev").
            guardrail_metrics: A list of metrics to evaluate.
            nametag: A user-defined name or tag for the evaluation.
            webhook: A URL to send a POST request to upon evaluation completion.

        Returns:
            An EvaluationResponse object with the details of the created evaluation.
        """
        if "user_prompt" not in model_input:
            raise ValueError("`model_input` must contain a 'user_prompt' key.")

        payload = {
            "model_input": model_input,
            "model_output": model_output,
            "model_used": model_used,
            "run_mode": run_mode,
            "guardrail_metrics": guardrail_metrics,
            "nametag": nametag,
            "webhook": webhook,
        }
        json_payload = {k: v for k, v in payload.items() if v is not None}

        response = self._request("POST", "/evaluate", json=json_payload)
        return EvaluationResponse.parse_obj(response.json())

    def get_evaluation(self, eval_id: str) -> EvaluationResponse:
        """
        Retrieves the status and results of a specific evaluation.

        Args:
            eval_id: The unique identifier of the evaluation.

        Returns:
            An EvaluationResponse object with the full, up-to-date details of the evaluation.
        """
        response = self._request("GET", f"/evaluate/{eval_id}")
        return EvaluationResponse.parse_obj(response.json())