"""
This module defines the `PerturbationOcclusion` class, which implements a generator composed of
modular perturbation and occlusion-based algorithms
"""

from __future__ import annotations

from collections.abc import Hashable, Iterable, Sequence
from typing import Any, TypeVar

import numpy as np
from smqtk_core.configuration import (
    from_config_dict,
    make_default_config,
    to_config_dict,
)
from smqtk_detection.interfaces.detect_image_objects import DetectImageObjects
from smqtk_image_io.bbox import AxisAlignedBoundingBox
from typing_extensions import Self

from xaitk_saliency.interfaces.gen_detector_prop_sal import GenerateDetectorProposalSaliency
from xaitk_saliency.interfaces.gen_object_detector_blackbox_sal import GenerateObjectDetectorBlackboxSaliency
from xaitk_saliency.interfaces.perturb_image import PerturbImage
from xaitk_saliency.utils.detection import format_detection
from xaitk_saliency.utils.masking import occlude_image_batch

C = TypeVar("C", bound="PerturbationOcclusion")


class PerturbationOcclusion(GenerateObjectDetectorBlackboxSaliency):
    """
    Generator composed of modular perturbation and occlusion-based algorithms.

    This implementation exposes its `fill` attribute as public.
    This allows it to be set during runtime as this is most often driven by the
    black-box algorithm used, if at all.
    """

    def __init__(
        self,
        perturber: PerturbImage,
        generator: GenerateDetectorProposalSaliency,
        fill: int | Sequence[int] | np.ndarray | None = None,
        threads: int | None = 0,
    ) -> None:
        """
        Generator composed of modular perturbation and occlusion-based algorithms.

        :param perturber: `PerturbImage` implementation instance for generating
            occlusion masks.
        :param generator: `GenerateDetectorProposalSaliency` implementation
            instance for generating saliency masks given occlusion masks and
            black-box detector outputs.
        :param fill: Optional fill for alpha-blending the occluded regions based on
            the masks generated by the given perturber.
            Can be a scalar value, a per-channel sequence or a shape-matched image.
        :param threads: Optional number threads to use to enable parallelism in
            applying perturbation masks to an input image.
            If 0, a negative value, or `None`, work will be performed on the
            main-thread in-line.
        """
        self._perturber = perturber
        self._generator = generator
        self.fill = fill
        self._threads = threads

    def _generate(
        self,
        ref_image: np.ndarray,
        bboxes: np.ndarray,
        scores: np.ndarray,
        blackbox: DetectImageObjects,
        objectness: np.ndarray | None = None,
    ) -> np.ndarray:
        ref_dets_mat = format_detection(bboxes, scores, objectness)

        pert_masks = self._perturber(ref_image)

        pert_dets = blackbox.detect_objects(
            occlude_image_batch(ref_image, pert_masks, fill=self.fill, threads=self._threads),
        )

        pert_dets_mat = _dets_to_formatted_mat(pert_dets)

        if pert_dets_mat.shape[1] == 0:
            return np.array([])

        return self._generator(
            ref_dets_mat,
            pert_dets_mat,
            pert_masks,
        )

    @classmethod
    def get_default_config(cls) -> dict[str, Any]:
        """
        Returns the default configuration for the PerturbationOcclusion.

        This method provides a default configuration dictionary, specifying default
        values for key parameters in the factory. It can be used to create an instance
        of the factory with preset configurations.

        Returns:
            dict[str, Any]: A dictionary containing default configuration parameters.
        """
        cfg = super().get_default_config()
        cfg["perturber"] = make_default_config(PerturbImage.get_impls())
        cfg["generator"] = make_default_config(GenerateDetectorProposalSaliency.get_impls())
        return cfg

    @classmethod
    def from_config(cls, config_dict: dict, merge_default: bool = True) -> Self:
        """
        Create a PerturbationOcclusion instance from a configuration dictionary.

        Args:
            config_dict (dict): Configuration dictionary with perturber details.
            merge_default (bool): Whether to merge with the default configuration.

        Returns:
            PerturbationOcclusion: An instance of PerturbationOcclusion.
        """
        config_dict = dict(config_dict)  # shallow-copy
        config_dict["perturber"] = from_config_dict(config_dict["perturber"], PerturbImage.get_impls())
        config_dict["generator"] = from_config_dict(
            config_dict["generator"],
            GenerateDetectorProposalSaliency.get_impls(),
        )
        return super().from_config(config_dict, merge_default=merge_default)

    def get_config(self) -> dict[str, Any]:
        """
        Get the configuration dictionary of the PerturbationOcclusion instance.

        Returns:
            dict[str, Any]: Configuration dictionary.
        """
        return {
            "perturber": to_config_dict(self._perturber),
            "generator": to_config_dict(self._generator),
            "fill": self.fill,
            "threads": self._threads,
        }


def _dets_to_formatted_mat(
    dets: Iterable[Iterable[tuple[AxisAlignedBoundingBox, dict[Hashable, float]]]],
) -> np.ndarray:
    """
    Converts detections, as returned by an implementation of
    ``DetectImageObjects``, into a detection matrix formatted for use with
    an implementation of ``GenerateDetectorProposalSaliency``.
    The order of the class scores in the resulting matrix follows the order of
    labels present in the first non-empty detection in the input set.

    :param dets: Detections, as returned by an implementation of
        ``DetectImageObjects``.

    :returns: Matrix of detections with shape
        [nImgs x nDets x (4+1+nClasses)].
        If the number of detections for each image is not consistent, the matrix
        will be padded with rows of ones, except for the objectness which is set
        to zero.
    """
    labels: list[Hashable] = []
    num_classes = 0
    dets_mat_list: list[np.ndarray] = []
    for img_idx, img_dets in enumerate(dets):
        img_bboxes = np.array([])
        img_scores = np.array([])
        img_objectness = np.array([])

        # reshape for vertical stacking
        img_bboxes.shape = (0, 4)
        img_scores.shape = (0, num_classes)

        for det in img_dets:
            obj = 1.0
            bbox = det[0]
            score_dict = det[1]

            # use class labels of first non-empty detection
            if num_classes == 0:
                labels = list(score_dict.keys())
                num_classes = len(labels)
                img_scores.shape = (0, num_classes)
                # reshape previous mats for padding later
                dets_mat_list[0:img_idx] = [np.array([]).reshape(0, 4 + 1 + num_classes) for _ in range(img_idx)]

            scores = [score_dict[label] for label in labels]

            # single class score only
            if len([score for score in scores if score > 0]) == 1:
                conf = max(scores)
                obj = conf  # replace objectness with class score
                scores[scores.index(conf)] = 1  # one-hot encode class score

            img_bboxes = np.vstack((img_bboxes, [*bbox.min_vertex, *bbox.max_vertex]))
            img_scores = np.vstack((img_scores, scores))
            img_objectness = np.hstack((img_objectness, obj))

        dets_mat_list.append(format_detection(img_bboxes, img_scores, img_objectness))

    return np.asarray(_format_output(dets_mat_list=dets_mat_list, num_classes=num_classes))


def _format_output(dets_mat_list: list[np.ndarray], num_classes: int) -> Sequence[np.ndarray]:
    # pad matrices
    num_dets = [dets_mat.shape[0] for dets_mat in dets_mat_list]
    max_dets = max(num_dets)
    pad_row = np.ones(4 + 1 + num_classes)
    # set objectness to zero
    pad_row[4] = 0
    for i, dets_mat in enumerate(dets_mat_list):
        size_diff = max_dets - dets_mat.shape[0]
        dets_mat_list[i] = np.vstack((dets_mat, np.tile(pad_row, (size_diff, 1))))
    return dets_mat_list
