import json
import re
from collections import OrderedDict
from unittest import TestCase, mock

import pytest
from conftool import configuration, drivers
from conftool.kvobject import Field, Entity
from conftool.tests.unit import (
    MockBackend,
    MockEntity,
    MockFreeEntity,
    MockJsonEntity,
)
from conftool.types import get_validator


class TestKVObject(TestCase):
    def setUp(self):
        Entity.backend = MockBackend({})
        Entity.config = configuration.Config(driver="")
        self.entity = MockEntity("Foo", "Bar", "test")

    def test_tags(self):
        """
        test `Entity.tags`
        """
        ent = MockEntity("Foo", "Bar", "test")
        self.assertEqual(ent.tags, {"foo": "Foo", "bar": "Bar"})

    def test_kvpath(self):
        """
        Test `Entity.kvpath`
        """
        self.assertEqual("Mock/entity/bar/baz", self.entity.kvpath("bar", "baz"))

    def test_query_success(self):
        """
        Test `KvObject.query` finds a valid result
        """
        MockEntity.backend.driver.all_keys = mock.Mock(
            return_value=[["Foo", "Bar", "test"], ["Foo", "Baz", "test1"]]
        )
        res = [el for el in MockEntity.query({"bar": re.compile("Bar")})]
        self.assertEqual("test", res[0].name)
        self.assertEqual(1, len(res))
        res = [el for el in MockEntity.query({"name": re.compile("tes.*")})]
        self.assertEqual(2, len(res))

    def test_all(self):
        """
        Test `KvObject.all` returns all the keys
        """
        MockEntity.backend.driver.all_keys = mock.Mock(
            return_value=[["Foo", "Bar", "test"], ["Foo", "Baz", "test1"]]
        )
        res = MockEntity.all()
        self.assertEqual(2, len(res))
        self.assertEqual("test", res[0].name)
        self.assertEqual("test1", res[1].name)

    def test_all_no_result(self):
        """
        Test `KvObject.all` returns an empty list when no result is available
        """
        MockEntity.backend.driver.all_keys = mock.Mock(return_value=[])
        self.assertEqual([], MockEntity.all())

    def test_query_no_result(self):
        """
        Test `KvObject.query` returns an empty list when no result is available
        """
        MockEntity.backend.driver.all_keys = mock.Mock(
            return_value=[["Foo", "Bar", "test"], ["Foo", "Baz", "test1"]]
        )
        res = [el for el in MockEntity.query({"bar": re.compile("Far")})]
        self.assertEqual([], res)

    def test_query_wrong_tag(self):
        """
        Test `KvObject.query` returns no result when we pass a wrong tag to it.
        """
        MockEntity.backend.driver.all_keys = mock.Mock(
            return_value=[["Foo", "Bar", "test"], ["Foo", "Baz", "test1"]]
        )
        with pytest.raises(ValueError, match=r": rosafante"):
            list(MockEntity.query({"rosafante": re.compile("nope")}))

    def test_properties(self):
        self.assertEqual(self.entity.name, "test")
        self.assertEqual(self.entity.key, "Mock/entity/Foo/Bar/test")
        self.assertEqual(self.entity.tags, {"foo": "Foo", "bar": "Bar"})

    def test_fields(self):
        self.assertEqual(set(self.entity.field_names()), {"a", "b"})
        self.assertEqual(self.entity.get_field("a"), Field("a", "int", 1, "This is a", example=42))
        self.assertEqual(self.entity.get_field("b"), Field("b", "string", "FooBar", hidden=True))

    def test_fetch(self):
        MockEntity.backend.driver.read = mock.Mock(
            return_value=drivers.ObjectWireRepresentation(
                {"a": 1, "b": "b-val"}, drivers.ObjectWireMetadata(47)
            )
        )
        with mock.patch("conftool.tests.unit.MockEntity.from_net") as mocker:
            a = MockEntity("Foo", "Bar", "test")
            mocker.assert_called_with({"a": 1, "b": "b-val"})
            assert a.revision_id == 47

            # Non-existent key?
            MockEntity.backend.driver.read.side_effect = drivers.NotFoundError("test")
            mocker.reset_mock()
            a = MockEntity("Foo", "Bar", "test")
            mocker.assert_called_with(None)
            assert a.revision_id < 0

            MockEntity.backend.driver.read.side_effect = drivers.BackendError("testtest")
            mocker.reset_mock()
            a.fetch()
            mocker.assert_not_called()

    def test_write(self):
        MockEntity.backend.driver.write = mock.Mock(return_value={"a": 5, "b": "meh"})
        obj = MockEntity("Foo", "Baz", "new")
        res = obj.write()
        MockEntity.backend.driver.write.assert_called_with(
            "Mock/entity/Foo/Baz/new", {"a": 1, "b": "FooBar"}
        )
        self.assertEqual(res, {"a": 5, "b": "meh"})
        obj = MockEntity("Foo", "Baz", "new")
        res = obj.write()
        # A driver exception gets passed to us
        MockEntity.backend.driver.write.side_effect = ValueError("bad json, bad!")
        self.assertRaises(ValueError, obj.write)

    def test_write_read_only(self):
        """Test writing when in read-only mode does not write"""
        MockEntity.backend.driver.write = mock.Mock(return_value={"a": 5, "b": "meh"})
        old_read_only = Entity.config.read_only
        try:
            Entity.config.read_only = True
            obj = MockEntity("Foo", "Baz", "new")
            assert obj.config
            obj.write()
            MockEntity.backend.driver.write.assert_not_called()
        finally:
            Entity.config.read_only = old_read_only

    def test_delete(self):
        MockEntity.backend.driver.delete = mock.Mock(return_value=None)
        obj = MockEntity("Foo", "Baz", "new")
        obj.delete()
        MockEntity.backend.driver.delete.assert_called_with("Mock/entity/Foo/Baz/new")
        # A driver exception gets passed to us
        MockEntity.backend.driver.delete.side_effect = drivers.BackendError("something")
        self.assertRaises(drivers.BackendError, obj.delete)

    def test_delete_read_only(self):
        """Test deleting when in read-only mode does not delete"""
        old_read_only = Entity.config.read_only
        MockEntity.backend.driver.delete = mock.Mock(return_value=None)
        try:
            Entity.config.read_only = True
            obj = MockEntity("Foo", "Baz", "new")
            obj.delete()
            MockEntity.backend.driver.delete.assert_not_called()
        finally:
            Entity.config.read_only = old_read_only

    def test_parse_tags(self):
        # Correct tags list
        taglist = ["bar=Bar", "foo=Foo"]
        self.assertEqual(MockEntity.parse_tags(taglist), ["Foo", "Bar"])
        # Additional tags are just discarded
        taglist = ["a=n", "bar=Bar", "foo=Foo"]
        self.assertEqual(MockEntity.parse_tags(taglist), ["Foo", "Bar"])

    def test_update(self):
        self.entity.write = mock.Mock()
        self.entity._set_value = mock.Mock(side_effect=self.entity._set_value)
        # Setting a value not in the schema doesn't do anything
        self.entity.update({"c": "meh"})
        self.entity._set_value.assert_not_called()
        # Setting a value in the schema does set it
        self.entity.update({"a": 10})
        self.entity._set_value.assert_called_with(
            "a", get_validator("int"), {"a": 10}, set_defaults=False
        )
        self.entity.write.assert_called_with()

    def test_validate(self):
        self.assertTrue(self.entity.validate({"a": 1, "b": "testtest"}))
        self.assertRaises(TypeError, self.entity.validate, {"a": 1, "b": "testtest", "c": True})
        self.assertRaises(ValueError, self.entity.validate, {"a": "test"})

    def test_to_net(self):
        self.entity.a = 100
        self.entity.b = "meoow"
        self.assertEqual(self.entity._to_net(), {"a": 100, "b": "meoow"})
        obj = MockEntity("a", "b", "c")
        obj.b = "fooBar"
        self.assertEqual(obj._to_net(), {"a": 1, "b": "fooBar"})
        del obj.b
        self.assertEqual(obj._to_net(), {"a": 1, "b": "FooBar"})

    def test_from_net(self):
        obj = MockEntity("a", "b", "c")
        obj.from_net({"a": 256})
        self.assertEqual(obj._to_net(), {"a": 256, "b": "FooBar"})

    def test_set_value(self):
        with mock.patch("conftool.tests.unit.MockEntity.fetch"):
            obj = MockEntity("a", "b", "c")
        # set an existing value
        obj._set_value("a", get_validator("int"), {"a": 256})
        self.assertEqual(obj.a, 256)
        # Set an inexistent value with no defaults
        obj._set_value("c", get_validator("string"), {})
        self.assertEqual(obj.c, "FooBar")

    def test_asdict(self):
        asdict = self.entity.asdict()
        self.assertIsInstance(asdict, OrderedDict)
        self.assertListEqual(sorted(list(asdict.keys())), ["tags", "test"])
        self.assertEqual(asdict["tags"], "foo=Foo,bar=Bar")
        self.assertEqual(asdict["test"]["a"], 1)

    def test_str(self):
        self.assertEqual(str(self.entity), json.dumps(self.entity.asdict()))

    def test_eq(self):
        ent = MockEntity("Foo", "Bar", "test")
        self.assertEqual(ent, self.entity)
        ent1 = MockEntity("Foo", "Bar", "test1")
        self.assertNotEqual(ent1, self.entity)
        ent2 = MockEntity("Foo2", "Bar", "test")
        self.assertNotEqual(ent2, self.entity)
        ent.a = 256
        self.assertNotEqual(ent, self.entity)

    def test_from_yaml_no_tags(self):
        Entity._tags = mock.MagicMock(return_value=[])
        self.assertEqual({"a": None}, Entity.from_yaml(["a"]))


class TestEntity(TestCase):
    def setUp(self):
        Entity.backend = MockBackend({})
        Entity.config = configuration.Config(driver="")
        self.entity = MockEntity("Foo", "Bar", "test")

    def test_init(self):
        self.assertEqual(self.entity.tags, {"foo": "Foo", "bar": "Bar"})
        self.assertEqual(self.entity.name, "test")
        # Now let's pass some wrong tags
        self.assertRaises(ValueError, MockEntity, "Foo", "Bar", "Baz", "test")

    def test_pprint(self):
        self.assertEqual(self.entity.pprint(), "Foo/Bar/test")

    def test_dir(self):
        self.assertEqual(MockEntity.dir("Foo", "Bar"), "Mock/entity/Foo/Bar")
        self.assertRaises(ValueError, MockEntity.dir, "Foo")


class TestFreeSchemaObject(TestCase):
    def setUp(self):
        Entity.backend = MockBackend({})
        Entity.config = configuration.Config(driver="")

    def test_init(self):
        a = MockFreeEntity("Foo", "Bar", "test", some_key="some_value")
        self.assertEqual(a._schemaless, {"some_key": "some_value"})

    def test_to_net(self):
        a = MockFreeEntity("Foo", "Bar", "test", some_key="some_value")
        a.a = 240
        self.assertEqual(a._to_net(), {"a": 240, "b": "FooBar", "some_key": "some_value"})

    def test_from_net(self):
        a = MockFreeEntity("Foo", "Bar", "test", some_key="some_value")
        a.from_net({"some_key": "another_value", "m": 5})
        self.assertEqual(a.a, 1)
        self.assertEqual(a._schemaless["some_key"], "another_value")

    def test_changed(self):
        a = MockFreeEntity("Foo", "Bar", "test", some_key="some_value")

        data = {"a": 1, "b": "FooBar", "some_key": "some_value"}
        self.assertEqual(data, a._to_net())
        self.assertFalse(a.changed(data))
        a.a = 2
        self.assertTrue(a.changed(data))


class TestJsonSchemaObject(TestCase):
    def setUp(self):
        Entity.backend = MockBackend({})
        Entity.config = configuration.Config(driver="")

    def test_init(self):
        a = MockJsonEntity("Foo", "Bar", "test")
        self.assertEqual(a.val, "")

    def test_validate(self):
        a = MockJsonEntity("Foo", "Bar", "test")
        self.assertTrue(a.validate({"val": "testvalue"}))
