# TextQL Python SDK

Python client library for TextQL with a clean, intuitive API.

## Installation

```bash
pip install textql
```

## Quick Start

```python
from textql import TextQLClient, ChatTools

# Initialize client
client = TextQLClient(
    api_key='your-api-key',
    base_url='https://staging.textql.com'
)

# Configure tools
tools = ChatTools(
    connector_ids=[513],
    web_search_enabled=True,
    sql_enabled=True,
    python_enabled=True
)

# Stream a chat conversation
stream = client.chat.stream(
    question="Tell me about the connected datasource",
    chat_id="existing-chat-id",  # Optional: continue existing chat
    tools=tools
)

for response in stream:
    if response.HasField('text'):
        print(response.text, end='', flush=True)
    elif response.HasField('metadata'):
        print(f"\nChat ID: {response.metadata.chat_id}")
        print(f"Status: {response.metadata.status}")
```

## Features

### Chat API

Stream or send chat messages with full tool configuration:

```python
# Streaming (recommended for real-time responses)
stream = client.chat.stream(
    question="What's the total revenue?",
    tools=tools
)

# Non-streaming (get complete response at once)
response = client.chat.chat(
    question="What's the total revenue?",
    tools=tools
)
print(response.response)
```

### Connectors API

List and manage data connectors:

```python
connectors = client.connectors.list()
for connector in connectors.connectors:
    print(f"{connector.id}: {connector.name}")
```

### Playbooks API

List and execute playbooks:

```python
playbooks = client.playbooks.list()
for playbook in playbooks.playbooks:
    print(f"{playbook.id}: {playbook.name}")
```

## ChatTools Configuration

Configure which capabilities are enabled for your chat:

```python
tools = ChatTools(
    connector_ids=[513, 514],      # Database connectors to use
    web_search_enabled=True,        # Enable web search
    sql_enabled=True,               # Enable SQL queries
    python_enabled=True,            # Enable Python execution
    ontology_enabled=False,         # Enable ontology queries
    tableau_enabled=False,          # Enable Tableau integration
    streamlit_enabled=False,        # Enable Streamlit apps
    powerbi_enabled=False,          # Enable PowerBI integration
    google_drive_enabled=False,     # Enable Google Drive access
    experimental_enabled=False,     # Enable experimental features
    auto_approve_enabled=False,     # Auto-approve code execution
)
```

## Context Manager Support

Use the client as a context manager for automatic cleanup:

```python
with TextQLClient(api_key='your-api-key', base_url='https://staging.textql.com') as client:
    response = client.chat.chat(question="Hello!")
    print(response.response)
# Channel automatically closed
```

## Requirements

- Python >= 3.8
- grpcio >= 1.68.0
- protobuf >= 5.28.0

## License

Copyright TextQL Labs
