import subprocess
from pathlib import Path

from acb.console import Console
from acb.depends import Inject, depends
from acb.logger import Logger


class AutofixCoordinator:
    @depends.inject
    def __init__(
        self,
        console: Inject[Console],
        pkg_path: Path,
        logger: Inject[Logger],
    ) -> None:
        self.console = console
        self.pkg_path = pkg_path
        # Bind logger context with name for tracing
        self.logger = logger.bind(logger="crackerjack.autofix")

    def apply_autofix_for_hooks(self, mode: str, hook_results: list[object]) -> bool:
        try:
            if self._should_skip_autofix(hook_results):
                return False

            if mode == "fast":
                return self._apply_fast_stage_fixes()
            elif mode == "comprehensive":
                return self._apply_comprehensive_stage_fixes(hook_results)
            else:
                self.logger.warning(f"Unknown autofix mode: {mode}")
                return False
        except Exception as e:
            self.logger.exception("Error applying autofix", error=str(e))
            return False

    def apply_fast_stage_fixes(self) -> bool:
        return self._apply_fast_stage_fixes()

    def apply_comprehensive_stage_fixes(self, hook_results: list[object]) -> bool:
        return self._apply_comprehensive_stage_fixes(hook_results)

    def run_fix_command(self, cmd: list[str], description: str) -> bool:
        return self._run_fix_command(cmd, description)

    def check_tool_success_patterns(self, cmd: list[str], result: object) -> bool:
        return self._check_tool_success_patterns(cmd, result)

    def validate_fix_command(self, cmd: list[str]) -> bool:
        return self._validate_fix_command(cmd)

    def validate_hook_result(self, result: object) -> bool:
        return self._validate_hook_result(result)

    def should_skip_autofix(self, hook_results: list[object]) -> bool:
        return self._should_skip_autofix(hook_results)

    def _apply_fast_stage_fixes(self) -> bool:
        return self._execute_fast_fixes()

    def _apply_comprehensive_stage_fixes(self, hook_results: list[object]) -> bool:
        failed_hooks = self._extract_failed_hooks(hook_results)
        if not failed_hooks:
            return True

        hook_specific_fixes = self._get_hook_specific_fixes(failed_hooks)

        if not self._execute_fast_fixes():
            return False

        all_successful = True
        for cmd, description in hook_specific_fixes:
            if not self._run_fix_command(cmd, description):
                all_successful = False

        return all_successful

    def _extract_failed_hooks(self, hook_results: list[object]) -> set[str]:
        failed_hooks = set()
        for result in hook_results:
            if (
                self._validate_hook_result(result)
                and getattr(result, "status", "") == "Failed"
            ):
                failed_hooks.add(getattr(result, "name", ""))
        return failed_hooks

    def _get_hook_specific_fixes(
        self, failed_hooks: set[str]
    ) -> list[tuple[list[str], str]]:
        fixes = []

        if "bandit" in failed_hooks:
            fixes.append((["uv", "run", "bandit", "-r", "."], "bandit analysis"))

        return fixes

    def _execute_fast_fixes(self) -> bool:
        fixes = [
            (["uv", "run", "ruff", "format", "."], "format code"),
            (["uv", "run", "ruff", "check", "--fix", "."], "fix code style"),
        ]

        all_successful = True
        for cmd, description in fixes:
            if not self._run_fix_command(cmd, description):
                all_successful = False

        return all_successful

    def _run_fix_command(self, cmd: list[str], description: str) -> bool:
        if not self._validate_fix_command(cmd):
            self.logger.warning(f"Invalid fix command: {cmd}")
            return False

        try:
            self.logger.info(f"Running fix command: {description}")
            result = subprocess.run(
                cmd,
                cwd=self.pkg_path,
                capture_output=True,
                text=True,
                timeout=300,
            )
            return self._handle_command_result(result, description)
        except Exception as e:
            self.logger.exception(
                f"Error running fix command: {description}", error=str(e)
            )
            return False

    def _handle_command_result(
        self, result: subprocess.CompletedProcess[str], description: str
    ) -> bool:
        if result.returncode == 0:
            self.logger.info(f"Fix command succeeded: {description}")
            return True

        if self._is_successful_fix(result):
            self.logger.info(f"Fix command applied changes: {description}")
            return True

        self.logger.warning(
            f"Fix command failed: {description}",
            returncode=result.returncode,
            stderr=result.stderr[:200] if result.stderr else "No stderr",
        )
        return False

    def _is_successful_fix(self, result: subprocess.CompletedProcess[str]) -> bool:
        success_indicators = [
            "fixed",
            "formatted",
            "reformatted",
            "updated",
            "changed",
            "removed",
        ]

        if hasattr(result, "stdout") and hasattr(result, "stderr"):
            stdout = getattr(result, "stdout", "") or ""
            stderr = getattr(result, "stderr", "") or ""

            if not isinstance(stdout, str):
                stdout = str(stdout)
            if not isinstance(stderr, str):
                stderr = str(stderr)
            output = stdout + stderr
        else:
            output = str(result)

        output_lower = output.lower()

        return any(indicator in output_lower for indicator in success_indicators)

    def _check_tool_success_patterns(self, cmd: list[str], result: object) -> bool:
        if not cmd:
            return False

        if hasattr(result, "returncode"):
            return self._check_process_result_success(result)

        if isinstance(result, str):
            return self._check_string_result_success(result)

        return False

    def _check_process_result_success(self, result: object) -> bool:
        if getattr(result, "returncode", 1) == 0:
            return True

        output = self._extract_process_output(result)
        return self._has_success_patterns(output)

    def _extract_process_output(self, result: object) -> str:
        stdout = getattr(result, "stdout", "") or ""
        stderr = getattr(result, "stderr", "") or ""

        if not isinstance(stdout, str):
            stdout = str(stdout)
        if not isinstance(stderr, str):
            stderr = str(stderr)

        return stdout + stderr

    def _check_string_result_success(self, result: str) -> bool:
        return self._has_success_patterns(result)

    def _has_success_patterns(self, output: str) -> bool:
        if not output:
            return False

        success_patterns = [
            "fixed",
            "formatted",
            "reformatted",
            "would reformat",
            "fixing",
        ]

        output_lower = output.lower()
        return any(pattern in output_lower for pattern in success_patterns)

    def _validate_fix_command(self, cmd: list[str]) -> bool:
        if not cmd or len(cmd) < 2:
            return False

        if cmd[0] != "uv":
            return False

        if cmd[1] != "run":
            return False

        allowed_tools = [
            "ruff",
            "bandit",
            "trailing-whitespace",
        ]

        if len(cmd) > 2 and cmd[2] in allowed_tools:
            return True

        return False

    def _validate_hook_result(self, result: object) -> bool:
        name = getattr(result, "name", None)
        status = getattr(result, "status", None)

        if not name or not isinstance(name, str):
            return False

        if not status or not isinstance(status, str):
            return False

        valid_statuses = ["Passed", "Failed", "Skipped", "Error"]
        if status not in valid_statuses:
            return False

        return True

    def _should_skip_autofix(self, hook_results: list[object]) -> bool:
        for result in hook_results:
            raw_output = getattr(result, "raw_output", None)
            if raw_output:
                output_lower = raw_output.lower()

                if (
                    "importerror" in output_lower
                    or "modulenotfounderror" in output_lower
                ):
                    self.logger.info("Skipping autofix for import errors")
                    return True
        return False
