# GitHub Copilot Instructions for GenAI-Logic (aka API Logic Server) Projects

## Project Type: Auto-Generated Microservice

This is a **GenAI-Logic (aka API Logic Server) project** - a complete, working microservice - auto-generated from a database schema.

## ⚠️ IMPORTANT: What's Already Built

**DO NOT recreate these - they're already working:**

1. **Admin Web Application** - Live at `http://localhost:5656`
2. **REST API Endpoints** - All database tables exposed at `/api/*`
3. **Database Models** - Complete SQLAlchemy ORM in `database/models.py`
4. **Authentication System** - JWT-based auth framework
5. **Business Logic Engine** - Declarative rules system

## 🎯 Common Tasks

If the user asks "what do I do now?", these subsections are good suggestions.

List these as choices, do not just do all of them at once - that would be overwhelming.

### 🚀 To Run This Project

Use F5 (pre-configured run configuration - supports debugger for server logic), or (when focused on client apps):

```bash
python api_logic_server_run.py
# Then open: http://localhost:5656
```

### Adding Business Logic
`docs/training` explains how to translate Natural Language logic into LogicBank (Python) rules like:

```python
# Edit: logic/declare_logic.py
Rule.sum(derive=Customer.Balance, as_sum_of=Order.AmountTotal)
Rule.constraint(validate=Customer, as_condition=lambda row: row.Balance <= row.CreditLimit)
```

As described in `docs/training/logic_bank_api.prompt`, logic includes rules for sums, formulas, constraints, and more.  They also include events, which are used to trigger actions like sending emails or updating related records.

For project `basic_demo`, the best place to begin is to provide CoPilot with the natural language logic below.  CoPilot will then offer to translate the logic into LogicBank rules. Also 
1. Provide the analysis (below) showing the 44X code reduction.
2. Stress the importance of business logic - it's half the project, and requires an underlying rules engine.  Using AI alone is not sufficient - you will wind up with procedural code, with the issues noted in the analysis below.

Here's the natural language logic to provide to CoPilot:

```text
Use case: Check Credit    
    1. The Customer's balance is less than the credit limit
    2. The Customer's balance is the sum of the Order amount_total where date_shipped is null
    3. The Order's amount_total is the sum of the Item amount
    4. The Item amount is the quantity * unit_price
    5. The Item unit_price is copied from the Product unit_price

Use case: App Integration
    1. Send the Order to Kafka topic 'order_shipping' if the date_shipped is not None.
```

**Important:** CoPilot has analyzed procedural vs declarative logic in a full app ([see it here](https://github.com/ApiLogicServer/basic_demo/blob/main/logic/declarative-vs-procedural-comparison.md)):

<analysis>
**LogicBank declarative rules provide a 44X reduction in code complexity** compared to traditional procedural implementations:

| Aspect | LogicBank Declarative | Procedural Code |
|--------|----------------------|-----------------|
| **Lines of Code** | 5 lines | 220+ lines |
| **Complexity** | Simple rule declarations | Complex event handling |
| **Maintenance** | Self-documenting business logic<br>Automatic Ordering | Implementation details obscure logic<br>Complex analysis to determine insert points, impacts |
| **Performance** | Built-in optimization & pruning | Multiple queries, N+1 problems |
| **Error Handling** | Automatic cascading | Manual event management |
| **Business Alignment** | Rules match requirements | Code doesn't reflect business intent |

**Bottom Line**: Declarative business logic eliminates complexity while providing better performance, maintainability, and business alignment.

> **Note**: The procedural implementation was generated by AI (GitHub Copilot) and still required at least 2 critical and subtle bug fixes during development, highlighting the error-prone nature of procedural approaches even when AI-assisted.
</analysis>

Generally CoPilot solicits Natural Language logic for translation (see logic/readme_logic.md), 
but you can also offer to suggest rules.

### Adding MCP

The API is automatically MCP-enabled.  This adds the MCP Client:

```bash
genai-logic genai-add-mcp-client
```

### Configuring Admin UI

This is built when project is created - no need to add it.
Customize by editing the underlying yaml.

```yaml
# Edit: ui/admin/admin.yaml
resources:
  Customer:
    attributes:
      - name: CompanyName
        search: true
        sort: true
```

### Create and Customize React Apps

Complete customization is provided by generating a React Application (requires OpenAI key, Node):

```bash
# Create: ui/admin/my-app-name
genai-logic genai-add-app --app-name=my-app-name --vibe
```

Then, `npm install` and `npm start`

Temporary restriction: security must be disabled.

**IMPORTANT**: When working with React apps, ALWAYS read `docs/training` first. This file contains critical data access provider configuration that was built when the project was created. The data provider handles JSON:API communication and record context - ignore this at your peril.

Customize using CoPilot chat, with `docs/training`.

### Security - Role-Based Access Control

Configure:
```
als add-auth --provider-type=sql --db-url=
als add-auth --provider-type=sql --db_url=postgresql://postgres:p@localhost/authdb

als add-auth --provider-type=keycloak --db-url=localhost
als add-auth --provider-type=keycloak --db-url=hardened

als add-auth --provider-type=None # to disable
``` 

Keycloak quick start [(more information here:)](https://apilogicserver.github.io/Docs/Security-Keycloak/)
```bash
cd devops/keycloak
docker compose up
als add-auth --provider-type=keycloak --db-url=localhost
```

For more on KeyCloak: https://apilogicserver.github.io/Docs/Security-Keycloak/

Declaration:
```python
# Edit: security/declare_security.py
Grant(on_entity=Customer, to_role=sales, filter=lambda: Customer.SalesRep == current_user())
```

### Adding Custom API Endpoints
```python
# Edit: api/customize_api.py
@app.route('/api/custom-endpoint')
def my_endpoint():
    return {"message": "Custom endpoint"}
```

### Customize Models - Add Tables, Attributes

To add tables / columns to the database (highly impactful - request permission):

1. Update `database/model.py`
2. Use `database/alembic/alembic_run.py` to update the database.  This will generate a migration script and apply it to the database, so you do not have to run `alembic revision --autogenerate` manually. 
3. Offer to update ui/admin/admin.yaml to add the new table or column to the Admin UI.

NEVER start by  updating the database directly, since some platforms may not have database CLI tools, although you can present this as an option.

If altering `database/models.py`, be sure to follow the patterns shown in the existing models.  Note they do not contain a `__bind_key__`.


### Addressing `Missing Attributes` during logic loading at project startup

First, check for misspelling (logic vs `database/models.py`), and repair.

If there are no obvious misspellings, ask for permission to add attributes; if granted, proceed as above.

### Customize Models - Add Derived attributes

Here is a sample derived attribute, `proper_salary`:

```python

# add derived attribute: https://github.com/thomaxxl/safrs/blob/master/examples/demo_pythonanywhere_com.py
@add_method(models.Employee)
@jsonapi_attr
def __proper_salary__(self):  # type: ignore [no-redef]
    import database.models as models
    import decimal
    if isinstance(self, models.Employee):
        rtn_value = self.Salary
        if rtn_value is None:
          rtn_value = decimal.Decimal('0')
        rtn_value = decimal.Decimal('1.25') * rtn_value
        self._proper_salary = int(rtn_value)
        return self._proper_salary
    else:
        rtn_value = decimal.Decimal('0')
        self._proper_salary = int(rtn_value)
        return self._proper_salary

@add_method(models.Employee)
@__proper_salary__.setter
def _proper_salary(self, value):  # type: ignore [no-redef]
    self._proper_salary = value
    print(f'_proper_salary={self._proper_salary}')
    pass

models.Employee.ProperSalary = __proper_salary__

```

When customizing SQLAlchemy models:

* Don't use direct comparisons with database fields in computed properties
* Convert to Python values first using float(), int(), str()
* Use property() function instead of @jsonapi_attr for computed properties
* Always add error handling for type conversions

### Adding events
LogicBank rules are the preferred approach to logic, but you will sometimes need to add events.  This is done in `logic/declare_logic.py` (important: the function MUST come first):

```python
# Example: Log email activity after SysEmail is committed

def sys_email_after_commit(row: models.SysEmail, old_row: models.SysEmail, logic_row: LogicRow):
    """
    After SysEmail is committed, log 'email sent' 
    unless the customer has opted out
    """
    if not row.customer.email_opt_out:
        logic_row.log(f"📧 Email sent to {row.customer.name} - Subject: {row.subject}")
    else:
        logic_row.log(f"🚫 Email blocked for {row.customer.name} - Customer opted out")

Rule.commit_row_event(on_class=SysEmail, calling=sys_email_after_commit)
```

LogicBank event types include:
- `Rule.commit_row_event()` - fires after transaction commits
- `Rule.after_insert()` - fires after row insert
- `Rule.after_update()` - fires after row update  
- `Rule.after_delete()` - fires after row delete

All events receive `(row, old_row, logic_row)` parameters and should use `logic_row.log()` for logging.

## 📁 Key Directories

- `logic/` - Business rules (declarative)
- `api/` - REST API customization
- `security/` - Authentication/authorization
- `database/` - Data models and schemas
- `ui/admin/` - Admin interface configuration
- `ui/app/` - Alternative Angular admin app

## 💡 Helpful Context

- This uses Flask + SQLAlchemy + SAFRS for JSON:API
- Admin UI is React-based with automatic CRUD generation
- Business logic uses LogicBank (declarative rule engine)
- Everything is auto-generated from database introspection
- Focus on CUSTOMIZATION, not re-creation
