import os
import subprocess
import sys

'''
Pushes the current database/models.py to the database.
python database/alembic/alembic_run.py [--non-interactive]
'''

def prompt(msg, non_interactive=False):
    if non_interactive:
        print(f"{msg}")
        print("Running in non-interactive mode...")
        return
    try:
        input(f"{msg}\nPress Enter to continue or Ctrl+C to abort...")
    except EOFError:
        print("Running in non-interactive mode (no stdin available)...")
        return

def run(cmd, env=None):
    print(f"Running: {cmd}")
    result = subprocess.run(cmd, shell=True, env=env)
    if result.returncode != 0:
        print(f"Command failed: {cmd}")
        sys.exit(result.returncode)

def main():
    # Check for non-interactive mode more safely
    non_interactive = "--non-interactive" in sys.argv
    if not non_interactive:
        try:
            non_interactive = not sys.stdin.isatty()
        except:
            # If we can't check stdin, assume non-interactive
            non_interactive = True
    
    orig_dir = os.getcwd()
    # Change to the database directory (parent of alembic directory) where alembic.ini is located
    db_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
    print(f"Changing directory to {db_dir}")
    os.chdir(db_dir)

    env = os.environ.copy()
    env["APILOGICPROJECT_NO_FLASK"] = "True"  # ~ export APILOGICPROJECT_NO_FLASK=True

    print("\n\nThis script will update your database schema to match models.py using Alembic.")
    print("Steps:")
    print("1. Set APILOGICPROJECT_NO_FLASK=True -- eg, export APILOGICPROJECT_NO_FLASK=True")
    print("2. Run: alembic upgrade head")
    print("3. Run: alembic revision --autogenerate -m \"message\"")
    prompt("Ready to proceed?", non_interactive)

    run("alembic upgrade head", env=env)
    print("Database schema updated to latest migration.")

    prompt("Now, a new migration will be generated to match models.py.", non_interactive)
    if non_interactive:
        msg = "autogenerated"
    else:
        try:
            msg = input("Enter a message for the migration (default: 'autogenerated'): ") or "autogenerated"
        except EOFError:
            msg = "autogenerated"
    run(f'alembic revision --autogenerate -m "{msg}"', env=env)

    # Find the latest migration file
    versions_dir = os.path.join(db_dir, "alembic", "versions")
    migration_files = sorted(
        [f for f in os.listdir(versions_dir) if f.endswith(".py")],
        key=lambda x: os.path.getmtime(os.path.join(versions_dir, x)),
        reverse=True
    )
    if migration_files:
        latest_file = os.path.join(versions_dir, migration_files[0])
        with open(latest_file, "r") as f:
            lines = f.readlines()
        with open(latest_file, "w") as f:
            for line in lines:
                f.write(line)
                if line.strip().startswith("def downgrade"):
                    f.write("    return\n")
                    break
    else:
        print("No migration file found.")
        sys.exit(1)

    # Apply the newly generated migration after modifying the migration file
    prompt(f"Migration file generated: {latest_file}\nIt is recommended to review this migration file before proceeding.", non_interactive)
    run("alembic upgrade head", env=env)

    print("Migration file updated: downgrade will do nothing.")
    print("Consider updating ui/admin/admin.yaml to reflect schema changes.")
    os.chdir(orig_dir)
    print("\nSuccess! Database schema and migrations are up to date.\n\n")

if __name__ == "__main__":
    main()