from datetime import datetime
from typing import Dict, List
from datetime import datetime
import uuid

class Source(str):
    pass

class Attachment:
    def __init__(self, id_attachment: str = "", uri: str = "", file_type = "", file_extension:str = "", size: int = 0):
        self.id = id_attachment
        self.uri = uri
        self.file_type = file_type
        self.file_extension = file_extension
        self.size = size

    def to_dict(self):
        return {
            "id": self.id,
            "uri": self.uri,
            "file_type": self.file_type,
            "file_extension": self.file_extension,
            "size": self.size
        }

class StructureModel:
    """Structure model for the data being used in the DDA.
    Allows:
    - To create a model from a dictionary.
    - To convert a model to a dictionary.
    - To get the attributes of the model as a list.

    """
    
    @classmethod
    def from_dict(cls, timeslot_dict):
        valid_keys = set(cls.__init__.__code__.co_varnames[1:])
        filtered_dict = {k: v for k, v in timeslot_dict.items() if k in valid_keys}
        return cls(**filtered_dict)
    
    def to_dict(self):
        return {k: v for k, v in self.__dict__.items() if v is not None}


class DatabaseEvent:
    guid = None
    end_time = None
    end_time_local = None


class EventData(StructureModel):
    """
    Time based events Also includes keystrocks and mousclicks just because they might be there, and need them.
    Not the Event Dataset that is on the database (having it's own table), but is on the staging area.

    Automated:
    - guid is generated automatically at creation (cannot be assigned).

    Implementation Note:
    - Have to declare each field aside of guid, Or will throw error, this is Intentional.

    NOTE:
    Timeslot related fields such as Hour, Minute, ts1, Will be None by default, and will be populated on the timeslot creation. Cannot be defined on the EventData creation.

    NOTE: 
    Mouseclicks and keystrokes are expected to have the following format:
    Atomically separated in minutes instead of seconds.
    {
        2024-02-08 14:46: 10,
        ...
        [year-month-date hour:minute]: [count]
    }


    Changes:
    2024-02-08 14:46:17
    - Updated fields defined on this date on the official documentation support.
    - Note that now event_guid must be passed if exists (they will only be generated automatically, if None was passed) This is to ensure that integrations with span_guid or event_guid are passed in correctly. (spanGUID was removed)

    2024-02-20 13:03:57
    - forced for eventData keypress and mouseclicks interactions to be defined as:
    {
        2024-02-08 14:46: 10,
        ...
        [year-month-date hour:minute]: [count]
    }

    """
    def __init__(self,
                #  Key Fields.
                 event_guid: str,
                 processing_guid: str,
                 user_id: str,
                 organization_guid: str,
                 organization_id: int,
                 integration_name: str,

                 application: str,
                 app: str,
                 app_type: str,
                 operation: str,
                 operation_type: str,
                 staging_detail_guid: str,
                 local_timezone: str,

                 timestamp: datetime,
                 end_time: datetime,
                 timestamp_local: datetime,

                 end_time_local: datetime,
                 duration: float,


                 description: str,
                 url: str,
                 site: str,
                 files: List[dict],
                 file_count: int,
                 action_type: str,
                 geolocation: dict,
                 ipv4: str,
                 local_ipv4: str,
                 size: int,
                 email_subject: str,
                 from_address: str,
                 to_address: str,
                 email_cc: str,
                 email_bcc: str,
                 email_imid: str,
                 phone_result: str,
                 record_url: str,
                 recording_url: str,
                 record_id: int,
                
                #  These are the only ones that are not in the original model
                 mouse_clicks: dict = {},
                 keystrokes: dict = {},

                # Pregenerated by default

                 tags: List[dict] = []


                 # Timeslot related fields such as Hour, Minute, ts1, Will be None by default, and will be populated on the timeslot creation. Cannot be defined on the EventData creation.
                 ):
        
        self.processing_guid = processing_guid
        self.event_guid = event_guid if event_guid is not None else str(uuid.uuid4())
        self.organization_guid = organization_guid
        self.organization_id = organization_id
        self.integration_name = integration_name

        self.user_id = user_id
        self.application = application
        
        self.app = app
        self.app_type = app_type
        self.operation = operation
        self.operation_type = operation_type
        self.staging_detail_guid = staging_detail_guid
        self.staging_guid = processing_guid
        self.local_timezone = local_timezone
        self.timestamp = timestamp
        self.end_time = end_time

        self.timestamp_local = timestamp_local
        self.end_time_local = end_time_local
        self.duration = duration
        self.description = description
        self.url = url
        self.site = site
        self.size = size
        self.files = files

        self.file_count = file_count
        self.action_type = action_type
        self.geolocation = geolocation
        self.ipv4 = ipv4
        self.local_ipv4 = local_ipv4
        self.email_subject = email_subject
        self.from_address = from_address
        self.to_address = to_address
        self.email_cc = email_cc
        self.email_bcc = email_bcc
        self.email_imid = email_imid
        self.phone_result = phone_result
        self.record_url = record_url
        self.recording_url = recording_url
        self.record_id = record_id
        self.tags = tags

        # Added for Timeslot support
        self.mouse_clicks = mouse_clicks
        self.keystrokes = keystrokes



        

class Event(StructureModel):
    """
    This is the time based event. Which will be used on modeling the database.

    NOTE: Staging_guid is generated automatically at creation (cannot be assigned).
    
    """
    def __init__(self,
                    guid: str,
                    organization_guid: str,
                    organization_id: int,
                    user_id: str,

                    application: str,
                    app: str,
                    app_type: str,
                    
                    operation: str,
                    operation_type: str,

                    integration_name: str,
                    
                    local_timezone: str,
                    timestamp: datetime,

                    end_time: datetime = '0001-01-01T00:00:00',
                    timestamp_local: datetime = datetime.now(),
                    end_time_local: datetime = datetime.now(),
                    duration: float = 0.0,
                    description: str = "",
                    url: str = "",
                    site: str = "",
                    files: List[dict] = [],
                    size: int = 0,
                    
                    file_count: int = 0,
                    action_type: str = "",
                    geolocation: dict = {},
                    ipv4: str = "",
                    local_ipv4: str = "",

                    email_subject: str = "",
                    from_address: str = "",
                    to_address: str = "",
                    email_cc: str = "",
                    email_bcc: str = "",
                    email_imid: str = "",
                    phone_result: str = "",
                    record_url: str = "",
                    recording_url: str = "",
                    record_id: int = 0,

                    tags: List[dict] = []
                 ):
        # Generate random guid

        self.staging_guid = str(uuid.uuid4()) # GUID Will be used to populate the Timeslots on creation.

        # Id is generated automatically at creation (cannot be assigned).
        self.guid = guid # Should be mapped from event_guid
        self.organization_guid = organization_guid
        self.organization_id = organization_id

        # self.span_guid = span_guid
        self.user_id = user_id
        self.application = application
        self.app = app
        self.app_type = app_type
        self.operation = operation
        self.operation_type = operation_type
        
        self.integration_name = integration_name

        self.local_timezone = local_timezone
        self.timestamp = timestamp

        if end_time != '0001-01-01T00:00:00':
            self.end_time = end_time

        self.timestamp_local = timestamp_local
        self.end_time_local = end_time_local
        self.duration = duration
        self.description = description
        self.url = url
        self.site = site
        self.files = files
        self.size = size
        self.file_count = file_count

        self.action_type = action_type
        self.geolocation = geolocation
        self.ipv4 = ipv4
        self.local_ipv4 = local_ipv4
        
        self.email_subject = email_subject
        self.from_address = from_address
        self.to_address = to_address
        self.email_cc = email_cc
        self.email_bcc = email_bcc
        self.email_imid = email_imid
        self.phone_result = phone_result
        self.record_url = record_url
        self.recording_url = recording_url
        self.record_id = record_id
        
        self.tags = tags


    @staticmethod
    def get_publishing_keys():
        """Returns all the attributes of the class.
        - 2023-06-14 14:33:16 Removed "files", until solution is found
        Returns:
            List[str]: List of attributes.
        """
        return [
            "guid", "organization_guid", "organization_id", "user_id", 
            "application", "app", "app_type", "operation", "operation_type", 
            "integration_name", "local_timezone", "timestamp", "end_time", 
            "timestamp_local", "end_time_local", "duration", "description", 
            "url", "site", "file_count", "action_type", "geolocation", 
            "ipv4", "local_ipv4", "email_subject", "from_address", 
            "to_address", "email_cc", "email_bcc", "email_imid", 
            "phone_result", "record_url", "recording_url", "record_id", 
            "tags"
        ]

class Timeslot(StructureModel):
    """Timeslot
    This is the timeslot based event in which an hour is divided into 6 slots of 10 minutes.
    """
    def __init__(self, 
                 event_guid: str,
                 organization_guid: str, 
                 
                 hour: int, minute: int, day: int, 
                 month: int, year: int, week: int, weekday: int,

                 hour_local: int, minute_local: int, day_local: int, month_local: int,
                 year_local: int, week_local: int, weekday_local: int, 
                 
                 mouse_clicks: int,
                 keystrokes: int,
                 processing_guid: str,
                 
                 ts1: int, ts5: int, ts10: int, ts15: int,
                 tl1: int, tl5: int, tl10: int, tl15: int,

                 event_end_time: datetime # Used for timeslot creation, not inserted on table.
                 ):
        # guid will be generated by the postgresql server upon insertion

        self.event_guid = event_guid
        self.organization_guid = organization_guid

        self.hour = hour
        self.minute = minute
        self.day = day
        self.month = month
        self.year = year
        self.week = week
        self.weekday = weekday

        self.hour_local = hour_local
        self.minute_local = minute_local
        self.day_local = day_local
        self.month_local = month_local
        self.year_local = year_local
        self.week_local = week_local
        self.weekday_local = weekday_local


        self.mouse_clicks = mouse_clicks
        self.keystrokes = keystrokes
        self.processing_guid = processing_guid
        
        self.ts1 = ts1
        self.ts5 = ts5
        self.ts10 = ts10
        self.ts15 = ts15

        self.tl1 = tl1
        self.tl5 = tl5
        self.tl10 = tl10
        self.tl15 = tl15

        self.event_end_time = event_end_time


    @staticmethod
    def get_publishing_keys():
        """Returns all the attributes of the class.

        Returns:
            List[str]: List of attributes.
        """
        return [
            "event_guid", "organization_guid", 
            "hour", "minute", "day", "month", "year", "week", "weekday",
            "hour_local", "minute_local", "day_local", "month_local", "year_local", "week_local", "weekday_local",
            "mouse_clicks", "keystrokes", "processing_guid",
            "ts1", "ts5", "ts10", "ts15", "tl1", "tl5", "tl10", "tl15"
        ]


class ProcessingTracker(StructureModel):
    """
    Model for Staging Events. 
    """
    def __init__(self, version = "", connector_guid="", type = "", organization_guid = "", item_count = 2, details = [], processed_time = datetime.now(), status = "", last_error_message = "", last_error_time = None, created_time = datetime.now(), updated_time = datetime.now()):
        self.guid = str(uuid.uuid4()) #Random GUID
        self.version = version
        self.connector_guid = connector_guid
        self.type = type
        self.organization_guid = organization_guid
        self.item_count = item_count
        self.details = details
        self.processed_time = processed_time
        self.status = status
        self.last_error_message = last_error_message
        self.last_error_time = last_error_time
        self.created_time = created_time
        self.updated_time = updated_time

    def to_dict(self):
        return {
            "guid": self.guid,
            "version": self.version,
            "connector_guid": self.connector_guid,
            "type": self.type,
            "organization_guid": self.organization_guid,
            "item_count": self.item_count,
            "details": self.details,
            "processed_time": self.processed_time,
            "status": self.status,
            "last_error_message": self.last_error_message,
            "last_error_time": self.last_error_time,
            "created_time": self.created_time,
            "updated_time": self.updated_time
        }
    
    def populate_properties_from_dict(self, data: dict):
        """
        Populate the instance properties from a dictionary.
        """
        if 'guid' in data:
            self.guid = data['guid']
        if 'version' in data:
            self.version = data['version']
        if 'connector_guid' in data:
            self.connector_guid = data['connector_guid']
        if 'activity' in data:
            self.type = data['activity']
        if 'organization_guid' in data:
            self.organization_guid = data['organization_guid']
        if 'item_count' in data:
            self.item_count = data['item_count']
        if 'details' in data:
            self.details = data['details']
        elif 'Details' in data:
            self.details = data['Details']
        if 'processed_time' in data:
            self.processed_time = data['processed_time']
        if 'status' in data:
            self.status = data['status']
        if 'last_error_message' in data:
            self.last_error_message = data['last_error_message']
        if 'last_error_time' in data:
            self.last_error_time = data['last_error_time']
        if 'created_time' in data:
            self.created_time = data['created_time']
        if 'updated_time' in data:
            self.updated_time = data['updated_time']
    

    def get_organization_guid(self):
        """Returns the organization guid of the event.
        Otherwise raises error
        """
        if self.organization_guid == "":
            raise Exception("Organization guid not found")
        else:
            return self.organization_guid

class ProcessingTracker(StructureModel):
    """
    Model for Staging Events. 
    """
    def __init__(self, 
                 source_guid = "", 
                 connector_guid="", 

                 status = "", 
                 s3_key = "",

                 organization_guid = "",
                 integration_name = "", 
                 integration_version = "", 
                 
                 source_date = datetime.now(), 
                 task = "", 
                 start_time = datetime.now(), 
                 end_time = datetime.now(), 
                 
                 error_count = 0, 
                 item_count = 0, 

                 
                 last_error_message = "", 
                 last_error_time = None, 
                 
                 received_time = datetime.now(), 
                 created_time = datetime.now(), 
                 updated_time = datetime.now()):
        
        self.processing_guid = str(uuid.uuid4()) #Random GUID

        self.source_guid = source_guid
        self.connector_guid = connector_guid

        self.status = status
        self.s3_key = s3_key

        self.organization_guid = organization_guid
        self.integration_name = integration_name
        self.integration_version = integration_version
        
        self.source_date = source_date
        self.task = task
        self.start_time = start_time
        self.end_time = end_time

        self.error_count = error_count
        self.item_count = item_count
        

        self.last_error_message = last_error_message
        self.last_error_time = last_error_time

        self.received_time = received_time
        self.created_time = created_time
        self.updated_time = updated_time

    def to_dict(self):
        return {
            "guid": self.processing_guid,

            "source_guid": self.source_guid,
            "connector_guid": self.connector_guid,
            
            "organization_guid": self.organization_guid,
            "integration_name": self.integration_name,
            "integration_version": self.integration_version,
            
            "source_date": self.source_date,
            "task": self.task,
            "start_time": self.start_time,
            "end_time": self.end_time,
            
            "error_count": self.error_count,
            "item_count": self.item_count,
            
            "status": self.status,
            "last_error_message": self.last_error_message,
            "last_error_time": self.last_error_time,
            
            "received_time": self.received_time,
            "created_time": self.created_time,
            "updated_time": self.updated_time
        }
    
    
    def populate_properties_from_dict(self, data: dict):
        """
        Populate the instance properties from a dictionary.
        """
        if 'guid' in data:
            self.processing_guid = data['guid']
        if 'source_guid' in data:
            self.source_guid = data['source_guid']
        if 'connector_guid' in data:
            self.connector_guid = data['connector_guid']
        if 'organization_guid' in data:
            self.organization_guid = data['organization_guid']
        if 'integration_name' in data:
            self.integration_name = data['integration_name']
        if 'integration_version' in data:
            self.integration_version = data['integration_version']
        if 'source_date' in data:
            self.source_date = data['source_date']
        if 'task' in data:
            self.task = data['task']
        if 'start_time' in data:
            self.start_time = data['start_time']
        if 'end_time' in data:
            self.end_time = data['end_time']
        if 'error_count' in data:
            self.error_count = data['error_count']
        if 'item_count' in data:
            self.item_count = data['item_count']
        if 'status' in data:
            self.status = data['status']
        if 'last_error_message' in data:
            self.last_error_message = data['last_error_message']
        if 'last_error_time' in data:
            self.last_error_time = data['last_error_time']
        if 'received_time' in data:
            self.received_time = data['received_time']
        if 'created_time' in data:
            self.created_time = data['created_time']
        if 'updated_time' in data:
            self.updated_time = data['updated_time']
    

    def get_organization_guid(self):
        """Returns the organization guid of the event.
        Otherwise raises error
        """
        if self.organization_guid == "":
            raise Exception("Organization guid not found")
        else:
            return self.organization_guid

class LogMessage():
    """
    A log message to be stored in the job_service.
    """

    def __init__(self, log_type: str, log_message: str, log_detail: str):
        self.log_date = datetime.now()
        self.log_type = log_type
        self.log_message = log_message
        self.log_detail = log_detail
        

